package com.xforceplus.ultraman.extension.changelog.history.impl;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.xforceplus.ultraman.extension.changelog.history.ChangelogFacade;
import com.xforceplus.ultraman.extension.changelog.history.EventOperation;
import com.xforceplus.ultraman.extension.changelog.history.domain.ChangelogEntity;
import com.xforceplus.ultraman.extension.changelog.history.domain.HistoryEntity;
import com.xforceplus.ultraman.extension.changelog.history.domain.KeyBasedQuery;
import com.xforceplus.ultraman.metadata.domain.vo.Page;
import com.xforceplus.ultraman.metadata.domain.vo.Summary;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.sdk.infra.logging.LoggingPattern;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;

import javax.sql.DataSource;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;
import java.util.stream.Stream;

import static com.xforceplus.ultraman.sdk.infra.logging.LoggingUtils.logErrorPattern;

/**
 * default implementation
 */
@Slf4j
public class DefaultChangelogFacadeImpl implements ChangelogFacade {

    private final static String QUERY_BY_ID_SQL = "SELECT * FROM %s_history_log WHERE ID = ?";
    private final static String QUERY_BY_KEY1_SQL = "SELECT * FROM %s_history_log WHERE KEY1 = ?";
    private final static String QUERY_BY_KEY2_SQL = "SELECT * FROM %s_history_log WHERE KEY1 = ?";
    private final static String QUERY_BY_KEY3_SQL = "SELECT * FROM %s_history_log WHERE KEY1 = ?";
    private final static String QUERY_CHANGELOG_BY_ID_SQL = "SELECT * FROM %s_changelog WHERE ID = ? ORDER BY CREATE_TIME ASC";

    private final static String QUERY_BY_ENTITY_CLASS_ID_SQL = "SELECT * FROM %s_history_log WHERE %s order by id ASC, create_time ASC limit ?,? ";
    private final static String COUNT_QUERY_BY_ENTITY_CLASS_ID_SQL = "SELECT COUNT(1) FROM %s_history_log WHERE %s ";

    private final static String QUERY_CHANGELOG_BY_ENTITY_CLASS_AND_ID_SQL = "SELECT * FROM %s_changelog WHERE %s order by id ASC, create_time ASC limit ?,?";
    private final static String COUNT_CHANGELOG_QUERY_BY_ENTITY_CLASS_ID_AND_ID_SQL = "SELECT COUNT(1) FROM %s_changelog WHERE %s ";

    private String[] entityClassIds = new String[]{"entityclassl0", "entityclassl1", "entityclassl2", "entityclassl3", "entityclassl4"};

    private JdbcTemplate jdbcTemplate;
    private EntityClassEngine engine;
    private String appCode;
    private ObjectMapper mapper;

    public DefaultChangelogFacadeImpl(DataSource dataSource, EntityClassEngine engine, ObjectMapper mapper) {
        this.jdbcTemplate = new JdbcTemplate(dataSource);
        this.engine = engine;
        this.appCode = engine.appCode();
        this.mapper = mapper;
    }

    @Override
    public Page<ChangelogEntity> findChangelogByEntityClass(KeyBasedQuery keyBasedQuery) {
        Long entityClassId = keyBasedQuery.getEntityClassId();
        int pageNo = keyBasedQuery.getPage();
        int pageSize = keyBasedQuery.getSize();
        Optional<IEntityClass> target = engine.load(entityClassId.toString(), null);
        if (target.isPresent()) {
            IEntityClass targetEntityClass = target.get();
            EntityClassGroup group = engine.describe(targetEntityClass, null);
            Collection<IEntityClass> fatherEntityClass = group.getFatherEntityClass();
            String entityClassIdsWhere = generateSqlString(fatherEntityClass, entityClassId);

            if(keyBasedQuery.getId() > 0) {
                if(entityClassIdsWhere !=  null && !entityClassIdsWhere.isEmpty()) {
                    entityClassIdsWhere = entityClassIdsWhere.concat(" and id = ").concat(Long.toString(keyBasedQuery.getId()));
                } else {
                    entityClassIdsWhere = "id = ".concat(Long.toString(keyBasedQuery.getId()));
                }
            }

            if(!StringUtils.isEmpty(keyBasedQuery.getKeys1())) {
                if(!StringUtils.isEmpty(entityClassIdsWhere)) {
                    entityClassIdsWhere = entityClassIdsWhere.concat(String.format(" and key1 = '%s'", keyBasedQuery.getKeys1()));
                } else {
                    entityClassIdsWhere = String.format("key1 = '%s'", keyBasedQuery.getKeys1());
                }
            }

            String formattedWhere = String.format(QUERY_CHANGELOG_BY_ENTITY_CLASS_AND_ID_SQL, appCode, entityClassIdsWhere);
            int offset = (pageNo - 1) * pageSize;
            int size = pageSize;
            List<ChangelogEntity> result = jdbcTemplate.query(formattedWhere
                    , ps -> {
                        ps.setLong(1, offset);
                        ps.setLong(2, size);
                    }, (rs, rowNum) -> toChangelogEntity(rs));

            String formattedCount = String.format(COUNT_CHANGELOG_QUERY_BY_ENTITY_CLASS_ID_AND_ID_SQL, appCode, entityClassIdsWhere);
            Integer totalCount = jdbcTemplate.queryForObject(formattedCount, Integer.class);

            //List<T> rows, Summary summary, long size, long current
            Summary summary = new Summary();
            summary.setTotal(totalCount);
            return new Page<>(result, summary, result.size(), 0);
        }
        return null;
    }

    @Override
    public Page<HistoryEntity> findHistoryByEntityClass(KeyBasedQuery keyBasedQuery) {
        Long entityClassId = keyBasedQuery.getEntityClassId();
        int pageNo = keyBasedQuery.getPage();
        int pageSize = keyBasedQuery.getSize();
        Optional<IEntityClass> target = engine.load(entityClassId.toString(), null);
        if (target.isPresent()) {
            IEntityClass targetEntityClass = target.get();
            EntityClassGroup group = engine.describe(targetEntityClass, null);
            Collection<IEntityClass> fatherEntityClass = group.getFatherEntityClass();
            String entityClassIdsWhere = generateSqlString(fatherEntityClass, entityClassId);

            if(keyBasedQuery.getId() > 0) {
                if(entityClassIdsWhere !=  null && !entityClassIdsWhere.isEmpty()) {
                    entityClassIdsWhere = entityClassIdsWhere.concat(" and id = ").concat(Long.toString(keyBasedQuery.getId()));
                } else {
                    entityClassIdsWhere = "id = ".concat(Long.toString(keyBasedQuery.getId()));
                }
            }

            if(!StringUtils.isEmpty(keyBasedQuery.getKeys1())) {
                if(!StringUtils.isEmpty(entityClassIdsWhere)) {
                    entityClassIdsWhere = entityClassIdsWhere.concat(String.format(" and key1 = '%s'", keyBasedQuery.getKeys1()));
                } else {
                    entityClassIdsWhere = String.format("key1 = '%s'", keyBasedQuery.getKeys1());
                }
            }

            String formattedWhere = String.format(QUERY_BY_ENTITY_CLASS_ID_SQL, appCode, entityClassIdsWhere);
            int offset = (pageNo - 1) * pageSize;
            int size = pageSize;
            List<HistoryEntity> result = jdbcTemplate.query(formattedWhere
                    , ps -> {
                        ps.setLong(1, offset);
                        ps.setLong(2, size);
                    }, (rs, rowNum) -> toHistoryEntity(rs));


            String formattedCount = String.format(COUNT_QUERY_BY_ENTITY_CLASS_ID_SQL, appCode, entityClassIdsWhere);
            Integer totalCount = jdbcTemplate.queryForObject(formattedCount, Integer.class);

            //List<T> rows, Summary summary, long size, long current
            Summary summary = new Summary();
            summary.setTotal(totalCount);
            return new Page<>(result, summary, result.size(), 0);
        }
        return null;
    }

    private String generateSqlString(Collection<IEntityClass> fatherEntityClass, Long entityClassId) {
        List<Long> id = new ArrayList<>();
        fatherEntityClass.forEach(x -> {
            id.add(x.id());
        });

        id.add(entityClassId);
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < entityClassIds.length; i++) {
            if (id.size() > i) {
                sb.append(entityClassIds[i]).append(" = ").append(id.get(i));
            }
        }

        return sb.toString();
    }

    @Override
    public HistoryEntity findHistoryById(Long id) {
        HistoryEntity result = jdbcTemplate.query(String.format(QUERY_BY_ID_SQL, appCode), ps -> ps.setLong(1, id), new ResultSetExtractor<HistoryEntity>() {
            @Override
            public HistoryEntity extractData(ResultSet rs) throws SQLException, DataAccessException {
                if (rs.next()) {
                    return toHistoryEntity(rs);
                } else {
                    return null;
                }
            }
        });

        return result;
    }


    private HistoryEntity toHistoryEntity(ResultSet rs) throws SQLException {
        HistoryEntity entity = new HistoryEntity();

        long id = rs.getLong("id");
        entity.setId(id);

        long l0 = rs.getLong("entityclassl0");
        long l1 = rs.getLong("entityclassl1");
        long l2 = rs.getLong("entityclassl2");
        long l3 = rs.getLong("entityclassl3");
        long l4 = rs.getLong("entityclassl4");

        String profile = rs.getString("profile");

        Optional<Long> targetId = Stream.of(l4, l3, l2, l1, l0).filter(x -> x > 0).findFirst();
        if (targetId.isPresent()) {
            Optional<IEntityClass> load = engine.load(targetId.get().toString(), profile);
            load.ifPresent(entityClass -> {
                entity.setEntityClassId(entityClass.id());
            });
        }

        String body = rs.getString("attr");
        Map<String, Object> bodyMap = Collections.emptyMap();

        try {
            bodyMap = mapper.readValue(body, Map.class);
        } catch (Throwable e) {
            logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                    , e);
        }

        entity.setBody(bodyMap);

        long createTime = rs.getLong("create_time");

        entity.setDelTimestamp(createTime);
        Long delUid = rs.getLong("create_user_id");
        String delName = rs.getString("create_user_name");
        entity.setDelUid(delUid);
        entity.setDelUser(delName);
        return entity;
    }

    @Override
    public HistoryEntity findHistoryByKey1(Long entityClassId, String... keys) {
        String key1 = String.join("%%", keys);
        HistoryEntity result = jdbcTemplate.query(String.format(QUERY_BY_KEY1_SQL, appCode)
                , ps -> ps.setString(1, key1), new ResultSetExtractor<HistoryEntity>() {
                    @Override
                    public HistoryEntity extractData(ResultSet rs) throws SQLException, DataAccessException {
                        if (rs.next()) {
                            return toHistoryEntity(rs);
                        } else {
                            return null;
                        }
                    }
                });
        return result;
    }

    @Override
    public HistoryEntity findHistoryByKey2(Long entityClassId, String... keys) {
        String key1 = String.join("%%", keys);
        HistoryEntity result = jdbcTemplate.query(String.format(QUERY_BY_KEY2_SQL, appCode)
                , ps -> ps.setString(1, key1), new ResultSetExtractor<HistoryEntity>() {
                    @Override
                    public HistoryEntity extractData(ResultSet rs) throws SQLException, DataAccessException {
                        if (rs.next()) {
                            return toHistoryEntity(rs);
                        } else {
                            return null;
                        }
                    }
                });
        return result;
    }

    @Override
    public HistoryEntity findHistoryByKey3(Long entityClassId, String... keys) {
        String key1 = String.join("%%", keys);
        HistoryEntity result = jdbcTemplate.query(String.format(QUERY_BY_KEY3_SQL, appCode)
                , ps -> ps.setString(1, key1), new ResultSetExtractor<HistoryEntity>() {
                    @Override
                    public HistoryEntity extractData(ResultSet rs) throws SQLException, DataAccessException {
                        if (rs.next()) {
                            return toHistoryEntity(rs);
                        } else {
                            return null;
                        }
                    }
                });
        return result;
    }

    @Override
    public List<ChangelogEntity> findChangeLogById(Long id) {
        List<ChangelogEntity> result = jdbcTemplate.query(String.format(QUERY_CHANGELOG_BY_ID_SQL, appCode)
                , ps -> ps.setLong(1, id), new RowMapper<ChangelogEntity>() {
                    @Override
                    public ChangelogEntity mapRow(ResultSet rs, int rowNum) throws SQLException {
                        return toChangelogEntity(rs);
                    }
                });
        return result;
    }

    private ChangelogEntity toChangelogEntity(ResultSet rs) throws SQLException {
        ChangelogEntity entity = new ChangelogEntity();

        entity.setCid(rs.getString("cid"));
        entity.setId(rs.getLong("id"));

        long l0 = rs.getLong("entityclassl0");
        long l1 = rs.getLong("entityclassl1");
        long l2 = rs.getLong("entityclassl2");
        long l3 = rs.getLong("entityclassl3");
        long l4 = rs.getLong("entityclassl4");

        String profile = rs.getString("profile");

        Optional<Long> targetId = Stream.of(l4, l3, l2, l1, l0).filter(x -> x > 0).findFirst();
        if (targetId.isPresent()) {
            Optional<IEntityClass> load = engine.load(targetId.get().toString(), profile);
            load.ifPresent(entityClass -> {
                entity.setEntityClassId(entityClass.id());
            });
        }

        String body = rs.getString("attr");
        Map<String, Object> bodyMap = Collections.emptyMap();

        try {
            bodyMap = mapper.readValue(body, Map.class);
        } catch (Throwable e) {
            logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                    , e);
        }

        entity.setBody(bodyMap);

        long createTime = rs.getLong("create_time");

        entity.setOpTimestamp(createTime);
        Long delUid = rs.getLong("create_user_id");
        String delName = rs.getString("create_user_name");
        entity.setOpUid(delUid);
        entity.setOpUser(delName);
        entity.setVer(rs.getInt("ver"));

        int type = rs.getInt("operation");
        if(type != 0) {
            EventOperation from = EventOperation.from(type);
            if(from != null) {
                entity.setOperation(from.name());
            }
        }

        return entity;
    }
}
