package com.xforceplus.ultraman.extension.changelog.history;

import akka.stream.*;
import akka.stream.javadsl.Sink;
import akka.stream.javadsl.Source;
import akka.stream.javadsl.SourceQueueWithComplete;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.xforceplus.ultraman.extension.changelog.history.domain.RecordOperator;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.sdk.core.event.EntityCreated;
import com.xforceplus.ultraman.sdk.core.event.EntityDeleted;
import com.xforceplus.ultraman.sdk.core.event.EntityEvent;
import com.xforceplus.ultraman.sdk.core.event.EntityUpdated;
import com.xforceplus.ultraman.sdk.infra.utils.JacksonDefaultMapper;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.transaction.event.TransactionalEventListener;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.ForkJoinPool;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.extension.changelog.utils.ChangelogHelper.extractKeys;
import static com.xforceplus.ultraman.extension.changelog.utils.ChangelogHelper.extractProfile;
import static com.xforceplus.ultraman.sdk.core.utils.MasterStorageHelper.ZONE_ID;

@Slf4j
public class ChangeLogEventListener {

    //INSERT INTO oqs.new20_changelog(cid, id, entityclassl0, entityclassl1, entityclassl2, entityclassl3, entityclassl4
    // , businessKey, ver, profile, create_time, create_user_id, create_user_name, attr, remark) VALUES
    //(12, 121, NULL, NULL, NULL, NULL, NULL, NULL, 0, NULL, NULL, NULL, NULL, NULL, NULL);
    private final static String INSERT_SQL = "INSERT INTO %s_changelog(cid, id, entityclassl0, entityclassl1, entityclassl2, entityclassl3, entityclassl4" +
            ", key1, key2, key3, ver, profile, create_time, create_user_id, create_user_name, attr, remark, operation) VALUES\n" +
            "(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    private ActorMaterializer materializer;
    /**
     * main datasource
     */
    @Qualifier("master")
    @Autowired
    private DataSource dataSource;

    @Autowired
    private EntityClassEngine engine;

    private SourceQueueWithComplete<EntityEvent> queue;
    private String appCode;

    @Autowired(required = false)
    private List<EventExtractor> eventExtractors = new ArrayList<>();
    /**
     * dedicate forkjoin pool for stream para
     */
    private ForkJoinPool forkJoinPool = new ForkJoinPool(4);

    public ChangeLogEventListener(ActorMaterializer mat, EntityClassEngine engine) {
        this.appCode = engine.appCode();
        this.materializer = mat;
        this.engine = engine;
        queue = Source.<EntityEvent>queue(100000, OverflowStrategy.backpressure())
                .groupedWithin(100, Duration.ofSeconds(5))
                .map(x -> {
                    try {
                        recordChangelog(x);
                    } catch (Throwable throwable) {
                        log.error("{}", throwable);
                    }
                    return "";
                })
                .log("changelog-record")
                .to(Sink.ignore())
                .withAttributes(ActorAttributes.withSupervisionStrategy(x -> Supervision.resume()))
                .run(materializer);
    }

    public void setDataSource(DataSource dataSource) {
        this.dataSource = dataSource;
    }

    public void setEventExtractors(List<EventExtractor> eventExtractors) {
        this.eventExtractors = eventExtractors;
    }

    /**
     * save entity event
     *
     * @param entityEvent
     * @return
     */
    private Tuple2<String, String> extractEntityEvent(EntityEvent entityEvent) {
        if (entityEvent instanceof EntityCreated) {
            String code = ((EntityCreated) entityEvent).getCode();
            Map<String, Object> data = ((EntityCreated) entityEvent).getData();
            String s = extractProfile(data);
            return Tuple.of(code, s);
        } else if (entityEvent instanceof EntityUpdated) {
            String code = ((EntityUpdated) entityEvent).getCode();
            Map<String, Object> data = ((EntityUpdated) entityEvent).getData();
            String s = extractProfile(data);
            return Tuple.of(code, s);
        } else if (entityEvent instanceof EntityDeleted) {
            String code = ((EntityDeleted) entityEvent).getCode();
            Map<String, Object> data = ((EntityDeleted) entityEvent).getData();
            String s = extractProfile(data);
            return Tuple.of(code, s);
        } else {
            return Tuple.of("", "");
        }
    }

    private void recordChangelog(List<EntityEvent> grouped) {

        Map<Tuple2<String, String>, List<EntityEvent>> entityGrouped = grouped
                .stream().collect(Collectors.groupingBy(this::extractEntityEvent));

        //TODO
        try (Connection connection = dataSource.getConnection();
             PreparedStatement statement = connection.prepareStatement(String.format(INSERT_SQL, appCode))) {
            entityGrouped.entrySet().stream()
                    .forEach(entry -> {
                        Tuple2<String, String> key = entry.getKey();
                        String code = key._1;
                        String profile = key._2;

                        if (StringUtils.isEmpty(code)) {
                            return;
                        }

                        Optional<IEntityClass> entityClass = engine.loadByCode(code, profile);
                        if (entityClass.isPresent()) {
                            List<EntityEvent> events = entry.getValue();
                            events.forEach(evt -> {
                                try {
                                    setValue(engine.describe(entityClass.get(), profile), evt, profile, statement);
                                    statement.addBatch();
                                } catch (SQLException e) {
                                    e.printStackTrace();
                                }
                            });
                        }
                    });

            statement.executeBatch();
        } catch (SQLException e) {
            e.printStackTrace();
            //TODO
        }
    }

    //TODO
    private void setValue(EntityClassGroup entityClassGroup, EntityEvent evt, String profile
            , PreparedStatement statement) throws SQLException {

        Long id = 0L;

        Optional<EventExtractor> extractor = eventExtractors.stream()
                .filter(x -> x.support(evt.getClass())).findFirst();

        if (!extractor.isPresent()) {
            log.warn("Evt {} has no extractor", evt);
            return;
        }

        EventExtractor eventExtractor = extractor.get();

        statement.setString(1, UUID.randomUUID().toString());

        //add id
        statement.setLong(2, eventExtractor.extractId(evt));

        RecordOperator operator = eventExtractor.extractUser(evt);

        long currentEntityId = entityClassGroup.getEntityClass().id();
        Collection<IEntityClass> fatherEntityClass = entityClassGroup.getFatherEntityClass();
        int index = 3;
        Iterator<IEntityClass> iterator = fatherEntityClass.iterator();
        while (iterator.hasNext()) {
            IEntityClass next = iterator.next();
            statement.setLong(index, next.id());
            index++;
        }

        statement.setLong(index, currentEntityId);

        while (index++ < 8) {
            statement.setLong(index, 0L);
        }

        //unique key
        List<String> keys = extractKeys(eventExtractor.getData(evt), entityClassGroup);

        //add key
        statement.setString(8, keys.get(0));
        statement.setString(9, keys.get(1));
        statement.setString(10, keys.get(2));

        //ver
        int ver = eventExtractor.extractVer(evt);
        statement.setInt(11, ver);

        //profile
        statement.setString(12, profile);

        //create time
        LocalDateTime now = LocalDateTime.now();
        Instant instant = now.atZone(ZONE_ID).toInstant();
        statement.setLong(13, instant.toEpochMilli());

        //create user id
        statement.setLong(14, operator.getUserId());
        statement.setString(15, operator.getUserName());

        //attr
        Map<String, Object> body = eventExtractor.extractChangedData(evt);
        try {
            String attr = JacksonDefaultMapper.OBJECT_MAPPER.writeValueAsString(body);
            statement.setString(16, attr);
        } catch (JsonProcessingException e) {
            e.printStackTrace();
        }

        //TODO remark
        statement.setString(17, "");
        statement.setInt(18, eventExtractor.extractType(evt).getType());
    }

    @TransactionalEventListener
    public void recordHistory(EntityEvent evt) {
        try {
            QueueOfferResult join = queue.offer(evt).toCompletableFuture().join();
            //make this in
            if (join == QueueOfferResult.dropped()) {
                //write to file to oss
                //TODO

            }
        } catch (Throwable throwable) {
            log.error("{}", throwable);
        }
    }
}
