package com.xforceplus.ultraman.extension.changelog.history;

import akka.stream.*;
import akka.stream.javadsl.Sink;
import akka.stream.javadsl.Source;
import akka.stream.javadsl.SourceQueueWithComplete;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.sdk.core.event.EntityDeleted;
import com.xforceplus.ultraman.sdk.infra.base.cdc.SystemAttachment;
import com.xforceplus.ultraman.sdk.infra.utils.JacksonDefaultMapper;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.transaction.event.TransactionalEventListener;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.ForkJoinPool;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.extension.changelog.utils.ChangelogHelper.extractKeys;
import static com.xforceplus.ultraman.extension.changelog.utils.ChangelogHelper.extractProfile;
import static com.xforceplus.ultraman.sdk.core.utils.MasterStorageHelper.ZONE_ID;

@Slf4j
public class HistoryEventListener {

    private final static String INSERT_SQL = "INSERT INTO %s_history_log(id, entityclassl0, entityclassl1, entityclassl2, entityclassl3" +
            ", entityclassl4, profile, key1, key2, key3, create_time, create_user_id, create_user_name, attr) VALUES\n" +
            "(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    private ActorMaterializer materializer;
    /**
     * main datasource
     */
    @Qualifier("master")
    @Autowired
    @Lazy
    private DataSource dataSource;

    @Autowired
    private EntityClassEngine engine;

    private SourceQueueWithComplete<EntityDeleted> queue;
    private String appCode;
    /**
     * dedicate forkjoin pool for stream para
     */
    private ForkJoinPool forkJoinPool = new ForkJoinPool(4);

    public HistoryEventListener(ActorMaterializer mat) {
        this.materializer = mat;
        queue = Source.<EntityDeleted>queue(10000, OverflowStrategy.backpressure())
                .groupedWithin(100, Duration.ofSeconds(5))
                .map(x -> {
                    try {
                        recordDelete(x);
                    } catch (Throwable throwable) {
                        log.error("{}", throwable);
                    }
                    return "";
                })
                .log("history-record")
                .to(Sink.ignore()).withAttributes(ActorAttributes.withSupervisionStrategy(x -> Supervision.resume()))
                .run(materializer);
    }

    public void setDataSource(DataSource dataSource) {
        this.dataSource = dataSource;
    }

    public void setEngine(EntityClassEngine engine) {
        this.engine = engine;
    }

    private void recordDelete(List<EntityDeleted> grouped) {
        if (appCode == null) {
            appCode = engine.appCode();
        }

        Map<Tuple2<String, String>, List<EntityDeleted>> entityGrouped = grouped
                .stream().collect(Collectors.groupingBy(x -> {
                    String code = x.getCode();
                    Map<String, Object> data = x.getData();
                    String s = extractProfile(data);
                    return Tuple.of(code, s);
                }));

        //TODO
        try (Connection connection = dataSource.getConnection();
             PreparedStatement statement = connection.prepareStatement(String.format(INSERT_SQL, appCode))) {
            entityGrouped.entrySet().stream()
                    .forEach(entry -> {
                        Tuple2<String, String> key = entry.getKey();
                        String code = key._1;
                        String profile = key._2;
                        Optional<IEntityClass> entityClass = engine.loadByCode(code, profile);
                        if (entityClass.isPresent()) {
                            List<EntityDeleted> events = entry.getValue();
                            events.forEach(evt -> {
                                try {
                                    setValue(engine.describe(entityClass.get(), profile), evt, profile, statement);
                                    statement.addBatch();
                                } catch (SQLException e) {
                                    e.printStackTrace();
                                }
                            });
                        }
                    });

            statement.executeBatch();
        } catch (SQLException e) {
            e.printStackTrace();
            //TODO
        }
    }

    //TODO
    private void setValue(EntityClassGroup entityClassGroup, EntityDeleted entityDeleted, String profile
            , PreparedStatement statement) throws SQLException {

        //add id
        statement.setLong(1, entityDeleted.getId());

        Map<String, Object> context = entityDeleted.getContext();
        Object attachment = context.get("attachment");
        String delUsername = null;
        long userid = 0L;
        if (attachment instanceof SystemAttachment) {
            delUsername = ((SystemAttachment) attachment).getDelUname();
            userid = ((SystemAttachment) attachment).getDelUId();
        }

        long currentEntityId = entityClassGroup.getEntityClass().id();
        Collection<IEntityClass> fatherEntityClass = entityClassGroup.getFatherEntityClass();
        int index = 2;
        Iterator<IEntityClass> iterator = fatherEntityClass.iterator();
        while (iterator.hasNext()) {
            IEntityClass next = iterator.next();
            statement.setLong(index, next.id());
            index++;
        }

        statement.setLong(index, currentEntityId);

        while (index++ < 7) {
            statement.setLong(index, 0L);
        }

        statement.setString(7, profile);

        Map<String, Object> body = entityDeleted.getData();

        List<String> keys = extractKeys(body, entityClassGroup);

        //add key
        statement.setString(8, keys.get(0));
        statement.setString(9, keys.get(1));
        statement.setString(10, keys.get(2));

        //create time
        LocalDateTime now = LocalDateTime.now();
        Instant instant = now.atZone(ZONE_ID).toInstant();
        statement.setLong(11, instant.toEpochMilli());

        //create user id
        statement.setLong(12, userid);
        statement.setString(13, delUsername);

        //attr

        try {
            String attr = JacksonDefaultMapper.OBJECT_MAPPER.writeValueAsString(body);
            statement.setString(14, attr);
        } catch (JsonProcessingException e) {
            e.printStackTrace();
        }
    }

    @TransactionalEventListener
    public void recordHistory(EntityDeleted entityDeleted) {
        try {
            Map<String, Object> data = entityDeleted.getData();
            if (data == null) {
                return;
            } else {
                QueueOfferResult join = queue.offer(entityDeleted).toCompletableFuture().join();
                //make this in
                if (join == QueueOfferResult.dropped()) {
                    //write to file to oss
                    //TODO
                }
            }
        } catch (Throwable throwable) {
            log.error("{}", throwable);
        }
    }
}
