package com.xforceplus.ultraman.extension.cluster.impl;

import com.xforceplus.ultraman.extension.cluster.ClusterNodeService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.jdbc.core.JdbcTemplate;

import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicLong;

import static com.xforceplus.ultraman.sdk.infra.logging.LoggingPattern.GENERAL_UTILS_ERROR;
import static com.xforceplus.ultraman.sdk.infra.logging.LoggingUtils.logErrorPattern;

/**
 * database implementation
 */
@Slf4j
public class DBClusterNodeServiceImpl implements ClusterNodeService {

    private static final String FIND_OQS = "SELECT ID FROM NODE_ID where node = ?";

    @Qualifier("main")
    @Autowired
    private JdbcTemplate mainJdbcTemplate;

    @Override
    public Integer getNodeId() {
        String name = currentNodeName();
        AtomicLong atomicLong = new AtomicLong();
        mainJdbcTemplate.query(FIND_OQS, ps -> {
            ps.setString(1, name);
        }, rs -> {
            long id = rs.getLong("id");
            atomicLong.set(id);

        });

        return Long.valueOf(atomicLong.get()).intValue();
    }

    private String currentNodeName() {
        String hostName = System.getenv("HOSTNAME");
        if (StringUtils.isEmpty(hostName)) {
            hostName = macName();
        }

        return Optional.ofNullable(hostName).orElseGet(() -> {
           return  UUID.randomUUID().toString();
        });
    }

    private String macName() {
        try {
            InetAddress ipAddress = InetAddress.getLocalHost();
            NetworkInterface networkInterface = NetworkInterface.getByInetAddress(ipAddress);

            if (networkInterface != null) {
                byte[] macAddressBytes = networkInterface.getHardwareAddress();

                if (macAddressBytes != null) {
                    StringBuilder macAddressBuilder = new StringBuilder();

                    for (byte b : macAddressBytes) {
                        macAddressBuilder.append(String.format("%02X:", b));
                    }

                    if (macAddressBuilder.length() > 0) {
                        macAddressBuilder.deleteCharAt(macAddressBuilder.length() - 1);
                    }

                    String macAddress = macAddressBuilder.toString();
                    return macAddress;
                } else {
                    return null;
                }
            } else {
                return null;
            }
        } catch (SocketException | UnknownHostException e) {
            logErrorPattern(log, GENERAL_UTILS_ERROR, "MacName", e);
            return null;
        }
    }
}
