package com.xforceplus.ultraman.extensions.messagecenter.listener;

import com.xforcecloud.noification.model.BaseResponse;
import com.xforcecloud.noification.model.MessageInfo;
import com.xforcecloud.noification.model.Scope;
import com.xforceplus.tech.base.core.context.ContextKeys;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.message.MetadataMessageSource;
import com.xforceplus.ultraman.sdk.core.bulk.exporter.listener.ExportEventAwareListener;
import com.xforceplus.ultraman.sdk.core.bulk.exporter.listener.ImportEventAwareListener;
import com.xforceplus.ultraman.sdk.core.event.EntityErrorExported;
import com.xforceplus.ultraman.sdk.core.event.EntityExported;
import com.xforceplus.ultraman.sdk.core.event.EntityImported;
import com.xforceplus.ultraman.sdk.infra.message.MessageConstants;
import com.xforceplus.ultraman.sdk.infra.utils.JacksonDefaultMapper;
import lombok.SneakyThrows;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.event.EventListener;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.scheduling.annotation.Async;
import org.springframework.web.client.RestTemplate;

import java.io.StringWriter;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.function.Supplier;

import static com.xforceplus.ultraman.sdk.infra.message.MessageConstants.*;

/**
 * message can config template
 */
public class MessageCenterBulkEventListener implements ExportEventAwareListener, ImportEventAwareListener {

    private Supplier<String> tokenSupplier;

    private String senderId;

    private String gatewayUrl;

    private String titleTemplate;
    private String contentTemplate;
    
    private MetadataMessageSource messageSource;

    private final RestTemplate restTemplate;

    private String routePattern = "%s/api/%s/message/v1/messages?appId=%s";

    private Logger logger = LoggerFactory.getLogger(MessageCenterBulkEventListener.class);

    private String contextPath = "";

    //    private final String defaultContentStr = "#set( $currentFileName = $fileName.split('-')[0] + '-' + $ldt.now().format($dtf.ofPattern('YYYYMMddHHmmSS')) ) \n" +
    //            "<a href='$downloadUrl'>$currentFileName</a>";

    private final String defaultContentStr = "<a href='$downloadUrl'>$fileName</a>";

    private DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    private boolean ignoreOnSync;

    /**
     * sendId is a appId for message
     * <p>
     * variable is download url and
     *
     * @param tokenSupplier
     * @param senderIdSupplier
     * @param gatewayUrl
     * @param restTemplate
     */
    public MessageCenterBulkEventListener(Supplier<String> tokenSupplier
            , Supplier<String> senderIdSupplier, Supplier<String> gatewayUrl
            , String contentTemplate, String titleTemplate, RestTemplate restTemplate
            , String contextPath, boolean ignoreOnSync, MetadataMessageSource messageSource) {
        //dynamic
        this.tokenSupplier = tokenSupplier;
        // only once
        this.senderId = senderIdSupplier.get();
        this.gatewayUrl = gatewayUrl.get();
        this.restTemplate = restTemplate;
        this.ignoreOnSync = ignoreOnSync;
        this.messageSource = messageSource;

        //MessageContent
        if (contentTemplate != null) {
            this.contentTemplate = contentTemplate;
        } else {
            this.contentTemplate = defaultContentStr;
        }

        if (titleTemplate != null) {
            this.titleTemplate = titleTemplate;
        } else {
            this.titleTemplate = messageSource.res(EXPORT_TITLE_SUCCESS_NAME);
        }

        this.contextPath = contextPath;

        Velocity.init();

        /**
         *   context.put("ldt", LocalDateTime.class);
         *   context.put("dtf", DateTimeFormatter.class);
         */
        //Velocity.addProperty("ldt", LocalDateTime.class);
        //Velocity.addProperty("dtf", DateTimeFormatter.class);
    }

    @SneakyThrows
    @Override
    @Async
    @EventListener(EntityErrorExported.class)
    public void errorListener(EntityErrorExported entityExported) {
        Map<String, Object> context = entityExported.getContext();

        if (context != null) {
            Object tenantIdObj = context.get(ContextKeys.LongKeys.TENANT_ID.name());
            Object userId = context.get(ContextKeys.LongKeys.ID.name());
            if (tenantIdObj != null) {
                Long tenantId = (Long) tenantIdObj;
                MessageInfo messageInfo = new MessageInfo();

                String fileName = entityExported.getFileName();
                String reason = entityExported.getReason();

                messageInfo.setScope(Scope.SINGLE);
                messageInfo.setTitle(messageSource.res(EXPORT_TITLE_ERROR_NAME));
                messageInfo.setContent(messageSource.res(EXPORT_CONTENT_ERROR_TPL, fileName, reason));
                messageInfo.setReceiverIds(Arrays.asList((Long) userId));
                messageInfo.setType(0);

                HttpHeaders headers = new HttpHeaders();
                MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
                headers.setContentType(type);
                headers.add("Accept", MediaType.APPLICATION_JSON.toString());
                headers.add("x-app-token", tokenSupplier.get());

                String finalAppId = Optional.ofNullable(entityExported.getNotifyContext())
                        .map(ctx -> ctx.get("appId"))
                        .map(Objects::toString).orElse(senderId);

                HttpEntity messageEntity = new HttpEntity<>(messageInfo, headers);
                String url = String.format(routePattern, gatewayUrl, tenantId, finalAppId);
                try {
                    logger.info(url);
                    logger.info(JacksonDefaultMapper.OBJECT_MAPPER.writeValueAsString(messageInfo));
                    ResponseEntity<BaseResponse> response = restTemplate.postForEntity(url, messageEntity, BaseResponse.class);
                    //TODO if check this response
                    if (response.hasBody()) {
                        logger.info("httpCode: {},response body: {}", response.getStatusCode().value(), JacksonDefaultMapper.OBJECT_MAPPER.writeValueAsString(response.getBody()));
                    }
                } catch (RuntimeException ex) {
                    logger.error("{}", ex);
                }
            }
        }
    }

    @SneakyThrows
    @Override
    @Async
    @EventListener(EntityExported.class)
    public void messageListener(EntityExported entityExported) {

        if ("sync".equalsIgnoreCase(entityExported.getExportType()) && ignoreOnSync) {
            //in sync
            return;
        } else if ("auto".equals(entityExported.getExportType()) && entityExported.isReturnBeforeDone() && ignoreOnSync) {
            return;
        }

        Map<String, Object> context = entityExported.getContext();

        //        ST content = new ST(this.contentTemplate, '$', '$');
        //        ST title = new ST(this.titleTemplate, '$', '$');
        //        ST defaultContent = new ST(defaultContentStr, '$', '$');

        if (context != null) {
            Object tenantIdObj = context.get(ContextKeys.LongKeys.TENANT_ID.name());
            Object userId = context.get(ContextKeys.LongKeys.ID.name());
            if (tenantIdObj != null) {
                Long tenantId = (Long) tenantIdObj;
                MessageInfo messageInfo = new MessageInfo();

                String downloadUrl = entityExported.getDownloadUrl();

                String finalDownloadUrl = contextPath + downloadUrl;

                String fileName = entityExported.getFileName();

                messageInfo.setScope(Scope.SINGLE);
                messageInfo.setTitle(getRendered(this.titleTemplate, fileName, finalDownloadUrl, entityExported.getEntityClassList(), () -> messageSource.res(EXPORT_TITLE_SUCCESS_NAME)));
                messageInfo.setContent(getRendered(this.contentTemplate, fileName, finalDownloadUrl, entityExported.getEntityClassList()
                        , () -> getRendered(defaultContentStr, finalDownloadUrl, fileName, entityExported.getEntityClassList(), () -> messageSource.res(FILE_DOWNLOAD_URL))));
                messageInfo.setReceiverIds(Arrays.asList((Long) userId));
                messageInfo.setType(0);

                HttpHeaders headers = new HttpHeaders();
                MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
                headers.setContentType(type);
                headers.add("Accept", MediaType.APPLICATION_JSON.toString());
                headers.add("x-app-token", tokenSupplier.get());

                String finalAppId = Optional.ofNullable(entityExported.getNotifyContext())
                        .map(ctx -> ctx.get("appId"))
                        .map(Objects::toString).orElse(senderId);

                HttpEntity messageEntity = new HttpEntity<>(messageInfo, headers);
                String url = String.format(routePattern, gatewayUrl, tenantId, finalAppId);
                try {
                    logger.info(url);
                    logger.info(JacksonDefaultMapper.OBJECT_MAPPER.writeValueAsString(messageInfo));
                    ResponseEntity<BaseResponse> response = restTemplate.postForEntity(url, messageEntity, BaseResponse.class);
                    //TODO if check this response
                    if (response.hasBody()) {
                        logger.info("httpCode: {},response body: {}", response.getStatusCode().value(), JacksonDefaultMapper.OBJECT_MAPPER.writeValueAsString(response.getBody()));
                    }
                } catch (RuntimeException ex) {
                    logger.error("{}", ex);
                }
            }
        }
    }


    private String getRendered(String template, String fileName, String downloadUrl, List<IEntityClass> entityClassList, Supplier<String> fallbackStr) {
        try {
            //            st.add("downloadUrl", downloadUrl);
            //            st.add("fileName", fileName);
            //            st.add("now", LocalDateTime.now());
            //            return st.render();

            VelocityContext context = new VelocityContext();
            context.put("fileName", fileName);
            context.put("downloadUrl", downloadUrl);
            context.put("entityCls", entityClassList);
            context.put("ldt", LocalDateTime.class);
            context.put("dtf", DateTimeFormatter.class);
            StringWriter writer = new StringWriter();
            Velocity.evaluate(context, writer, "info", template);
            return writer.toString();

        } catch (Exception ex) {
            logger.error("{}", ex);
            return fallbackStr.get();
        }
    }

    @SneakyThrows
    @Async
    @EventListener(EntityImported.class)
    @Override
    public void messageListener(EntityImported entityImported) {
        if ("sync".equalsIgnoreCase(entityImported.getImportType()) && ignoreOnSync) {
            //in sync
            return;
        }

        if (entityImported.getImportCmd().getOnlyCheck()) {
            return;
        }

        Map<String, Object> context = entityImported.getContext();
        if (context != null) {
            Object tenantIdObj = context.get(ContextKeys.LongKeys.TENANT_ID.name());
            Object userId = context.get(ContextKeys.LongKeys.ID.name());
            Long tenantId = (Long) tenantIdObj;
            MessageInfo messageInfo = new MessageInfo();
            messageInfo.setScope(Scope.SINGLE);
//            String title = "导入执行完成，文件名：" + entityImported.getFilename();
            messageInfo.setTitle(messageSource.res(IMPORT_TITLE_SUCCESS_NAME, entityImported.getFilename()));
//            String content = "完成于 " + LocalDateTime.now().format(formatter);
            messageInfo.setContent(messageSource.res(OPERATE_COMPLETE_TIME, LocalDateTime.now().format(formatter)));
            Map<String, Object> notifyContext = entityImported.getNotifyContext();
            if (notifyContext != null) {
                String errorFileDownloadUrl = (String) notifyContext.get("errorFileDownloadUrl");
                if (errorFileDownloadUrl != null) {
                    String finalDownloadUrl = contextPath + errorFileDownloadUrl;
                    if (errorFileDownloadUrl.startsWith("http")) {
                        finalDownloadUrl = errorFileDownloadUrl;
                    }
                    messageInfo.setContent(messageInfo.getContent()
                            + "</br>" + messageSource.res(IMPORT_ERROR_ITEM_DOWNLOAD) 
                            + getRendered(this.defaultContentStr, messageSource.res(IMPORT_ERROR_ITEM_DOWNLOAD_TITLE
                                    , entityImported.getFilename()), finalDownloadUrl, null
                            , () -> messageSource.res(FILE_DOWNLOAD_URL)));
                }
                Object importResultInfo = notifyContext.get("importResultInfo");
                if (importResultInfo != null) {
                    messageInfo.setContent(messageInfo.getContent()
                            + "</br>" + messageSource.res(IMPORT_RESULT_INFO) 
                            + importResultInfo.toString().replaceAll("\\n", "</br>"));
                }
            }
            HttpHeaders headers = new HttpHeaders();
            MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
            headers.setContentType(type);
            headers.add("Accept", MediaType.APPLICATION_JSON.toString());
            headers.add("x-app-token", tokenSupplier.get());

            String finalAppId = Optional.ofNullable(notifyContext)
                    .map(ctx -> ctx.get("appId"))
                    .map(Objects::toString).orElse(senderId);
            messageInfo.setReceiverIds(Arrays.asList((Long) userId));
            messageInfo.setType(0);

            HttpEntity messageEntity = new HttpEntity<>(messageInfo, headers);
            String url = String.format(routePattern, gatewayUrl, tenantId, finalAppId);
            try {
                logger.info(url);
                logger.info(JacksonDefaultMapper.OBJECT_MAPPER.writeValueAsString(messageInfo));
                ResponseEntity<BaseResponse> response = restTemplate.postForEntity(url, messageEntity, BaseResponse.class);
                //TODO if check this response
                if (response.hasBody()) {
                    logger.info("httpCode: {},response body: {}", response.getStatusCode().value(), JacksonDefaultMapper.OBJECT_MAPPER.writeValueAsString(response.getBody()));
                }
            } catch (RuntimeException ex) {
                logger.error("{}", ex);
            }
        }
    }
}
