package com.xforceplus.ultraman.oqsengine.plus.history.handler;

import com.xforceplus.ultraman.metadata.cdc.OqsEngineEntity;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.plus.common.iterator.DataIterator;
import com.xforceplus.ultraman.oqsengine.plus.history.dto.HistoryTaskInfo;
import com.xforceplus.ultraman.oqsengine.plus.history.dto.Status;
import com.xforceplus.ultraman.oqsengine.plus.history.sql.HistoryTaskStorage;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.MasterStorage;
import com.xforceplus.ultraman.sdk.infra.base.id.LongIdGenerator;
import com.xforceplus.ultraman.sdk.infra.base.id.SnowflakeLongIdGenerator;
import com.xforceplus.ultraman.sdk.infra.base.id.node.TimeRandomNodeIdGenerator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Resource;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutorService;

/**
 * Created by justin.xu on 12/2023.
 *
 * @since 1.8
 */
public class DefaultHistoryStorage implements HistoryStorage {

    final Logger logger = LoggerFactory.getLogger(DefaultHistoryStorage.class);

    private static final int BATCH_QUERY_SIZE = 2048;
    private int querySize = BATCH_QUERY_SIZE;

    @Resource
    private MasterStorage masterStorage;

    @Resource
    private EntityClassEngine engine;

    @Resource
    private HistoryTaskStorage historyTaskStorage;

    private ExecutorService asyncThreadPool;
    private LongIdGenerator idGenerator;


    /**
     * construct.
     */
    public DefaultHistoryStorage(int querySize, ExecutorService executorService) {
        if (querySize > 0) {
            this.querySize = querySize;
        }

        this.asyncThreadPool = executorService;
        this.idGenerator = new SnowflakeLongIdGenerator(new TimeRandomNodeIdGenerator());
    }



    @Override
    public void doDocumentation(IEntityClass entityClass, HistoryTaskInfo historyTaskInfo) throws SQLException {
        //  init
        logger.info("pending documentation task, batchId {}, taskId {}, entityClass {}, profile {}, start {}, end {}, op {}",
                historyTaskInfo.getBatchId(), historyTaskInfo.getTaskId(), entityClass.id(), entityClass.profile(),
                historyTaskInfo.getYear(), historyTaskInfo.getMonth(), HistoryTaskInfo.Op.DOCUMENTATION.name()
        );

        String table = entityClass.masterWriteTable(null != historyTaskInfo.getFilterProfile() &&
                !historyTaskInfo.getFilterProfile().isEmpty());

        historyTaskInfo.setTableName(table);

        if (0 == historyTaskStorage.build(historyTaskInfo)) {
            historyTaskInfo.setMessage("TASK NOT START, TASK BUILDING FAILED.");
        }

        asyncThreadPool.submit(() -> {
            handleDocumentationTask(historyTaskInfo, entityClass);
        });
    }


    @Override
    public Collection<HistoryTaskInfo> doDeletes(Collection<Long> batchId, Long userId, String userName) throws SQLException {

        Collection<HistoryTaskInfo> deletes = new ArrayList<>();
        for (Long id : batchId) {
            Collection<HistoryTaskInfo> batches =
                    historyTaskStorage.lists(id);

            for  (HistoryTaskInfo historyTaskInfo : batches) {
                historyTaskInfo.setDeleteTime(System.currentTimeMillis());
                historyTaskInfo.setDeleteUserName(userName);
                historyTaskInfo.setDeleteUser(userId);

                if (historyTaskInfo.getStatus() != Status.FINISH_DOCUMENTATION.getCode()) {
                    logger.warn("task not finish documentation, can't not handle delete task");
                    throw new RuntimeException("task not finish documentation, can't not handle delete task");
                }

                Optional<IEntityClass> entityClassOp =
                        engine.load(historyTaskInfo.getEntity() + "", historyTaskInfo.getProfile());

                if (!entityClassOp.isPresent()) {
                    historyTaskInfo.setMessage("DOCUMENTATIONS FAILED, ENTITY_CLASS NOT FOUND.");
                    historyTaskStorage.status(historyTaskInfo, Status.FINISH_DOCUMENTATION);
                    throw new RuntimeException("DOCUMENTATIONS FAILED, ENTITY_CLASS NOT FOUND.");
                }

                asyncThreadPool.submit(() -> {
                    handleDeleteTask(historyTaskInfo, entityClassOp.get());
                });

                deletes.add(historyTaskInfo);
            }
        }

        return deletes;
    }

    @Override
    public Collection<HistoryTaskInfo> list(long batchId) throws SQLException {
        return historyTaskStorage.lists(batchId);
    }

    @Override
    public Collection<HistoryTaskInfo> queryByTaskId(long taskId) throws SQLException {
        return historyTaskStorage.selectByTaskId(taskId);
    }

    private void handleDeleteTask(HistoryTaskInfo historyTaskInfo, IEntityClass entityClass) {
        DataIterator<Long> iterator = null;
        int staticStatus = historyTaskInfo.getStatus();
        try {
            iterator = masterStorage.iteratorIdHistory(entityClass, historyTaskInfo.getFilterProfile(),
                    historyTaskInfo.historyTable(), historyTaskInfo.getRangeStart(), historyTaskInfo.getRangeEnd(),
                    historyTaskInfo.getDeletePoint(), querySize);

            List<Long> ids = new ArrayList<>();

            int updateFlag = 0;
            int frequency = 10;

            historyTaskInfo.setStatus(Status.RUNNING_DELETE.getCode());
            historyTaskInfo.setMessage("DELETES RUN");
            historyTaskStorage.status(historyTaskInfo, Status.RUNNING_DELETE);

            boolean isCanceled = false;
            while (iterator.hasNext()) {
                long id = iterator.next();
                ids.add(id);

                if (ids.size() == querySize) {
                    int deleteSize = delete(ids, entityClass, historyTaskInfo.getRangeStart(), historyTaskInfo.getRangeEnd(),
                            historyTaskInfo.getFilterProfile(), historyTaskInfo.getTableName());

                    historyTaskInfo.setDeletePoint(id);
                    historyTaskInfo.addDeletes(deleteSize);

                    updateFlag++;
                    //  每拉10次更新一次任务状态.
                    if (updateFlag == frequency) {
                        if (0 == historyTaskStorage.update(historyTaskInfo)) {
                            isCanceled = true;
                            break;
                        }
                        updateFlag = 0;
                    }
                    ids.clear();
                }
            }

            if (!isCanceled) {
                if (!ids.isEmpty()) {
                    int deleteSize = delete(ids, entityClass, historyTaskInfo.getRangeStart(),
                            historyTaskInfo.getRangeEnd(), historyTaskInfo.getFilterProfile(), historyTaskInfo.getTableName());

                    historyTaskInfo.setDeletePoint(ids.get(ids.size() - 1));
                    historyTaskInfo.addDeletes(deleteSize);
                }
                historyTaskInfo.setMessage("DELETES FINISH");

                historyTaskStorage.status(historyTaskInfo, Status.FINISH_DELETE);
            }
        } catch (Throwable t) {
            historyTaskInfo.setMessage(String.format("DELETES FAILED, MESSAGE : %s", t.getMessage()));
            try {
                historyTaskStorage.status(historyTaskInfo, Status.toBatchStatus(staticStatus));
            } catch (SQLException e) {
                throw new RuntimeException(e);
            }
        } finally {
            if (iterator != null) {
                try {
                    iterator.destroy();
                } catch (Exception ex) {
                    logger.error(ex.getMessage(), ex);
                }
            }
        }
    }


    private void handleDocumentationTask(HistoryTaskInfo historyTaskInfo, IEntityClass entityClass) {

        DataIterator<OqsEngineEntity> iterator = null;
        try {
            //  错误的任务将直接设置为任务失败.
            if (historyTaskInfo.getStatus() == Status.ERROR.getCode()) {
                return;
            }

            int updateFlag = 0;
            int frequency = 10;

            iterator = masterStorage.iteratorEntityHistory(
                    entityClass,
                    historyTaskInfo.getFilterProfile(),
                    historyTaskInfo.getTableName(),
                    historyTaskInfo.getRangeStart(),
                    historyTaskInfo.getRangeEnd(),
                    historyTaskInfo.getCheckPoint(),
                    querySize);

            List<OqsEngineEntity> ids = new ArrayList<>();
            boolean isCanceled = false;
            while (iterator.hasNext()) {
                OqsEngineEntity v = iterator.next();

                ids.add(v);

                if (ids.size() == querySize) {
                    documentation(ids, entityClass, historyTaskInfo.getFilterProfile(), historyTaskInfo.historyTable());

                    historyTaskInfo.setCheckPoint(v.getId());
                    historyTaskInfo.addTotal(ids.size());
                    historyTaskInfo.addDocumentations(ids.size());
                    historyTaskInfo.setMessage("DOCUMENTATIONS RUN");

                    ids.clear();

                    updateFlag++;
                    //  每拉10次更新一次任务状态.
                    if (updateFlag == frequency) {
                        if (0 == historyTaskStorage.update(historyTaskInfo)) {
                            isCanceled = true;
                            break;
                        }
                        updateFlag = 0;
                    }
                }
            }

            if (!isCanceled) {
                if (!ids.isEmpty()) {
                    documentation(ids, entityClass, historyTaskInfo.getFilterProfile(), historyTaskInfo.historyTable());
                    historyTaskInfo.addTotal(ids.size());
                    historyTaskInfo.addDocumentations(ids.size());
                }
                historyTaskInfo.setMessage("DOCUMENTATIONS FINISH");

                historyTaskStorage.status(historyTaskInfo, Status.FINISH_DOCUMENTATION);
            }
        } catch (Throwable t) {
            historyTaskInfo.setMessage("DOCUMENTATIONS ERROR :" + t.getMessage());
            //  任务处理失败, 设置任务为失败状态.
            try {
                historyTaskStorage.status(historyTaskInfo, Status.ERROR);
            } catch (Exception ex) {
                //  打印错误，这个异常将被忽略.
                logger.error(ex.getMessage(), ex);
            }
        } finally {
            if (iterator != null) {
                try {
                    iterator.destroy();
                } catch (Exception ex) {
                    logger.error(ex.getMessage(), ex);
                }
            }
        }
    }

    public void documentation(List<OqsEngineEntity> oqsEngineEntities, IEntityClass entityClass, String filterProfile, String historyTable) {
        masterStorage.documentation(oqsEngineEntities, entityClass, filterProfile, historyTable);
    }

    public int delete(List<Long> ids, IEntityClass entityClass, long startTime, long endTime, String requestProfile, String table) {
        return masterStorage.batchDelete(ids, entityClass, startTime, endTime, requestProfile, table);
    }
}


