package com.xforceplus.ultraman.oqsengine.plus.history.sql;

import com.xforceplus.ultraman.oqsengine.plus.common.dto.BatchStatus;
import com.xforceplus.ultraman.oqsengine.plus.history.dto.HistoryTaskInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Optional;


/**
 * 任务储存命令.
 *
 * @author xujia 2020/8/24
 * @since 1.8
 */
public class TaskStorageCommand {

    final Logger logger = LoggerFactory.getLogger(TaskStorageCommand.class);

    private String tableName;

    public TaskStorageCommand(String tableName) {
        this.tableName = tableName;
    }

    /**
     * 查找活动任务.
     */
    public Collection<HistoryTaskInfo> selectActive(DataSource dataSource, long id) throws SQLException {
        try (Connection connection = dataSource.getConnection()) {
            return doSelectActive(connection, id);
        }
    }

    /**
     * 查询单独任务.
     */
    public Collection<HistoryTaskInfo> selectByTaskId(DataSource dataSource, long taskId) throws SQLException {
        String sql = String.format(SQL.SELECT_SQL_TASK_ID, tableName);

        try (Connection connection = dataSource.getConnection();
             PreparedStatement st = connection.prepareStatement(sql)) {
            st.setLong(1, taskId); // task_id
            if (logger.isDebugEnabled()) {
                logger.debug(st.toString());
            }
            return executeListResults(st);
        }
    }


    /**
     * 列出活动任务.
     */
    public Collection<HistoryTaskInfo> lists(DataSource dataSource, long batchId) throws SQLException {
        String selectSql = String.format(SQL.LIST_CURRENT_BATCH, tableName);

        try (Connection connection = dataSource.getConnection();
             PreparedStatement st = connection.prepareStatement(selectSql)) {
            st.setLong(1, batchId); //  batchId

            if (logger.isDebugEnabled()) {
                logger.debug(st.toString());
            }

            return executeListResults(st);
        }
    }



    /**
     * 构造任务执行.
     */
    public int build(DataSource dataSource, HistoryTaskInfo taskInfo) throws SQLException {
        String sql = String.format(SQL.BUILD_SQL, tableName);

        Connection connection = dataSource.getConnection();
        try {
            return doBuild(connection, sql, taskInfo);
        } catch (SQLException e) {
            throw e;
        } finally {
            connection.close();
        }
    }

    private Collection<HistoryTaskInfo> doSelectActive(Connection connection, long id) throws SQLException {
        String sql = String.format(SQL.SELECT_SQL_ACTIVE, tableName);
        try (PreparedStatement st = connection.prepareStatement(sql)) {
            st.setLong(1, id); // id

            return executeListResults(st);
        }
    }

    private int doBuild(Connection connection, String sql, HistoryTaskInfo taskInfo) throws SQLException {
        try (PreparedStatement st = connection.prepareStatement(sql)) {
            int pos = 1;
            st.setLong(pos++, taskInfo.getId());
            //  taskId
            st.setLong(pos++, taskInfo.getTaskId());
            //  batchId
            st.setLong(pos++, taskInfo.getBatchId());
            //  part
            st.setLong(pos++, taskInfo.getPart());
            //  entityClassId
            st.setLong(pos++, taskInfo.getEntity());
            //  profile
            st.setString(pos++, taskInfo.getProfile());
            //  filterProfile
            st.setString(pos++, taskInfo.getFilterProfile());
            // startTime
            st.setLong(pos++, taskInfo.getRangeStart());
            // endTime
            st.setLong(pos++, taskInfo.getRangeEnd());
            // batchStatus
            st.setLong(pos++, taskInfo.getStatus());
            //  year
            st.setInt(pos++, taskInfo.getYear());
            //  month
            st.setInt(pos++, taskInfo.getMonth());
            // batchCreateTime
            if (taskInfo.getCreateTime() == 0) {
                taskInfo.setCreateTime(System.currentTimeMillis());
            }
            st.setLong(pos++, taskInfo.getCreateTime());
            // checkpoint
            st.setLong(pos++, taskInfo.getCheckPoint());
            //  message
            st.setString(pos++, taskInfo.getMessage());
            //  op
            st.setInt(pos++, taskInfo.getOp());
            //  opUser
            st.setLong(pos++, taskInfo.getOpUser());
            //  opUserName
            st.setString(pos++, taskInfo.getOpUserName());
            //  table
            st.setString(pos++, taskInfo.getTableName());

            if (logger.isDebugEnabled()) {
                logger.debug(st.toString());
            }

            return st.executeUpdate();
        }
    }

    /**
     * 更新任务.
     */
    public int update(DataSource dataSource, HistoryTaskInfo taskInfo) throws SQLException {
        String sql = updateSql(taskInfo);

        try (Connection connection = dataSource.getConnection();
            PreparedStatement st = connection.prepareStatement(sql)) {

            int pos = 1;

            //  batchUpdateTime
            st.setLong(pos++, taskInfo.getUpdateTime());

            //  batchDeleteTime
            st.setLong(pos++, taskInfo.getDeleteTime());

            //  Status
            st.setInt(pos++, taskInfo.getStatus());

            //  batchSize
            st.setLong(pos++, taskInfo.getTotal());

            //  document size
            st.setLong(pos++, taskInfo.getDocuments());

            //  delete size
            st.setLong(pos++, taskInfo.getDeletes());

            //  checkPoint
            st.setLong(pos++, taskInfo.getCheckPoint());

            //  deletePoint
            st.setLong(pos++, taskInfo.getDeletePoint());

            //  message
            if (null != taskInfo.getMessage()) {
                st.setString(pos++, taskInfo.getMessage());
            }

            //  deleteUser
            st.setLong(pos++, taskInfo.getDeleteUser());

            //  deleteUserName
            st.setString(pos++, taskInfo.getDeleteUserName());

            //  taskId
            st.setLong(pos, taskInfo.getId());

            if (logger.isDebugEnabled()) {
                logger.debug(st.toString());
            }

            return st.executeUpdate();
        }
    }

    private String updateSql(HistoryTaskInfo taskInfo) {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("update ").append(tableName);
        stringBuilder.append(" set updatetime = ?");
        stringBuilder.append(",").append(" deletetime = ?");
        stringBuilder.append(",").append(" status = ?");
        stringBuilder.append(",").append(" total = ?");
        stringBuilder.append(",").append(" documents = ?");
        stringBuilder.append(",").append(" deletes = ?");
        stringBuilder.append(",").append(" checkpoint = ?");
        stringBuilder.append(",").append(" deletepoint = ?");

        if (null != taskInfo.getMessage()) {
            stringBuilder.append(", ").append("message = ?");
        }
        stringBuilder.append(",").append(" deleteuser = ?");
        stringBuilder.append(",").append(" deleteusername = ?");

        stringBuilder.append(" ").append("where id = ? and status not in (3, 4)");

        return stringBuilder.toString();
    }

    /**
     * 设置任务状态.
     */
    public int status(DataSource dataSource, long taskId, BatchStatus batchStatus, String message) throws SQLException {
        String sql = String.format(SQL.STATUS_SQL, tableName);

        try (Connection connection = dataSource.getConnection();
             PreparedStatement st = connection.prepareStatement(sql)) {
            //  batchUpdateTime
            st.setLong(1, System.currentTimeMillis());
            //  Status
            st.setInt(2, batchStatus.getCode());
            //  message
            st.setString(3, message);
            //  taskId
            st.setLong(4, taskId);

            if (logger.isDebugEnabled()) {
                logger.debug(st.toString());
            }

            return st.executeUpdate();
        }
    }

    private Collection<HistoryTaskInfo> executeListResults(PreparedStatement st) throws SQLException {
        try (ResultSet rs = st.executeQuery()) {
            if (logger.isDebugEnabled()) {
                logger.debug(st.toString());
            }

            Collection<HistoryTaskInfo> taskInfoList = new ArrayList<>();
            while (rs.next()) {
                taskInfoList.add(fillByResultSet(rs));
            }
            return taskInfoList;
        }
    }

    private HistoryTaskInfo fillByResultSet(ResultSet rs) throws SQLException {

        HistoryTaskInfo taskInfo = new HistoryTaskInfo(
                rs.getLong("id"), rs.getLong("batchid"),
                rs.getLong("taskid"), rs.getLong("entity"),
                rs.getString("profile"), rs.getInt("op"));

        taskInfo.setPart(rs.getInt("part"));
        taskInfo.setYear(rs.getInt("years"));
        taskInfo.setMonth(rs.getInt("months"));
        taskInfo.setTableName(rs.getString("tablename"));
        taskInfo.setFilterProfile(rs.getString("filterprofile"));
        taskInfo.setRangeStart(rs.getLong("starts"));
        taskInfo.setRangeEnd(rs.getLong("ends"));
        taskInfo.setStatus(rs.getInt("status"));
        taskInfo.setCreateTime(rs.getLong("createtime"));
        taskInfo.setUpdateTime(rs.getLong("updatetime"));
        taskInfo.setDeleteTime(rs.getLong("deletetime"));
        taskInfo.setTotal(rs.getLong("total"));
        taskInfo.setDocuments(rs.getLong("documents"));
        taskInfo.setDeletes(rs.getLong("deletes"));
        taskInfo.setCheckPoint(rs.getLong("checkpoint"));
        taskInfo.setDeletePoint(rs.getLong("deletepoint"));
        taskInfo.setDeleteTime(rs.getLong("deletetime"));
        taskInfo.setOpUser(rs.getLong("opuser"));
        taskInfo.setOpUserName(rs.getString("opusername"));
        taskInfo.setDeleteUser(rs.getLong("deleteuser"));
        taskInfo.setDeleteUserName(rs.getString("deleteusername"));
        taskInfo.setMessage(rs.getString("message"));

        return taskInfo;
    }

}

