package com.xforceplus.ultraman.sdk.infra.base.id;

import java.nio.charset.StandardCharsets;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.zip.CRC32;
import javax.sql.DataSource;

/**
 * 一个基于MYSQL的ID生成器.<br>
 * 其保证了同一命名空间的唯一性和连续性.<br>
 * 其依赖一个信息表如下.
 * <pre>
 *  CREATE TABLE `idinfo` (
 *   `id`         bigint       NOT NULL AUTO_INCREMENT COMMENT '从1开始的锁编号',
 *   `barrel`     json         NOT NULL                COMMENT '桶,用以记录不同命名空间的id最后值.',
 *   PRIMARY KEY (`id`)
 * ) ENGINE=InnoDB DEFAULT CHARSET=utf8 COMMENT='自增编号信息表';
 * </pre>
 * 其中每一行记录都是预先生成好的记录,其barrel的值为'{}',是一个JSON字段类型.<br>
 * 在生成ID时,会使用更新语句在 barrel 字段中增加一个JSON的KEY.<br>
 * 其使用了CRC32算法来平均分散每一个命名空间在不同的行记录上.<br>
 * 使用预生成的好处是防止在没有记录的情况下,并发创建造成的S锁升级为X锁的死锁.<br>
 *
 * @author dongbin
 * @version 0.1 2022/12/3 22:53
 * @since 1.8
 */
public class MysqlLongIdGenerator implements LongIdGenerator {

    /*
    表示成功的操作数量.
     */
    private static final int SUCCESS = 1;
    /*
     * 默认的命名空间.
     */
    private static final String DEFAULT_NAMESPACE = "d";

    /*
    所有记录ID的JSON KEY 的前辍.
     */
    private static final String KEY_PREIFX = "i";

    /*
    默认的表名.
     */
    private static final String DEFAULT_TABLE_NAME = "idinfo";

    /*
    默认自增使用的变量名.
     */
    private static final String DEFAULT_VARIABLE_NAME = "xplatid";

    /*
    默认的slot数量.
     */
    private static final long DEFAULT_SLOT_SIZE = 1000;

    /*
    操作等待时间.
     */
    private static final int TIMEOUT_SECOND = 3;

    /*
    自增的SQL.
    如果不存在将从1开始.
    表名,储存新值的变量名, 命名空间,命名空间, slot.
     */
    private static final String INCR_SQL =
        "UPDATE %s SET barrel = @%s := JSON_SET(barrel, '$.\"%s\"', cast(IFNULL(barrel->>'$.\"%s\"', 0) as SIGNED) + 1) where id = %d";

    /*
    重置指定命名空间的计算为1开始.
    表名, 命名空间, slot.
     */
    private static final String RESET_SQL =
        "UPDATE %s SET barrel = JSON_REMOVE(barrel, '$.\"%s\"') where id = %d";

    /*
    读取自增值SQL.
    变量名,命名空间名.
     */
    private static final String READ_VARIABLE_SQL = "select JSON_UNQUOTE(JSON_EXTRACT(@%s,'$.\"%s\"')) AS v";

    private DataSource dataSource;
    private String tableName;

    private String variableName;

    private long slotSize;


    private MysqlLongIdGenerator() {
    }

    @Override
    public Long next() {
        return next(DEFAULT_NAMESPACE);
    }

    @Override
    public Long next(String nameSpace) {
        checkNs(nameSpace);
        try {
            return doMysqlIncr(wrapperNs(nameSpace));
        } catch (SQLException ex) {
            throw new RuntimeException(ex.getMessage(), ex);
        }
    }

    @Override
    public void reset() {
        reset(DEFAULT_NAMESPACE);
    }

    @Override
    public void reset(String nameSpace) {
        checkNs(nameSpace);
        try {
            if (!doMysqlReset(wrapperNs(nameSpace))) {
                if (DEFAULT_NAMESPACE == nameSpace) {
                    throw new IllegalStateException(String.format("Cannot reset the default namespace correctly."));
                } else {
                    throw new IllegalStateException(
                        String.format("Cannot reset the %s namespace correctly.", nameSpace));
                }
            }
        } catch (SQLException ex) {
            throw new RuntimeException(ex.getMessage(), ex);
        }
    }

    @Override
    public boolean supportNameSpace() {
        return true;
    }

    @Override
    public boolean isContinuous() {
        return true;
    }

    @Override
    public boolean isPartialOrder() {
        return true;
    }

    private void checkNs(String ns) {
        if (ns == null || ns.isEmpty()) {
            throw new IllegalArgumentException("Invalid namespace.");
        }
    }

    // 等于0表示自增失败.
    private long doMysqlIncr(String ns) throws SQLException {
        long slot = calculateSlot(ns);
        // 表名,储存新值的变量名, 命名空间,命名空间, id.
        String incrSql = String.format(INCR_SQL, tableName, variableName, ns, ns, slot);
        // 新的自增值.
        long newValue = 0;
        try (Connection conn = dataSource.getConnection()) {
            conn.setAutoCommit(false);

            try (Statement st = conn.createStatement()) {
                st.setQueryTimeout(TIMEOUT_SECOND);

                int size = st.executeUpdate(incrSql);

                if (size == SUCCESS) {
                    // 变量名,命名空间名.
                    String readSql = String.format(READ_VARIABLE_SQL, variableName, ns);
                    try (ResultSet rs = st.executeQuery(readSql)) {
                        rs.next();

                        newValue = rs.getLong(1);
                    }
                }
            }
            conn.commit();
        }

        return newValue;
    }

    private boolean doMysqlReset(String ns) throws SQLException {
        long slot = calculateSlot(ns);
        // 表名, 命名空间, slot.
        int size;
        String resetSql = String.format(RESET_SQL, tableName, ns, slot);
        try (Connection conn = dataSource.getConnection()) {
            conn.setAutoCommit(false);

            try (Statement st = conn.createStatement()) {
                st.setQueryTimeout(TIMEOUT_SECOND);

                size = st.executeUpdate(resetSql);
            }

            conn.commit();
        }

        // 1 > 0 error
        return SUCCESS == size;
    }

    private long readMysqlSlotSize() throws SQLException {
        try (Connection conn = dataSource.getConnection()) {
            try (Statement st = conn.createStatement()) {
                try (ResultSet rs = st.executeQuery(String.format("SELECT COUNT(*) FROM %s", tableName))) {
                    rs.next();
                    return rs.getLong(1);
                }
            }
        }
    }

    private String wrapperNs(String ns) {
        return String.format("%s_%s", KEY_PREIFX, ns);
    }

    private long calculateSlot(String key) {
        CRC32 crc32 = new CRC32();
        byte[] bytes = key.getBytes(StandardCharsets.UTF_8);
        crc32.update(bytes);
        long value = crc32.getValue();
        long slot = value % slotSize;
        if (slot == 0) {
            return slot + 1;
        } else {
            return slot;
        }
    }

    /**
     * 构造器.
     */
    public static final class Builder {
        private DataSource dataSource;
        private String tableName = DEFAULT_TABLE_NAME;
        private String variableName = DEFAULT_VARIABLE_NAME;
        private long slotSize = DEFAULT_SLOT_SIZE;

        private Builder() {}

        public static Builder anMysqlLongIdGenerator() {
            return new Builder();
        }

        public Builder withDataSource(DataSource dataSource) {
            this.dataSource = dataSource;
            return this;
        }

        public Builder withTableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public Builder withVariableName(String variableName) {
            this.variableName = variableName;
            return this;
        }

        public Builder withSlotSize(long slotSize) {
            this.slotSize = slotSize;
            return this;
        }

        /**
         * 构造实例.
         */
        public MysqlLongIdGenerator build() {
            MysqlLongIdGenerator mysqlLongIdGenerator = new MysqlLongIdGenerator();
            mysqlLongIdGenerator.dataSource = this.dataSource;
            mysqlLongIdGenerator.slotSize = this.slotSize;
            mysqlLongIdGenerator.variableName = this.variableName;
            mysqlLongIdGenerator.tableName = this.tableName;

            long size = 0;
            try {
                size = mysqlLongIdGenerator.readMysqlSlotSize();
            } catch (SQLException e) {
                throw new RuntimeException(e.getMessage(), e);
            }

            if (this.slotSize > size) {
                throw new RuntimeException(
                    String.format("The %s slot is set up, but there are only %s slots.", this.slotSize, size));
            }
            return mysqlLongIdGenerator;
        }
    }
}
