package com.xforceplus.ultraman.sdk.infra.base.timerwheel;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.IntStream;
import javax.annotation.PreDestroy;

import com.xforceplus.ultraman.sdk.infra.base.thread.ExecutorHelper;
import org.jctools.queues.MpscUnboundedArrayQueue;

/**
 * 时间轮转算法实现.<br>
 * 所有的时间单位都以毫秒为单位.<br>
 * 默认分隔为512个槽位,每一个槽位时间区间为100毫秒.<br>
 * <em>注意: 相同的对象以equals比较.</em>
 *
 * @param <T> 管理的元素类型.
 * @author dongbin
 * @version 1.0 2020-03-10 11:53:19
 * @since 1.5
 */
public class TimerWheel<T> implements ITimerWheel<T> {

    private static final int DEFAULT_SLOT_NUMBER = 512;
    private static final int DEFAULT_DURATION = 100;

    /*
     * 时间单位
     */
    private final TimeUnit timeUnit = TimeUnit.MILLISECONDS;
    /*
     * 每个slot间隔时间
     */
    private final long duration;
    private final TimeoutNotification<T> notification;
    private final int slotNumber;
    private final Slot[] slots;
    private int currentSlot;
    /*
    执行线程.
     */
    private final ExecutorService worker;

    /*
    等待加入slot的队列.
     */
    private final Queue<Timeout> waitingAddQueue;
    /*
    元素内部速查表.
     */
    private final Map<T, Timeout> valueTimeoutIndex;

    /**
     * 以默认的512个槽位,和100毫秒为间隔,并失效不通知构造一个新的时间轮.
     */
    public TimerWheel() {
        this(-1, -1, null);
    }

    /**
     * 以默认的512个槽位,和100毫秒为间隔,失效会进行通知.
     *
     * @param notification 失效通知器.
     */
    public TimerWheel(TimeoutNotification<T> notification) {
        this(-1, -1, notification);
    }

    /**
     * 指定槽位数量,指定间隔毫秒数,并且不通知失效.
     *
     * @param slotNumber 槽位数量
     * @param duration   槽位间隔毫秒.
     */
    public TimerWheel(int slotNumber, long duration) {
        this(slotNumber, duration, null);
    }

    /**
     * 指定槽位数量,指定间隔毫秒数,并且通知失效.
     *
     * @param slotNumber   槽位数量.
     * @param duration     槽位间隔毫秒.
     * @param notification 失效通知器.
     */
    public TimerWheel(int slotNumber, long duration, TimeoutNotification<T> notification) {

        if (duration <= 0) {
            this.duration = DEFAULT_DURATION;
        } else {
            this.duration = duration;
        }

        if (slotNumber <= 3) {
            this.slotNumber = DEFAULT_SLOT_NUMBER;
        } else {
            this.slotNumber = slotNumber;
        }

        if (notification == null) {
            this.notification = t -> TimeoutNotification.OVERDUE;
        } else {
            this.notification = notification;
        }

        this.currentSlot = 0;

        slots = IntStream.range(0, this.slotNumber).mapToObj(i -> new Slot()).toArray(Slot[]::new);

        waitingAddQueue = new MpscUnboundedArrayQueue(16);
        valueTimeoutIndex = new ConcurrentHashMap<>(16);

        // 只会提交一个任务.
        worker = new ThreadPoolExecutor(
            1,
            1,
            0L,
            TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue(),
            ExecutorHelper.buildNameThreadFactory("time-wheel", true)
        );

        worker.submit(new PointTask(this.duration));
    }

    @PreDestroy
    @Override
    public void destroy() throws Exception {
        this.worker.shutdownNow();
    }

    /**
     * 增加一个在指定时间过期的对象.
     *
     * @param target   需要过期的对象.
     * @param duration 存活的持续时间.
     */
    @Override
    public void add(T target, long duration) {
        if (target == null) {
            throw new NullPointerException("Target object is null!");
        }
        if (duration <= 0) {
            return;
        }

        if (!valueTimeoutIndex.containsKey(target)) {

            Timeout<T> timeout = new Timeout(target, duration);

            waitingAddQueue.offer(timeout);
            valueTimeoutIndex.put(target, timeout);
        }
    }

    /**
     * 增加一个在指定时间过期的目标.
     *
     * @param target     目标实例.
     * @param expireDate 到期时间.
     */
    @Override
    public void add(T target, Date expireDate) {
        add(target, expireDate.getTime() - System.currentTimeMillis());
    }

    /**
     * 判断是否存在指定对象.
     *
     * @param target 要检查的目标对象.
     * @return true存在, false不存在.
     */
    @Override
    public boolean exist(T target) {
        if (target == null) {
            return false;
        }

        return this.valueTimeoutIndex.containsKey(target);
    }

    /**
     * 从环中删除目标.不会触发过期回调.
     *
     * @param target 目标.
     */
    @Override
    public void remove(T target) {
        if (target == null) {
            return;
        }

        Timeout<T> timeout = this.valueTimeoutIndex.get(target);
        if (timeout != null) {
            timeout.cancel();
            /*
            元素仍然存在于slot中,等待执行线程清理.
             */
            this.valueTimeoutIndex.remove(target);
        }
    }

    /**
     * 当前环中总共持有过期目标.
     *
     * @return 总共还有多少未过期目标.
     */
    @Override
    public int size() {
        return valueTimeoutIndex.size();
    }

    //计算round
    private int calculateRuound(int virtualSlot) {
        return virtualSlot / slotNumber;
    }

    private int calculateActuallySlot(int virtualSlot) {
        return virtualSlot % slotNumber;
    }

    //计算虚似slot.会超过最大slot位.
    private int calculateVirtualSlot(long timeout) {
        return (int) (currentSlot + timeout / duration);
    }

    private class Slot<T> {

        private final List<Timeout<T>> timeouts = new ArrayList<>();

        public void add(Timeout<T> timeout) {

            timeouts.add(timeout);

        }

        public List<Timeout<T>> expire() {
            List<Timeout<T>> expireList = new ArrayList<>();
            Iterator<Timeout<T>> iter = timeouts.iterator();
            Timeout<T> timeout;
            while (iter.hasNext()) {
                timeout = iter.next();
                if (timeout.getRound() <= 0) {
                    expireList.add(timeout);
                    iter.remove();
                } else {
                    timeout.reduceRound();
                }

            }
            return expireList;
        }

        public void remove(Object target) {
            final int notFound = -1;
            int index = notFound;
            for (int i = 0; i < timeouts.size(); i++) {
                if (timeouts.get(i).getValue().equals(target)) {
                    index = i;
                    break;
                }
            }
            if (index > notFound) {
                timeouts.remove(index);
            }
        }

        @Override
        public String toString() {
            return "Slot{" + "timeouts=" + timeouts + '}';
        }

    }

    /*
    超时淘汰元素.
     */
    private static class Timeout<T> {

        private boolean cancelled;
        private int round;
        private final long duration;
        private final T value;

        public Timeout(T value, long duration) {
            this.value = value;
            this.duration = duration;
            this.round = 0;
        }

        public T getValue() {
            return value;
        }

        public long getDuration() {
            return duration;
        }

        public int getRound() {
            return round;
        }

        public void initRound(int round) {
            this.round = round;
        }

        public int reduceRound() {
            return this.round--;
        }

        public void cancel() {
            this.cancelled = true;
        }

        public boolean isCancelled() {
            return cancelled;
        }
    }

    private class PointTask implements Runnable {
        private final long duration;

        public PointTask(long duration) {
            this.duration = duration;
        }

        @Override
        public void run() {

            List<Timeout<T>> expireList;
            Slot<T> slot;
            do {

                slot = slots[currentSlot];
                expireList = slot.expire();

                // 新增加的任务.
                processWaitingAddTimeout();

                currentSlot = (currentSlot + 1) % slotNumber;


                long resultTime;
                try {
                    for (Timeout<T> timeout : expireList) {
                        // 取消的任务不触发回调.
                        if (!timeout.isCancelled()) {
                            resultTime = notification.notice(timeout.getValue());
                            valueTimeoutIndex.remove(timeout.getValue());
                            if (resultTime > TimeoutNotification.OVERDUE) {
                                add(timeout.getValue(), resultTime);
                            }
                        }
                    }
                } catch (Throwable ex) {

                    //不处理任何异常,这里只是为了防止线程意外中止.
                    ex.printStackTrace(System.err);
                }

                try {
                    timeUnit.sleep(duration);
                } catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                }

            } while (!Thread.interrupted());
        }

        // 处理需要增加的元素.
        private void processWaitingAddTimeout() {
            Timeout<T> timeout = null;
            for (; ; ) {
                timeout = waitingAddQueue.poll();
                if (timeout == null) {
                    break;
                } else {
                    if (!timeout.isCancelled()) {
                        long specialTimeout = timeout.getDuration();
                        if (specialTimeout < this.duration) {
                            specialTimeout = this.duration;
                        }
                        int virtualSlotIndex = calculateVirtualSlot(specialTimeout);
                        int actuallySlotIndex = calculateActuallySlot(virtualSlotIndex);
                        int round = calculateRuound(virtualSlotIndex);

                        timeout.initRound(round);

                        Slot slot = slots[actuallySlotIndex];
                        slot.add(timeout);
                    }
                }
            }
        }

    }

}
