package com.xforceplus.ultraman.metadata.sync.grpc.autoconfigure;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.util.DefaultInstantiatorStrategy;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.protobuf.util.JsonFormat;
import com.xforceplus.ultraman.config.ConfigConverter;
import com.xforceplus.ultraman.config.ConfigurationEngine;
import com.xforceplus.ultraman.config.EventStrategy;
import com.xforceplus.ultraman.config.event.ChangeList;
import com.xforceplus.ultraman.config.json.JsonConfigNode;
import com.xforceplus.ultraman.config.storage.ConfigurationStorage;
import com.xforceplus.ultraman.config.storage.impl.DefaultFileConfigurationStorage;
import com.xforceplus.ultraman.config.storage.impl.DefaultInMemoryConfigurationStorage;
import com.xforceplus.ultraman.config.strategy.DiscardStrategy;
import com.xforceplus.ultraman.config.strategy.VersiondDiscardStrategy;
import com.xforceplus.ultraman.config.strategy.impl.DefaultJsonEventStrategy;
import com.xforceplus.ultraman.metadata.component.GlobalInited;
import com.xforceplus.ultraman.metadata.domain.VersionedJsonConfig;
import com.xforceplus.ultraman.metadata.grpc.DictUpResult;
import com.xforceplus.ultraman.metadata.grpc.ModuleUpResult;
import com.xforceplus.ultraman.metadata.sync.grpc.ConfigType;
import com.xforceplus.ultraman.metadata.sync.grpc.config.ExecutionConfig;
import com.xforceplus.ultraman.metadata.sync.grpc.event.ConfigChangeEvent;
import com.xforceplus.ultraman.metadata.sync.grpc.listener.ConfigListener;
import com.xforceplus.ultraman.metadata.sync.grpc.listener.ModuleEventListener;
import com.xforceplus.ultraman.sdk.infra.base.AuthConfig;
import com.xforceplus.ultraman.sdk.infra.logging.LoggingPattern;
import lombok.extern.slf4j.Slf4j;
import org.objenesis.strategy.StdInstantiatorStrategy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.SmartInitializingSingleton;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;

import java.io.IOException;
import java.util.List;
import java.util.Optional;

import static com.xforceplus.ultraman.sdk.infra.logging.LoggingUtils.logErrorPattern;

/**
 * runtime configuration
 */
@Slf4j
@Order(Ordered.HIGHEST_PRECEDENCE)
@ConditionalOnProperty(value = "xplat.oqsengine.sdk.enabled", matchIfMissing = true)
@Configuration
public class RuntimeConfigAutoConfiguration {

    private Logger logger = LoggerFactory.getLogger(RuntimeConfigAutoConfiguration.class);

    @ConditionalOnMissingBean(Kryo.class)
    @Bean
    public Kryo kryo() {
        Kryo kryo = new Kryo();
        kryo.setInstantiatorStrategy(new DefaultInstantiatorStrategy(new StdInstantiatorStrategy()));
        kryo.setRegistrationRequired(false);
        return kryo;
    }

    //TODO config
    @ConditionalOnMissingBean(EventStrategy.class)
    @Bean
    public EventStrategy jsonJsonEventStrategy() {
        return new DefaultJsonEventStrategy();
    }

    //TODO config
    @ConditionalOnMissingBean(DiscardStrategy.class)
    @Bean
    public DiscardStrategy discardStrategy() {
        return new VersiondDiscardStrategy();
    }

    @ConditionalOnMissingBean(ObjectMapper.class)
    @Bean
    public ObjectMapper objectMapper() {
        ObjectMapper mapper = new ObjectMapper();

        //Ignore
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        return mapper;
    }

    @ConditionalOnMissingBean(ConfigurationStorage.class)
    @ConditionalOnProperty(value = "xplat.oqsengine.sdk.config.mem.enabled", matchIfMissing = true)
    @ConditionalOnBean(value = {EventStrategy.class, DiscardStrategy.class})
    @Bean
    public ConfigurationStorage memStorage(
            EventStrategy eventStratregy
            , DiscardStrategy discardStrategy
    ) {
        ConfigurationStorage memStorage = new DefaultInMemoryConfigurationStorage(
                eventStratregy
                , discardStrategy);

        return memStorage;
    }

    @ConditionalOnMissingBean(ConfigurationStorage.class)
    @ConditionalOnProperty(value = "xplat.oqsengine.sdk.config.file.enabled", matchIfMissing = false)
    @ConditionalOnBean(value = {EventStrategy.class, DiscardStrategy.class})
    @Bean
    public ConfigurationStorage fileStorage(@Value("${xplat.oqsengine.sdk.config.file.root:/}") String root
            , Kryo kryo
            , EventStrategy eventStratregy
            , DiscardStrategy discardStrategy
    ) {
        ConfigurationStorage fileStorage = new DefaultFileConfigurationStorage(
                root
                , kryo
                , eventStratregy
                , discardStrategy);

        return fileStorage;
    }

    // @ConditionalOnBean(value = { Kryo.class, ConfigurationStorage.class })
    @ConditionalOnProperty(value = "xplat.oqsengine.sdk.config.module.enabled", matchIfMissing = true)
    @Bean("moduleConfigEngine")
    public ConfigurationEngine<ModuleUpResult, JsonConfigNode> engineForModule(ConfigurationStorage storage, ObjectMapper mapper) {

        ConfigurationEngine<ModuleUpResult, JsonConfigNode> engine = new ConfigurationEngine<>();
        engine.setConfigurationStorage(storage);

        ConfigConverter<ModuleUpResult, JsonConfigNode> converter = customConfig -> {
            try {
                String json = JsonFormat.printer().print(customConfig);
                return new VersionedJsonConfig(customConfig.getVersion(), ConfigType.BO.name(), "" + customConfig.getId(), mapper.readTree(json), null);
            } catch (IOException e) {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                        , e);
            }
            return null;
        };

        engine.setConverter(converter);
        return engine;
    }

    @ConditionalOnProperty(value = "xplat.oqsengine.sdk.config.dict.enabled", matchIfMissing = true)
    @Bean("dictConfigEngine")
    public ConfigurationEngine<DictUpResult, JsonConfigNode> engineForDict(ConfigurationStorage storage, ObjectMapper mapper, AuthConfig authConfig) {

        ConfigurationEngine<DictUpResult, JsonConfigNode> engine = new ConfigurationEngine<>();
        engine.setConfigurationStorage(storage);


        ConfigConverter<DictUpResult, JsonConfigNode> converter = customConfig -> {
            try {
                String json = JsonFormat.printer().print(customConfig);
                if (!customConfig.getDictsList().isEmpty()) {
                    return new VersionedJsonConfig(customConfig.getDicts(0).getVersion(), ConfigType.DICT.name(), authConfig.getAppId(), mapper.readTree(json), null);
                }
            } catch (IOException e) {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR
                        , e);
            }
            return null;
        };

        engine.setConverter(converter);

        return engine;
    }

    @Bean
    public Object configurationRebuild(ConfigurationStorage configurationStorage
            , Optional<ConfigurationEngine<DictUpResult, JsonConfigNode>> dictConfigEngineOp
            , Optional<ConfigurationEngine<ModuleUpResult, JsonConfigNode>> moduleConfigEngineOp
            , ApplicationEventPublisher publisher
            , GlobalInited inited
    ) {
        return new SmartInitializingSingleton() {

            @Override
            public void afterSingletonsInstantiated() {
                List<ChangeList> changeList = configurationStorage
                        .rebuild();

                logger.info("Rebuilding-------------------");

                changeList
                        .stream()
                        .forEach(x -> {
                            try {
                                publisher.publishEvent(new ConfigChangeEvent(x.getType(), x));
                            } catch (Exception ex) {
                                logger.error("{}", ex);
                            }
                        });

                dictConfigEngineOp.map(ConfigurationEngine::getObservable).ifPresent(ob -> {
                    ob.subscribe(x -> {
                        logger.info("Get New Dict-------------------");
                        publisher.publishEvent(new ConfigChangeEvent(x.getType(), x));
                    }, throwable -> {
                        logger.error("{}", throwable.getMessage());
                    });
                });

                moduleConfigEngineOp.map(ConfigurationEngine::getObservable).ifPresent(ob -> {
                    ob.subscribe(x -> {
                        logger.info("Get New Module List-------------------");
                        publisher.publishEvent(new ConfigChangeEvent(x.getType(), x));
                        logger.info("Should inserted");
                    }, throwable -> {
                        logger.error("{}", throwable.getMessage());
                    });
                });

                logger.info("Waiting for init");
                inited.waiting();
            }
        };
    }


    @Bean
    public ConfigListener configListener(AuthConfig authConfig){
        return new ConfigListener(authConfig);
    }

    @Bean
    public ModuleEventListener moduleEventListener(ExecutionConfig config, AuthConfig authConfig){
        return new ModuleEventListener(config, authConfig);
    }
}
