package com.xforceplus.ultraman.metadata.sync.grpc.store.repository.impl;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Sets;
import com.xforceplus.ultraman.metadata.domain.vo.dto.*;
import com.xforceplus.ultraman.metadata.domain.vo.dto.ApiDetails;
import com.xforceplus.ultraman.metadata.entity.*;
import com.xforceplus.ultraman.metadata.entity.legacy.impl.EntityClass;
import com.xforceplus.ultraman.metadata.entity.legacy.impl.Relation;
import com.xforceplus.ultraman.metadata.grpc.*;
import com.xforceplus.ultraman.metadata.sync.grpc.store.VersionService;
import com.xforceplus.ultraman.metadata.sync.grpc.store.impl.BoNode;
import com.xforceplus.ultraman.metadata.sync.grpc.store.impl.DefaultVersionService;
import com.xforceplus.ultraman.metadata.sync.grpc.store.impl.tables.*;
import com.xforceplus.ultraman.metadata.sync.grpc.store.repository.LegacyMetadataRepository;
import com.xforceplus.ultraman.metadata.sync.grpc.store.utils.FieldHelper;
import com.xforceplus.ultraman.metadata.sync.grpc.store.utils.RowUtils;
import com.xforceplus.ultraman.metadata.sync.grpc.utils.VersionUtils;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.metamodel.UpdateSummary;
import org.apache.metamodel.UpdateableDataContext;
import org.apache.metamodel.data.DataSet;
import org.apache.metamodel.data.Row;
import org.apache.metamodel.delete.DeleteFrom;
import org.apache.metamodel.insert.InsertInto;
import org.apache.metamodel.pojo.MapTableDataProvider;
import org.apache.metamodel.pojo.PojoDataContext;
import org.apache.metamodel.pojo.TableDataProvider;
import org.apache.metamodel.schema.Table;
import org.apache.metamodel.util.SimpleTableDef;
import org.springframework.context.ApplicationEventPublisher;

import java.util.*;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.ultraman.metadata.sync.grpc.store.utils.RowUtils.getRowValue;

/**
 * legacy implementation
 * TODO abstract this class with pojo
 * TODO refactor this more typed
 * TODO since the store is not thread-safe using read / write lock instead of sync keywords
 */
@Slf4j
public class MetadataRepositoryInMemoryImpl implements LegacyMetadataRepository {

    private ReentrantReadWriteLock rwLock = new ReentrantReadWriteLock();

    private VersionService versionService;

    private int maxVersion = 3;

    private static String DEFAULT_PROFILE = "";

    private ObjectMapper mapper;

    private static String[] preserved = new String[]{"query", "queryOne", "create", "update", "delete", "batchUpdate", "batchAdd", "batchDelete"};

    public MetadataRepositoryInMemoryImpl() {
        this(null, -1, null);
    }

    public MetadataRepositoryInMemoryImpl(ObjectMapper mapper, int maxVersion, ApplicationEventPublisher publisher) {

        if (maxVersion > 0) {
            this.maxVersion = maxVersion;
        }

        this.versionService = new DefaultVersionService(this.maxVersion, publisher);
        this.versionService.initVersionedDC(this.maxVersion, this::generateNewDC);
        if (mapper == null) {
            mapper = new ObjectMapper();
        }
        this.mapper = mapper;
    }

    @Override
    public List<String> codes() {
        return read(() -> {
            return versionService.getBoModuleMapping().entrySet().stream().map(x -> x.getKey().getCode()).collect(Collectors.toList());
        });
    }

    /**
     * generate the new pojo updateContext
     *
     * @return
     */
    private UpdateableDataContext generateNewDC() {

        TableDataProvider[] tableDataProviders = Stream.of(
                        new ModuleTable()
                        , new BoTable()
                        , new ApiTable()
                        , new FieldTable()
                        , new RelationTable()
                        , new AppTable())
                .map(x -> {
                    SimpleTableDef tableDef = new SimpleTableDef(x.name(), x.columns());
                    return new MapTableDataProvider(tableDef, x.getStore());
                }).toArray(TableDataProvider[]::new);

        return new PojoDataContext("metadata", tableDataProviders);
    }

    private Map<String, ApiItem> toApiItemMap(DataSet apis) {
        Map<String, ApiItem> map = new HashMap<>();
        while (apis.next()) {
            Row row = apis.getRow();
            String code = getRowValue(row, ApiTable.CODE).map(String::valueOf).orElse("");
            String url = getRowValue(row, ApiTable.URL).map(String::valueOf).orElse("");
            String method = getRowValue(row, ApiTable.METHOD).map(String::valueOf).orElse("");
            ApiItem apiItem = new ApiItem(url, method);
            map.put(code, apiItem);
        }
        return map;
    }

    private List<FieldItem> toFieldItemList(DataSet fields) {
        List<FieldItem> items = new ArrayList<>();

        while (fields.next()) {
            Row row = fields.getRow();
            FieldItem fieldItem = new FieldItem();
            fieldItem.setCode(getRowValue(row, FieldTable.CODE).map(String::valueOf).orElse(""));
            fieldItem.setDisplayType(getRowValue(row, FieldTable.DISPLAY_TYPE).map(String::valueOf).orElse(""));
            fieldItem.setEditable(getRowValue(row, FieldTable.EDITABLE).map(String::valueOf).orElse(""));
            fieldItem.setEnumCode(getRowValue(row, FieldTable.ENUM_CODE).map(String::valueOf).orElse(""));
            fieldItem.setMaxLength(getRowValue(row, FieldTable.MAX_LENGTH).map(String::valueOf).orElse(""));
            fieldItem.setName(getRowValue(row, FieldTable.NAME).map(String::valueOf).orElse(""));
            fieldItem.setRequired(getRowValue(row, FieldTable.REQUIRED).map(String::valueOf).orElse(""));
            fieldItem.setType(getRowValue(row, FieldTable.FIELD_TYPE).map(String::valueOf).orElse(""));
            fieldItem.setSearchable(getRowValue(row, FieldTable.SEARCHABLE).map(String::valueOf).orElse(""));
            fieldItem.setDictId(getRowValue(row, FieldTable.DICT_ID).map(String::valueOf).orElse(""));
            fieldItem.setDefaultValue(getRowValue(row, FieldTable.DEFAULT_VALUE).map(String::valueOf).orElse(""));
            fieldItem.setPrecision(getRowValue(row, FieldTable.PRECISION).map(String::valueOf).orElse(""));
            fieldItem.setUniqueName(getRowValue(row, FieldTable.UNIQUE_NAME).map(String::valueOf).orElse(""));
            //TODO
            fieldItem.setRelationshipEntity(null);
            items.add(fieldItem);
        }

        return items;
    }

    private <T> T read(Supplier<T> supplier) {
        rwLock.readLock().lock();
        try {
            return supplier.get();
        } finally {
            rwLock.readLock().unlock();
        }
    }

    private void write(Supplier<Void> supplier) {
        rwLock.writeLock().lock();
        try {
            supplier.get();
        } finally {
            rwLock.writeLock().unlock();
        }
    }

    @Override
    public BoItem getBoDetailById(String id, String profile) {

        return read(() -> {

            Tuple2<String, UpdateableDataContext> dcTuple = versionService.getCurrentVersionDCForBoById(Long.parseLong(id));

            UpdateableDataContext dc = dcTuple._2;

            if (dc == null) {
                return null;
            }

            //TODO
            DataSet boDetails = dc.query().from(BoTable.TABLE_NAME)
                    .selectAll()
                    .where(BoTable.ID).eq(id)
                    .execute();

            if (boDetails.next()) {
                Row ds = boDetails.getRow();
                DataSet apis = dc.query()
                        .from(ApiTable.TABLE_NAME)
                        .selectAll().where(ApiTable.BO_ID)
                        .eq(id)
                        .and(ApiTable.PROFILE).in(profile, DEFAULT_PROFILE)
                        .execute();

                Map<String, ApiItem> apiItemMap = toApiItemMap(apis);

                DataSet fields = dc.query()
                        .from(FieldTable.TABLE_NAME)
                        .selectAll()
                        .where(FieldTable.BO_ID).eq(id)
                        .and(FieldTable.PROFILE).in(profile, DEFAULT_PROFILE)
                        .execute();

                List<FieldItem> fieldItemList = toFieldItemList(fields);

                //deal with rel
                DataSet rels = dc.query()
                        .from(RelationTable.TABLE_NAME)
                        .selectAll()
                        .where(RelationTable.BO_ID).eq(id)
                        .and(RelationTable.PROFILE).in(profile, DEFAULT_PROFILE)
                        .execute();

                List<Row> rows = rels.toRows();

                List<String> relIds = rows
                        .stream()
                        .map(x -> RowUtils.getRowValue(x, RelationTable.JOIN_BO_ID)
                                .map(String::valueOf).orElse(""))
                        .collect(Collectors.toList());

                List<FieldItem> relField = this.loadRelationField(rows, row -> {

                    String joinBoId = RowUtils.getRowValue(row, RelationTable.JOIN_BO_ID)
                            .map(String::valueOf)
                            .orElse("");

                    DataSet boDs = dc.query().from(BoTable.TABLE_NAME)
                            .selectAll()
                            .where(BoTable.ID).eq(joinBoId)
                            .execute();

                    if (boDs.next()) {

                        Row bo = boDs.getRow();
                        String boCode = RowUtils.getRowValue(bo, BoTable.CODE)
                                .map(String::valueOf)
                                .orElse("");

                        SoloItem soloItem = new SoloItem();
                        soloItem.setId(Long.valueOf(joinBoId));

                        return new FieldItem(
                                boCode.concat(".id")
                                , boCode.concat(".id")
                                , FieldType.LONG.getType()
                                , ""
                                , "false"
                                , "true"
                                , "false"
                                , null
                                , null
                                , "0"
                                , ""
                                , ""
                                , soloItem
                                , ""
                        );
                    }
                    return null;
                });

                List<FieldItem> fieldTotalItems = new LinkedList<>();
                fieldTotalItems.addAll(fieldItemList);
                fieldTotalItems.addAll(relField);

                BoItem boItem = new BoItem();
                boItem.setApi(apiItemMap);
                boItem.setFields(fieldTotalItems);
                boItem.setParentEntityId(
                        RowUtils.getRowValue(ds, BoTable.PARENT_ID)
                                .map(String::valueOf).orElse(""));
                boItem.setSubEntities(relIds);
                return boItem;
            }
            return null;
        });
    }

    /**
     * @param moduleUpResult
     * @param tenantId
     * @param appId
     */
    @Override
    public void save(ModuleUpResult moduleUpResult, String tenantId, String appId) {

        write(() -> {

            saveApp(moduleUpResult.getAppUp(), versionService.getGlobalDC());

            String version = moduleUpResult.getVersion();
            long moduleId = moduleUpResult.getId();


            log.info("------- Version {} Got For {}", version, moduleId);

            versionService.saveModule(moduleId, version
                    , moduleUpResult.getBoUpsList().stream().flatMap(x -> {
                        Stream<BoNode> single = Stream.of(new BoNode(x.getCode(), Long.parseLong(x.getId())));
                        Stream<BoNode> nestOne = x.getBoUpsList().stream().map(sub -> new BoNode(sub.getCode(), Long.parseLong(sub.getId())));
                        return Stream.concat(single, nestOne);
                    }).collect(Collectors.toList()));

            UpdateableDataContext versionedDCForModule = versionService.getVersionedDCForModule(moduleId, version);

            //clear all the
            clearCurrentModule(moduleId, versionedDCForModule);

            moduleUpResult.getBoUpsList().forEach(boUp -> {
                //insert bo
                insertBo(moduleId, boUp, versionedDCForModule);
                log.info("Insert Bo:{}", boUp.getId());
            });
            return null;
        });
    }

    /**
     * clear all module related resource in current dc
     */
    private void clearCurrentModule(long moduleId, UpdateableDataContext dc) {
        write(() -> {
            DataSet boIds = dc.query().from(getTable(BoTable.TABLE_NAME, dc))
                    .select(BoTable.ID)
                    .where(BoTable.MODULE_ID)
                    .eq(moduleId).execute();

            boIds.toRows().forEach(x -> {
                String boId = x.getValue(0).toString();
                log.info("Clear Bo:{}", boId);
                clearAllBoIdRelated(boId, moduleId, dc);
            });

            return null;
        });
    }

//    /**
//     * load sub simple entity
//     *
//     * @param boId
//     * @param dc
//     * @param ver
//     * @return
//     */
//    private List<IEntityClass> loadChildEntityClass(String boId, UpdateableDataContext dc, String ver) {
//        return read(() -> {
//            DataSet boDs = dc.query()
//                    .from(BoTable.TABLE_NAME)
//                    .selectAll().where(BoTable.PARENT_ID).eq(boId)
//                    .execute();
//            List<Row> rows = boDs.toRows();
//            rows.stream().map(row -> {
//                toSimpleEntityClass();
//            });
//            return Collections.emptyList();
//        });
//    }

    /**
     * no more parent
     * no relation
     * no relation fields
     *
     * @return
     */
    private Optional<IEntityClass> loadParentEntityClassWithRelation(String boId, String profile, UpdateableDataContext dc, String ver) {

        return read(() -> Optional.ofNullable(dc).flatMap(x -> {
            DataSet boDs = dc.query()
                    .from(BoTable.TABLE_NAME)
                    .selectAll().where(BoTable.ID).eq(boId)
                    .execute();
            if (boDs.next()) {
                Row row = boDs.getRow();

                String code = RowUtils.getRowValue(row, BoTable.CODE).map(String::valueOf).orElse("");

                //raw ids
                List<IEntityField> entityFields = loadFields(boId, profile, dc);

                //relation ids

                DataSet relDs = dc.query()
                        .from("rels")
                        .selectAll()
                        .where("boId")
                        .eq(boId)
                        .execute();

                List<Row> relsRows = relDs.toRows();

                List<Tuple2<Relation, IEntityClass>> relatedEntityClassList = relsRows.stream().map(relRow -> {
                    Optional<String> relatedBoIdOp = RowUtils.getRowValue(relRow, "joinBoId").map(String::valueOf);
                    return relatedBoIdOp.flatMap(relBo -> {
                        return loadRelationEntityClass(boId, relBo, relRow, code, dc, profile, ver);
                    });
                }).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());

                //deal Relation
                List<IEntityClass> entityClassList = new LinkedList<>();
                List<Relation> relationList = new LinkedList<>();

                List<IEntityField> allFields = new LinkedList<>();
                allFields.addAll(entityFields);

                relatedEntityClassList.forEach(tuple -> {
                    entityClassList.add(tuple._2());
                    relationList.add(tuple._1());
                });

                //append all rel fields to fields
                relationList.stream().filter(rel -> {
                    return FieldLikeRelationType.from(rel.getRelationType())
                            .map(FieldLikeRelationType::isOwnerSide)
                            .orElse(false);
                }).forEach(rel -> allFields.add(rel.getEntityField()));

                return Optional.of(new EntityClass(Long.valueOf(boId)
                        , VersionUtils.toVersionInt(ver)
                        , code
                        , Collections.emptyList()
                        , Collections.emptyList()
                        , null
                        , Collections.emptyList()
                        , allFields));
            }

            return Optional.empty();
        }));
    }

    private Optional<IEntityClass> loadParentEntityClassWithoutRelation(String boId, UpdateableDataContext dc, String profile, String ver) {
        return read(() -> Optional.ofNullable(dc).flatMap(x -> {
            DataSet boDs = dc.query()
                    .from(BoTable.TABLE_NAME)
                    .selectAll().where(BoTable.ID).eq(boId)
                    .execute();
            if (boDs.next()) {
                Row row = boDs.getRow();

                String code = RowUtils.getRowValue(row, BoTable.CODE).map(String::valueOf).orElse("");

                //raw ids
                List<IEntityField> entityFields = loadFields(boId, profile, dc);

                return Optional.of(new EntityClass(Long.valueOf(boId)
                        , VersionUtils.toVersionInt(ver)
                        , code, Collections.emptyList()
                        , Collections.emptyList()
                        , null
                        , Collections.emptyList()
                        , entityFields));
            }

            return Optional.empty();
        }));
    }

    /**
     * load related entity class
     *
     * @param boId
     * @return
     */
    private Optional<Tuple2<Relation, IEntityClass>> loadRelationEntityClass(String ownerId, String boId, Row relRow, String mainBoCode
            , UpdateableDataContext contextDC, String profile, String ver) {

        return read(() -> {
            String relationType = RowUtils.getRowValue(relRow, RelationTable.REL_TYPE)
                    .map(String::valueOf)
                    .orElse("");

            String name = RowUtils.getRowValue(relRow, RelationTable.REL_NAME)
                    .map(String::valueOf)
                    .orElse("");

            Long joinBoId = RowUtils.getRowValue(relRow, RelationTable.JOIN_BO_ID)
                    .map(String::valueOf)
                    .map(Long::valueOf)
                    .orElse(0L);

            Long relId = RowUtils.getRowValue(relRow, RelationTable.ID)
                    .map(String::valueOf)
                    .map(Long::valueOf)
                    .orElse(0L);

            return findOneById(BoTable.TABLE_NAME, boId, contextDC).map(row -> {
                Optional<IEntityClass> parentEntityClass = RowUtils
                        .getRowValue(row, BoTable.PARENT_ID)
                        .map(String::valueOf)
                        .flatMap(x -> this.loadParentEntityClassWithoutRelation(x, contextDC, profile, ver));

                String subCode = RowUtils.getRowValue(row, BoTable.CODE)
                        .map(String::valueOf).orElse("");

                List<IEntityField> listFields = new LinkedList<>();

                //assemble relation Field
                /**
                 *  used as dto
                 *   public Relation(Long id, String name, String entityClassName, String ownerClassName, String relationType) {
                 */
                Relation relation = new Relation(relId, name, joinBoId, subCode, mainBoCode, relationType);
                relation.setRelOwnerClassId(Long.parseLong(ownerId));

                FieldLikeRelationType.from(relationType).ifPresent(x -> {
                    IEntityField relField = x.getField(relation);
                    relation.setEntityField(relField);

                    if (!x.isOwnerSide()) {
                        listFields.add(relField);
                    }
                });

                listFields.addAll(loadFields(boId, profile, contextDC));
                //assemble entity class
                IEntityClass entityClass = new EntityClass(Long.valueOf(boId)
                        , VersionUtils.toVersionInt(ver)
                        , subCode
                        , Collections.emptyList()
                        , Collections.emptyList()
                        , parentEntityClass.orElse(null)
                        , Collections.emptyList()
                        , listFields);

                return Tuple.of(relation, entityClass);
            });
        });
    }

    @Override
    public Optional<IEntityClass> loadByCode(String tenantId, String appCode, String boCode, String profile) {

        return read(() -> {
            Tuple2<String, UpdateableDataContext> currentVersionDCForBoByCode = versionService.getCurrentVersionDCForBoByCode(boCode);
            if (currentVersionDCForBoByCode != null) {
                String appVersion = getCurrentModuleVersion();
                String boVersion = currentVersionDCForBoByCode._1();
                //if current version is not present the bo maybe cleared
                if (!boVersion.equals(appVersion)) {
                    return Optional.empty();
                }
            }

            UpdateableDataContext dc = Optional.ofNullable(currentVersionDCForBoByCode).map(Tuple2::_2).orElse(null);
            return Optional.ofNullable(dc).flatMap(contextDC ->
                    this.loadByCode(tenantId, appCode, boCode, contextDC, profile, currentVersionDCForBoByCode._1()));
        });
    }

    @Override
    public List<IEntityClass> findSubEntitiesById(String tenantId, String appId, String parentId, String profile) {
        return read(() -> {
            String appVersion = getCurrentModuleVersion();
            return Optional.ofNullable(versionService.getCurrentVersionDCForBoById(Long.parseLong(parentId)))
                    .filter(x -> x._1().equals(appVersion))
                    .map(x -> this.findSubEntitiesById(tenantId, appId, parentId, x._2(), profile, x._1()))
                    .orElseGet(Collections::emptyList);
        });
    }

    @Override
    public List<IEntityClass> findSubEntitiesById(String tenantId, String appId, String parentId, String profile, String version) {
        return read(() -> {
            return Optional.ofNullable(versionService.getVersionedDCForBoById(Long.parseLong(parentId), version))
                    .map(x -> this.findSubEntitiesById(tenantId, appId, parentId, x, profile, version))
                    .orElseGet(Collections::emptyList);
        });
    }

    private List<IEntityClass> findSubEntitiesById(String tenantId, String appId, String parentId, UpdateableDataContext contextDC, String profile, String ver) {
        return read(() -> {
            DataSet boDs = contextDC.query()
                    .from(BoTable.TABLE_NAME)
                    .selectAll()
                    .where(BoTable.PARENT_ID)
                    .eq(parentId)
                    .execute();

            List<Row> rows = boDs.toRows();

            return rows.stream().map(row -> toEntityClass(row, contextDC, profile, ver))
                    .filter(Optional::isPresent)
                    .map(Optional::get).collect(Collectors.toList());
        });
    }

    private List<IEntityClass> findSubEntitiesByCode(String tenantId, String appId, String parentCode, UpdateableDataContext contextDC, String profile) {

        return read(() -> {
            DataSet boDs = contextDC.query()
                    .from(BoTable.TABLE_NAME)
                    .selectAll()
                    .where(BoTable.CODE)
                    .eq(parentCode)
                    .execute();

            if (boDs.next()) {
                String id = RowUtils
                        .getRowValue(boDs.getRow(), BoTable.ID)
                        .map(String::valueOf).orElse("");
                return findSubEntitiesById(tenantId, appId, id, profile);
            }

            return Collections.emptyList();
        });
    }

    @Override
    public List<IEntityClass> findSubEntitiesByCode(String tenantId, String appId, String parentCode, String profile) {

        return read(() -> {
            Tuple2<String, UpdateableDataContext> currentVersionDCForBoByCode = versionService.getCurrentVersionDCForBoByCode(parentCode);

            if (currentVersionDCForBoByCode != null) {
                String appVersion = getCurrentModuleVersion();
                String boVersion = currentVersionDCForBoByCode._1;
                if (!appVersion.equals(boVersion)) {
                    return Collections.emptyList();
                }
            } else {
                return Collections.emptyList();
            }

            UpdateableDataContext contextDC = currentVersionDCForBoByCode._2();
            return Optional.ofNullable(contextDC)
                    .map(x -> findSubEntitiesByCode(tenantId, appId, parentCode, x, profile))
                    .orElseGet(Collections::emptyList);
        });
    }

    @Override
    public List<IEntityClass> findSubEntitiesByCode(String tenantId, String appId, String parentCode, String profile, String version) {

        return read(() -> {
            UpdateableDataContext contextDC = versionService.getVersionedDCForBoByCode(parentCode, version);
            return Optional.ofNullable(contextDC)
                    .map(x -> findSubEntitiesByCode(tenantId, appId, parentCode, x, profile))
                    .orElseGet(Collections::emptyList);
        });
    }

    /**
     * load an entity class
     *
     * @param tenantId
     * @param appCode
     * @param boId
     * @return
     */
    @Override
    public Optional<IEntityClass> load(String tenantId, String appCode, String boId, String profile) {

        return read(() -> {
            Tuple2<String, UpdateableDataContext> currentVersionDCForBoById = versionService.getCurrentVersionDCForBoById(Long.parseLong(boId));
            if (currentVersionDCForBoById != null) {
                String appVersion = getCurrentModuleVersion();
                String boVersion = currentVersionDCForBoById._1;
                if (!boVersion.equals(appVersion)) {
                    return Optional.empty();
                }
            }
            UpdateableDataContext dc = Optional.ofNullable(currentVersionDCForBoById).map(Tuple2::_2).orElse(null);
            return Optional.ofNullable(dc).flatMap(x -> this.load(tenantId, appCode, boId, x, profile, currentVersionDCForBoById._1()));
        });
    }

    @Override
    public Optional<IEntityClass> load(String tenantId, String appCode, String boId, String profile, String version) {
        return read(() -> {
            UpdateableDataContext contextDc = versionService.getVersionedDCForBoById(Long.parseLong(boId), version);
            return Optional.ofNullable(contextDc).flatMap(x -> load(tenantId, appCode, boId, x, profile, version));
        });
    }

    @Override
    public Optional<IEntityClass> loadByCode(String tenantId, String appCode, String boCode, String profile, String version) {

        return read(() -> {
            UpdateableDataContext contextDc = versionService.getVersionedDCForBoByCode(boCode, version);
            return Optional.ofNullable(contextDc).flatMap(x -> loadByCode(tenantId, appCode, boCode, x, profile, version));
        });
    }


    /**
     * load by version
     *
     * @param tenantId
     * @param appCode
     * @param boCode
     * @param contextDC
     * @return
     */
    public Optional<IEntityClass> loadByCode(String tenantId, String appCode, String boCode, UpdateableDataContext contextDC, String profile, String ver) {
        return read(() -> {
            DataSet boDs = contextDC.query()
                    .from(BoTable.TABLE_NAME)
                    .selectAll()
                    .where(BoTable.CODE).eq(boCode)
                    .execute();

            if (boDs.next()) {
                Optional<IEntityClass> iEntityClass = toEntityClass(boDs.getRow(), contextDC, profile, ver);
                return iEntityClass.map(x -> {
                    List<IEntityClass> simpleSubEntities = findSimpleSubEntities(x, tenantId, appCode, Long.toString(x.id()), contextDC, profile, ver);
                    if (simpleSubEntities.isEmpty()) {
                        log.debug("[Meta]Current entity {}:{} has no child", x.code(), x.id());
                    }
                    x.resetChildEntityClass(simpleSubEntities);
                    return x;
                });
            } else {
                return Optional.empty();
            }
        });
    }


    private Optional<IEntityClass> load(String tenantId, String appCode, String boId, UpdateableDataContext contextDC, String profile, String ver) {

        log.debug("load class {} with contextDC {}", boId, contextDC);
        return read(() -> {
            DataSet boDs = contextDC.query()
                    .from(BoTable.TABLE_NAME)
                    .selectAll().where(BoTable.ID).eq(boId)
                    .execute();
            if (boDs.next()) {
                Optional<IEntityClass> iEntityClass = toEntityClass(boDs.getRow(), contextDC, profile, ver);
                return iEntityClass.map(x -> {
                    List<IEntityClass> simpleSubEntities = findSimpleSubEntities(x, tenantId, appCode, boId, contextDC, profile, ver);
                    if (simpleSubEntities.isEmpty()) {
                        log.debug("[Meta]Current entity {}:{} has no child", x.code(), x.id());
                    }
                    x.resetChildEntityClass(simpleSubEntities);
                    return x;
                });
            } else {
                return Optional.empty();
            }
        });
    }

    public List<IEntityClass> findSimpleSubEntities(IEntityClass iEntityClass, String tenantId, String appId
            , String parentId, UpdateableDataContext contextDC, String profile, String ver) {
        return read(() -> {
            DataSet boDs = contextDC.query()
                    .from(BoTable.TABLE_NAME)
                    .selectAll()
                    .where(BoTable.PARENT_ID)
                    .eq(parentId)
                    .execute();

            List<Row> rows = boDs.toRows();

            return rows.stream().map(row -> toSimpleEntityClass(row, iEntityClass, contextDC, profile, ver))
                    .collect(Collectors.toList());
        });
    }

    private IEntityClass toSimpleEntityClass(Row row, IEntityClass parent, UpdateableDataContext contextDC, String profile, String ver) {
        return read(() -> {
            String code = RowUtils.getRowValue(row, "code").map(String::valueOf).orElse("");
            String boId = RowUtils.getRowValue(row, "id").map(String::valueOf).orElse("0");
            String name = RowUtils.getRowValue(row, "name").map(String::valueOf).orElse("");

            List<IEntityField> fields = loadFields(boId, profile, contextDC);

            EntityClass entityClass = new EntityClass(Long.valueOf(boId)
                    , VersionUtils.toVersionInt(ver)
                    , code, name, Collections.emptyList(), Collections.emptyList()
                    , parent, fields);
            return entityClass;
        });
    }

    /**
     * how to build a entity class
     * check if has parent --> load parent info
     * |-> deal parent fields
     * find relation --> build relation entity
     * |-> build relation fields
     * deal with self fields
     *
     * @param row
     * @return
     */
    private Optional<IEntityClass> toEntityClass(Row row, UpdateableDataContext contextDC, String profile, String ver) {

        return read(() -> {

            String code = RowUtils.getRowValue(row, "code").map(String::valueOf).orElse("");
            String boId = RowUtils.getRowValue(row, "id").map(String::valueOf).orElse("0");
            String name = RowUtils.getRowValue(row, "name").map(String::valueOf).orElse("");
            String type = RowUtils.getRowValue(row, "type").map(String::valueOf).orElse("0");
            List<IEntityField> fields = loadFields(boId, profile, contextDC);

            //build up parentClass
            String parentId = RowUtils.getRowValue(row, "parentId").map(String::valueOf).orElse("");

            Optional<IEntityClass> parentEntityClassOp = loadParentEntityClassWithRelation(parentId, profile, contextDC, ver);

            List<Row> relationRows = new LinkedList<>();

            //deal relation Classes
            DataSet relDs = contextDC.query()
                    .from("rels")
                    .selectAll()
                    .where("boId")
                    .eq(boId)
                    .execute();

            relationRows.addAll(relDs.toRows());

            List<Row> relsRows = relationRows;

            List<Tuple2<Relation, IEntityClass>> relatedEntityClassList = relsRows.stream().map(relRow -> {
                Optional<String> relatedBoIdOp = RowUtils.getRowValue(relRow, "joinBoId").map(String::valueOf);
                return relatedBoIdOp.flatMap(x -> {
                    return loadRelationEntityClass(boId, x, relRow, code, contextDC, profile, ver);
                });
            }).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());

            //deal Relation
            List<IEntityClass> entityClassList = new LinkedList<>();
            List<IRelation> relationList = new LinkedList<>();

            List<IEntityField> allFields = new LinkedList<>();
            allFields.addAll(fields);

            relatedEntityClassList.forEach(tuple -> {
                entityClassList.add(tuple._2());
                relationList.add(tuple._1());
            });

            //append all rel fields to fields
            relationList.stream().filter(x -> {
                return FieldLikeRelationType.from(x.getRelationType())
                        .map(FieldLikeRelationType::isOwnerSide)
                        .orElse(false);
            }).forEach(x -> allFields.add(x.getEntityField()));

            /**
             *           EntityClass(long id,
             *                        int ver,
             *                        String code,
             *                        Collection<Relation> relations,
             *                        Collection<IEntityClass> entityClasses,
             *                        IEntityClass extendEntityClass,
             *                        Collection<IEntityField> fields) {
             */
            EntityClass entityClass = new EntityClass(Long.valueOf(boId)
                    , VersionUtils.toVersionInt(ver)
                    , code, name, relationList, entityClassList
                    , parentEntityClassOp.orElse(null), allFields);

            /**
             * set bo type
             */
            if (!StringUtils.isEmpty(type) && "external".equals(type)) {
                //
                entityClass.setType(1);
            } else {
                entityClass.setType(0);
            }

            return Optional.of(entityClass);
        });
    }

    private <U> List<U> loadRelationField(List<Row> relations, Function<Row, U> mapper) {
        return relations.stream().filter(row -> {
                    return RowUtils.getRowValue(row, RelationTable.REL_TYPE)
                            .map(String::valueOf)
                            .filter(type -> type.equalsIgnoreCase("onetoone")
                                    || type.equalsIgnoreCase("manytoone"))
                            .isPresent();
                }).map(mapper).filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    private Table getTable(String tableName, UpdateableDataContext contextDC) {
        return read(() -> {
            return contextDC.getTableByQualifiedLabel("metadata." + tableName);
        });
    }

    private Optional<Row> findOneById(String tableName, String id, UpdateableDataContext dc) {

        return read(() -> {
            DataSet ds = dc.query().from(tableName)
                    .selectAll()
                    .where("id").eq(id)
                    .execute();

            if (ds.next()) {
                return Optional.ofNullable(ds.getRow());
            }
            return Optional.empty();
        });
    }

    /**
     * nothing todo with the related entity
     *
     * @param boId
     */
    @Override
    public void clearAllBoIdRelated(String boId, Long moduleId, UpdateableDataContext dc) {
        write(() -> {
            UpdateSummary updateSummary = dc.executeUpdate(callback -> {
                callback.deleteFrom(getTable(BoTable.TABLE_NAME, dc)).where(BoTable.ID).eq(boId).execute();
                callback.deleteFrom(getTable(ApiTable.TABLE_NAME, dc)).where(ApiTable.BO_ID).eq(boId).execute();
                callback.deleteFrom(getTable(FieldTable.TABLE_NAME, dc)).where(FieldTable.BO_ID).eq(boId).execute();
                callback.deleteFrom(getTable(RelationTable.TABLE_NAME, dc)).where(RelationTable.BO_ID).eq(boId).execute();
            });

            return null;
        });
    }

    @Override
    public SimpleBoItem findOneById(String boId) {
        return read(() -> {
            Tuple2<String, UpdateableDataContext> currentVersionDCForBoById = versionService.getCurrentVersionDCForBoById(Long.parseLong(boId));
            if (currentVersionDCForBoById != null) {
                String appVersion = getCurrentModuleVersion();
                String boVersion = currentVersionDCForBoById._1;
                if (!boVersion.equals(appVersion)) {
                    return null;
                }
            }

            UpdateableDataContext dc = Optional.ofNullable(currentVersionDCForBoById).map(Tuple2::_2).orElse(null);
            return Optional.ofNullable(dc).map(x -> this.findOneById(boId, dc)).orElse(null);
        });
    }

    @Override
    public SimpleBoItem findOneById(String boId, String version) {
        return read(() -> {
            UpdateableDataContext dc = versionService.getVersionedDCForBoById(Long.parseLong(boId), version);
            return Optional.ofNullable(dc).map(x -> this.findOneById(boId, dc)).orElse(null);
        });
    }

    public SimpleBoItem findOneById(String boId, UpdateableDataContext contextDC) {

        return read(() -> {
            DataSet boDs = contextDC.query()
                    .from(BoTable.TABLE_NAME)
                    .selectAll().where(BoTable.ID).eq(boId)
                    .execute();
            if (boDs.next()) {
                SimpleBoItem simpleBoItem = new SimpleBoItem();
                Row row = boDs.getRow();
                simpleBoItem.setCode(RowUtils.getRowValue(row, BoTable.CODE).map(String::valueOf).orElse(""));
                simpleBoItem.setParentId(RowUtils.getRowValue(row, BoTable.PARENT_ID).map(String::valueOf).orElse(""));
                simpleBoItem.setId(boId);
                simpleBoItem.setCname(RowUtils.getRowValue(row, BoTable.NAME).map(String::valueOf).orElse(""));
                return simpleBoItem;
            } else {
                return null;
            }
        });
    }

    //TODO
    @Override
    public List<IEntityClass> findAllEntities(String profile) {
        String appVersion = getCurrentModuleVersion();
        return read(() -> {
            return versionService.getBoModuleMapping().entrySet().stream().map(x -> {
                        Long boId = x.getKey().getId();
                        LinkedList<Tuple2<Long, String>> value = x.getValue();
                        String version = value.getLast()._2();
                        if (version.equals(appVersion)) {
                            UpdateableDataContext versionedDCForBoId = versionService.getVersionedDCForBoById(boId, version);
                            log.debug("CurrentContext is {}", versionedDCForBoId);
                            return Optional.ofNullable(versionedDCForBoId).flatMap(dc -> load("", "", String.valueOf(boId), dc, profile, version));
                        } else {
                            return Optional.<IEntityClass>empty();
                        }
                    }).filter(Optional::isPresent).map(Optional::get)
                    .collect(Collectors.toList());
        });
    }

//    /**
//     * get all with profile
//     * @return
//     */
//    public List<IEntityClass> findAllEntities() {
//
//        return read(() -> {
//            return versionService.getBoModuleMapping().entrySet().stream().map(x -> {
//                Long boId = x.getKey().getId();
//                LinkedList<Tuple2<Long, String>> value = x.getValue();
//                String version = value.getLast()._2();
//
//                UpdateableDataContext versionedDCForBoId = versionService.getVersionedDCForBoById(boId, version);
//                log.debug("CurrentContext is {}", versionedDCForBoId);
//                return Optional.ofNullable(versionedDCForBoId).flatMap(dc -> load("", "", String.valueOf(boId), dc, profile, version));
//            }).filter(Optional::isPresent).map(Optional::get)
//                    .collect(Collectors.toList());
//        });
//    }


    private List<IEntityClass> findAllEntities(UpdateableDataContext contextDC, String profile, String ver) {

        return read(() -> {

            DataSet boDs = contextDC.query()
                    .from(BoTable.TABLE_NAME)
                    .selectAll()
                    .execute();

            List<Row> rows = boDs.toRows();

            return rows.stream().map(x -> this.toEntityClass(x, contextDC, profile, ver))
                    .filter(Optional::isPresent)
                    .map(Optional::get).collect(Collectors.toList());
        });
    }

    //TODO
    @Override
    public CurrentVersion currentVersion() {
        return read(() -> {
            return new CurrentVersion(versionService.getCurrentVersion());
        });
    }

    private void insertBoTable(String id, String moduleId, String code, String parentId, String name
            , String rootFlag, String sysType, String domainCode, String domainName, long domainRootId
            , String type
            , UpdateableDataContext contextDC) {
        write(() -> {

            InsertInto insert = new InsertInto(getTable(BoTable.TABLE_NAME, contextDC))
                    .value(BoTable.ID, id)
                    .value(BoTable.CODE, code)
                    .value(BoTable.PARENT_ID, parentId)
                    .value(BoTable.NAME, name)
                    .value(BoTable.MODULE_ID, moduleId)
                    .value(BoTable.ROOT_FLAG, rootFlag)
                    .value(BoTable.SYS_TYPE, sysType)
                    .value(BoTable.DOMAIN_CODE, domainCode)
                    .value(BoTable.DOMAIN_NAME, domainName)
                    .value(BoTable.DOMAIN_ROOT_ID, domainRootId)
                    .value(BoTable.TYPE, type);

            contextDC.executeUpdate(insert);
            return null;
        });
    }

    /**
     * app has no other
     *
     * @param appUp
     * @param contextDC
     */
    private void saveApp(AppUp appUp, UpdateableDataContext contextDC) {
        write(() -> {

            DeleteFrom deleteFrom = new DeleteFrom(getTable(AppTable.TABLE_NAME, contextDC))
                    .where(AppTable.APP_ID).eq(appUp.getId());
            contextDC.executeUpdate(deleteFrom);

            InsertInto insert = new InsertInto(getTable(AppTable.TABLE_NAME, contextDC))
                    .value(AppTable.APP_ID, appUp.getId())
                    .value(AppTable.APP_CODE, appUp.getCode())
                    .value(AppTable.APP_NAME, appUp.getName())
                    .value(AppTable.APP_BRANCH, appUp.getBranch())
                    .value(AppTable.APP_VERSION, appUp.getVersion());
            contextDC.executeUpdate(insert);
            return null;
        });
    }

    /**
     * bo insert
     *
     * @param boUp
     */
    private void insertBo(long moduleId, BoUp boUp, UpdateableDataContext contextDC) {

        write(() -> {

            insertBoTable(boUp.getId()
                    , String.valueOf(moduleId)
                    , boUp.getCode()
                    , boUp.getParentBoId()
                    , boUp.getName()
                    , boUp.getRootFlag()
                    , boUp.getSysType()
                    , boUp.getDomainCode()
                    , boUp.getDomainName()
                    , boUp.getDomainRootId()
                    , boUp.getBoType()
                    , contextDC);

            //save relations
            boUp.getRelationsList().forEach(rel -> {
                InsertInto insertRel = new InsertInto(
                        getTable(RelationTable.TABLE_NAME, contextDC))
                        .value(RelationTable.ID, rel.getId())
                        .value(RelationTable.BO_ID, rel.getBoId())
                        .value(RelationTable.JOIN_BO_ID, rel.getJoinBoId())
                        .value(RelationTable.IDENTITY, rel.getIdentity())
                        .value(RelationTable.REL_TYPE, rel.getRelationType())
                        .value(RelationTable.REL_NAME, rel.getRelName())
                        .value(RelationTable.BO_FIELD, rel.getBoField())
                        .value(RelationTable.JOIN_FIELD, rel.getJoinField())
                        .value(RelationTable.PROFILE, rel.getProfile());
                contextDC.executeUpdate(insertRel);
            });

            //insert apis
            boUp.getApisList().forEach(api -> {
                insertApi(api, boUp.getId(), contextDC);
            });

            //insert fields
            boUp.getFieldsList().forEach(field -> {
                insertField(field, boUp.getId(), contextDC);
            });

            //maybe not need
            //insert sub bo
            //save if not exist
            boUp.getBoUpsList().stream()
                    .filter(relatedBo -> !findOneById(BoTable.TABLE_NAME, relatedBo.getId(), contextDC).isPresent())
                    .forEach(relatedBo -> {
                        insertBoTable(relatedBo.getId()
                                , String.valueOf(moduleId)
                                , relatedBo.getCode()
                                , relatedBo.getParentBoId()
                                , relatedBo.getName()
                                , relatedBo.getRootFlag()
                                , relatedBo.getSysType()
                                , relatedBo.getDomainCode()
                                , relatedBo.getDomainName()
                                , relatedBo.getDomainRootId()
                                , relatedBo.getBoType()
                                , contextDC);

                        //save fields
                        //insert apis
                        relatedBo.getApisList().forEach(api -> {
                            insertApi(api, relatedBo.getId(), contextDC);
                        });

                        //insert fields
                        relatedBo.getFieldsList().forEach(field -> {
                            insertField(field, relatedBo.getId(), contextDC);
                        });
                    });
            return null;
        });
    }

    private String getBooleanString(String input) {
        if ("1".equals(input) || "true".equals(input)) {
            return "true";
        } else {
            return "false";
        }
    }

    private void insertField(Field field, String boId, UpdateableDataContext contextDC) {
        write(() -> {

            String editable = getBooleanString(field.getEditable());
            String searchable = getBooleanString(field.getSearchable());
            String identifier = getBooleanString(field.getIdentifier());
            String required = getBooleanString(field.getRequired());

            boolean hasFormula = field.hasCalculator();

            //TODO replace with table
            InsertInto insert = new InsertInto(getTable(FieldTable.TABLE_NAME, contextDC))
                    .value(FieldTable.BO_ID, boId)
                    .value(FieldTable.ID, field.getId())
                    .value(FieldTable.CODE, field.getCode())
                    .value(FieldTable.DISPLAY_TYPE, field.getDisplayType())
                    .value(FieldTable.EDITABLE, editable)
                    .value(FieldTable.ENUM_CODE, field.getEnumCode())
                    .value(FieldTable.MAX_LENGTH, field.getMaxLength())
                    .value(FieldTable.NAME, field.getName())
                    .value(FieldTable.REQUIRED, required)
                    .value(FieldTable.FIELD_TYPE, field.getFieldType())
                    .value(FieldTable.SEARCHABLE, searchable)
                    .value(FieldTable.DICT_ID, field.getDictId())
                    .value(FieldTable.DEFAULT_VALUE, field.getDefaultValue())
                    .value(FieldTable.PRECISION, String.valueOf(field.getPrecision()))
                    .value(FieldTable.IDENTIFIER, identifier)
                    .value(FieldTable.VALIDATE_RULE, field.getValidateRule())
                    .value(FieldTable.UNIQUE_NAME, field.getUniqueName())
                    .value(FieldTable.INDEX_FLAG, field.getIndexFlag())
                    .value(FieldTable.DIMENSION_FlAG, field.getDimensionFlag())
                    .value(FieldTable.PROFILE, field.getProfile())
                    .value(FieldTable.CALCULATE_TYPE, field.getCalculator().getCalculateType())
                    .value(FieldTable.FORMULA, hasFormula ? field.getCalculator().getExpression() : "")
                    .value(FieldTable.FORMULA_ARGS, hasFormula ? field.getCalculator().getArgumentsList().stream().collect(Collectors.joining(",")) : "");
            contextDC.executeUpdate(insert);
            return null;
        });
    }

    private void insertApi(Api api, String boId, UpdateableDataContext dc) {
        write(() -> {

            Map<String, Object> config = new HashMap<>();

            /**
             * gen rules
             */
            List<Map<String, Object>> rules = api.getDetailsList().stream().map(x -> {
                Map<String, Object> rule = new HashMap<>();
                rule.put("expression", x.getRule());
                rule.put("type", x.getType());
                return rule;
            }).collect(Collectors.toList());

            config.put("rules", rules);

            String configJson = "{}";
            try {
                configJson = mapper.writeValueAsString(config);
            } catch (Exception ex) {
                log.error("{}", ex);
            }

            InsertInto insert = new InsertInto(getTable(ApiTable.TABLE_NAME, dc))
                    .value(ApiTable.BO_ID, boId)
                    .value(ApiTable.URL, api.getUrl())
                    .value(ApiTable.EXTERNAL_URL, api.getExternalUrl())
                    .value(ApiTable.CODE, api.getCode())
                    .value(ApiTable.METHOD, api.getMethod())
                    .value(ApiTable.PARAM, api.getParam())
                    .value(ApiTable.RESPONSE_DATA, api.getResponseData())
                    //fixed header
                    .value(ApiTable.RESPONSE_HEADER, api.getRequestHeader())
                    .value(ApiTable.AUTH_CODE, api.getAuthCode())
                    .value(ApiTable.CONFIG, configJson)
                    .value(ApiTable.PROFILE, api.getProfile());
            dc.executeUpdate(insert);
            return null;
        });
    }

//------------------------------------------------------------------------------------------------------------

    private List<IEntityField> loadFields(String id, String profile, UpdateableDataContext dc) {
        return read(() -> {

            DataSet fieldDs = dc.query().from(FieldTable.TABLE_NAME)
                    .selectAll()
                    .where(FieldTable.BO_ID).eq(id)
                    .and(FieldTable.PROFILE).in(profile, "")
                    .execute();
            return fieldDs.toRows().stream()
                    .map(FieldHelper::toEntityClassField)
                    .collect(Collectors.toList());
        });
    }

    //TODO if can rename
    @Override
    public List<BoInfoVo> loadSonBoInfoVo(String code, String profile) {
        return read(() -> {
            Tuple2<String, UpdateableDataContext> currentVersionDCForBoByCode = versionService.getCurrentVersionDCForBoByCode(code);
            UpdateableDataContext dc = Optional.ofNullable(currentVersionDCForBoByCode).map(Tuple2::_2).orElse(null);
            return loadSonBoInfoByCode(code, dc);
        });
    }

    //TODO if filter bo
    @Override
    public List<BoInfoVo> getAllBoInfo(String profile) {
        return read(() -> {
            return versionService.getBoModuleMapping().entrySet().stream().map(x -> {
                        Long boId = x.getKey().getId();
                        LinkedList<Tuple2<Long, String>> value = x.getValue();
                        String version = value.getLast()._2();

                        UpdateableDataContext versionedDCForBoId = versionService.getVersionedDCForBoById(boId, version);
                        log.debug("CurrentContext is {}", versionedDCForBoId);
                        return Optional
                                .ofNullable(versionedDCForBoId)
                                .flatMap(dc -> loadBoInfo(String.valueOf(boId), dc));
                    }).filter(Optional::isPresent).map(Optional::get)
                    .collect(Collectors.toList());
        });
    }

    @Override
    public BoInfoVo getBoInfo(String code, String profile) {
        return read(() -> {
            Tuple2<String, UpdateableDataContext> currentVersionDCForBoByCode = versionService.getCurrentVersionDCForBoByCode(code);
            UpdateableDataContext dc = Optional.ofNullable(currentVersionDCForBoByCode).map(Tuple2::_2).orElse(null);
            return this.loadBoInfoByCode(code, dc).orElse(null);
        });
    }

    /**
     * current module is only one
     *
     * @return
     */
    private String getCurrentModuleVersion() {
        Map<Long, String> currentVersion = versionService.getCurrentVersion();
        Collection<String> values = currentVersion.values();
        if (!values.isEmpty()) {
            return values.stream().findFirst().orElse(null);
        } else {
            return null;
        }
    }

    @Override
    public AppItem getCurrentApp() {
        DataSet rows = versionService.getGlobalDC().query().from(AppTable.TABLE_NAME)
                .selectAll().limit(1).execute();

        AppItem appItem = null;

        List<Row> resultRow = rows.toRows();
        if (!resultRow.isEmpty()) {
            Row row = resultRow.get(0);
            appItem = new AppItem();
            String appCode = RowUtils.getRowValue(row, AppTable.APP_CODE).map(String::valueOf).orElse("");
            String appVersion = RowUtils.getRowValue(row, AppTable.APP_VERSION).map(String::valueOf).orElse("");
            String appBranch = RowUtils.getRowValue(row, AppTable.APP_BRANCH).map(String::valueOf).orElse("");
            appItem.setAppCode(appCode);
            appItem.setAppVersion(appVersion);
            appItem.setAppBranch(appBranch);
            return appItem;
        }

        return null;
    }

    /**
     * load self and related
     * TODO
     *
     * @param code
     * @return
     */
    @Override
    public List<BoInfoVo> loadRelatedBoInfoVo(String code, String profile) {
        return read(() -> {
            Tuple2<String, UpdateableDataContext> currentVersionDCForBoByCode = versionService.getCurrentVersionDCForBoByCode(code);
            UpdateableDataContext dc = Optional.ofNullable(currentVersionDCForBoByCode).map(Tuple2::_2).orElse(null);
            Set<Long> circleDetected = new HashSet<>();
            List<BoInfoVo> relatedBoList = new LinkedList<>();
            this.loadRelatedBoInfoVoByCode(code, dc, circleDetected, relatedBoList);
            return relatedBoList;
        });
    }

    @Override
    public ApiDetails loadApiByCodeAndEntityClass(String actionCode, long boId) {
        return read(() -> {
            Tuple2<String, UpdateableDataContext> currentVersionDCForBoById = versionService.getCurrentVersionDCForBoById(boId);
            UpdateableDataContext dc = Optional.ofNullable(currentVersionDCForBoById).map(Tuple2::_2).orElse(null);


            DataSet dataSet = dc.query()
                    .from(ApiTable.TABLE_NAME).selectAll()
                    .where(ApiTable.BO_ID).eq(boId).and(ApiTable.CODE).eq(actionCode)
                    .execute();

            List<Row> rows = dataSet.toRows();

            if (!rows.isEmpty()) {
                Row row = rows.get(0);
                //build an api

                /*
                  config json value with rule
                 */
                Optional<String> configJson = getRowValue(row, ApiTable.CONFIG)
                        .map(Object::toString);

                Optional<String> fixedHeaderJson = getRowValue(row, ApiTable.RESPONSE_HEADER)
                        .map(Object::toString);

                /*
                 * authCode
                 */
                Optional<String> authCode = getRowValue(row, ApiTable.AUTH_CODE)
                        .map(Object::toString);

                Optional<String> method = getRowValue(row, ApiTable.METHOD)
                        .map(Object::toString);

                Optional<String> externalUrl = getRowValue(row, ApiTable.EXTERNAL_URL)
                        .map(Object::toString);

                Optional<String> url = getRowValue(row, ApiTable.URL)
                        .map(Object::toString);

                url.orElseThrow(() -> new RuntimeException("URL is required " + actionCode));

                String value = configJson.orElse("{}");
                Map<String, Object> configMap = Collections.emptyMap();
                try {
                    configMap = mapper.readValue(value
                            , new TypeReference<Map<String, Object>>() {
                            });
                } catch (Exception ex) {

                }

                String fixedHeader = fixedHeaderJson.orElse("[]");
                List<Map<String, Object>> headerMap = Collections.emptyList();
                try {
                    headerMap = mapper.readValue(fixedHeader
                            , new TypeReference<List<Map<String, Object>>>() {
                            });
                } catch (Exception ex) {
                    ex.printStackTrace();
                }

                ApiDetails apiDetails = new ApiDetails();
                apiDetails.setAuthCode(authCode.orElse(""));
                apiDetails.setCode(actionCode);
                apiDetails.setMethod(method.orElse("GET"));
                apiDetails.setExternalUrl(externalUrl.orElse(""));
                apiDetails.setUrl(url.orElse(""));
                apiDetails.setFixedHeaders(headerMap);

                Object rules = configMap.get("rules");
                if (rules != null) {
                    List<Map<String, Object>> typedRules = (List<Map<String, Object>>) rules;
                    List<ApiRule> apiRules = typedRules.stream().map(tr -> {
                        Object expression = tr.get("expression");
                        Object type = tr.get("type");
                        if (expression != null && type != null) {
                            ApiRule apiRule = new ApiRule();
                            apiRule.setType(type.toString());
                            apiRule.setExpression(expression.toString());
                            return apiRule;
                        }

                        return null;
                    }).filter(Objects::nonNull).collect(Collectors.toList());
                    apiDetails.setRules(apiRules);
                }

                return apiDetails;
            } else {
                return null;
            }
        });
    }

    @Override
    public Set<String> findCustomActionsById(Long id) {

        return read(() -> {
            Tuple2<String, UpdateableDataContext> currentVersionDCForBoByCode = versionService.getCurrentVersionDCForBoById(id);
            UpdateableDataContext dc = Optional.ofNullable(currentVersionDCForBoByCode).map(Tuple2::_2).orElse(null);
            DataSet row = dc.query().from(ApiTable.TABLE_NAME)
                    .select(ApiTable.CODE).where(ApiTable.BO_ID).eq(id).execute();

            return row.toRows().stream().map(x -> getRowValue(x, ApiTable.CODE))
                    .filter(Optional::isPresent)
                    .map(Optional::get)
                    .map(Object::toString)
                    .filter(x -> {
                        return Arrays.stream(preserved).noneMatch(y -> y.equals(x));
                    })
                    .collect(Collectors.toSet());
        });
    }

    private void loadNested(BoInfoVo mainBo, UpdateableDataContext dc, Set<Long> circleDetected, List<BoInfoVo> relatedBoList) {
        if (circleDetected.contains(mainBo.getId())) {
            return;
        }

        circleDetected.add(mainBo.getId());
        relatedBoList.add(mainBo);
        //construct all related
        List<RelationshipVo> relationshipVos = mainBo.getRelationshipVos();
        if (relationshipVos != null && !relationshipVos.isEmpty()) {
            Set<Long> relationCodes = relationshipVos
                    .stream()
                    .map(RelationshipVo::getBoId)
                    .collect(Collectors.toSet());

            /**
             * get all elements not in circleDetected
             */
            Sets.SetView<Long> intersection = Sets.intersection(circleDetected, relationCodes);

            for (Long boId : intersection) {
                //
                loadRelatedBoInfoVoById(boId, dc, circleDetected, relatedBoList);
            }
        }
    }

    private void loadRelatedBoInfoVoById(Long id, UpdateableDataContext dc
            , Set<Long> circleDetected
            , List<BoInfoVo> relatedBoList) {
        read(() -> {
            Optional<BoInfoVo> boInfoVo = loadBoInfo(id.toString(), dc);
            if (boInfoVo.isPresent()) {
                relatedBoList.add(boInfoVo.get());
                loadNested(boInfoVo.get(), dc, circleDetected, relatedBoList);
            }
            return null;
        });
    }

    private void loadRelatedBoInfoVoByCode(String code, UpdateableDataContext dc
            , Set<Long> circleDetected
            , List<BoInfoVo> relatedBoList) {
        read(() -> {
            Optional<BoInfoVo> boInfoVo = loadBoInfoByCode(code, dc);
            if (boInfoVo.isPresent()) {
                relatedBoList.add(boInfoVo.get());
                loadNested(boInfoVo.get(), dc, circleDetected, relatedBoList);
            }
            return null;
        });
    }

    private List<BoInfoVo> loadSonBoInfoByCode(String code, UpdateableDataContext contextDC) {
        return read(() -> {
            DataSet boDs = contextDC.query()
                    .from(BoTable.TABLE_NAME)
                    .selectAll().where(BoTable.CODE).eq(code)
                    .execute();
            List<Row> rows = boDs.toRows();
            if (!rows.isEmpty()) {
                Row row = rows.get(0);
                Optional<Object> rowValue = getRowValue(row, BoTable.ID);
                if (rowValue.isPresent()) {
                    String boId = rowValue.get().toString();
                    DataSet sonBoDs = contextDC.query()
                            .from(BoTable.TABLE_NAME)
                            .selectAll().where(BoTable.PARENT_ID).eq(boId)
                            .execute();
                    List<Row> sonRow = sonBoDs.toRows();
                    return sonRow.stream().map(x -> toBoInfo(x, contextDC))
                            .filter(Optional::isPresent)
                            .map(Optional::get)
                            .collect(Collectors.toList());
                }
            }

            return Collections.emptyList();
        });
    }

    private Optional<BoInfoVo> loadBoInfoByCode(String code, UpdateableDataContext contextDC) {
        return read(() -> {
            DataSet boDs = contextDC.query()
                    .from(BoTable.TABLE_NAME)
                    .selectAll().where(BoTable.CODE).eq(code)
                    .execute();
            if (boDs.next()) {
                return toBoInfo(boDs.getRow(), contextDC);
            } else {
                return Optional.empty();
            }
        });
    }

    private Optional<BoInfoVo> loadBoInfo(String boId, UpdateableDataContext contextDC) {
        return read(() -> {
            DataSet boDs = contextDC.query()
                    .from(BoTable.TABLE_NAME)
                    .selectAll().where(BoTable.ID).eq(boId)
                    .execute();
            if (boDs.next()) {
                return toBoInfo(boDs.getRow(), contextDC);
            } else {
                return Optional.empty();
            }
        });
    }

    private Optional<BoInfoVo> toBoInfo(Row row, UpdateableDataContext contextDC) {
        return read(() -> {

            String code = RowUtils.getRowValue(row, BoTable.CODE).map(String::valueOf).orElse("");
            String boId = RowUtils.getRowValue(row, BoTable.ID).map(String::valueOf).orElse("0");
            String name = RowUtils.getRowValue(row, BoTable.NAME).map(String::valueOf).orElse("");
            //build up parentClass
            String parentId = RowUtils.getRowValue(row, BoTable.PARENT_ID).map(String::valueOf).orElse("");

            /**
             *
             */
            List<BoFieldVo> parentFields = Collections.emptyList();
            List<RelationshipVo> parentRelationshipVos = Collections.emptyList();
            if (StringUtils.isNotEmpty(parentId)) {
                Optional<BoInfoVo> boInfoVo = loadBoInfo(parentId, contextDC);
                if (boInfoVo.isPresent()) {
                    BoInfoVo parentBoInfoVo = boInfoVo.get();
                    parentFields = parentBoInfoVo.getFields();
                    parentRelationshipVos = parentBoInfoVo.getRelationshipVos();
                }
            }

            String sysType = RowUtils.getRowValue(row, BoTable.SYS_TYPE).map(String::valueOf).orElse("");

            String domainName = RowUtils.getRowValue(row, BoTable.DOMAIN_NAME).map(String::valueOf).orElse("");
            String domainCode = RowUtils.getRowValue(row, BoTable.DOMAIN_CODE).map(String::valueOf).orElse("");
            String domainRootId = RowUtils.getRowValue(row, BoTable.DOMAIN_ROOT_ID).map(String::valueOf).orElse("");
            String rootFlag = RowUtils.getRowValue(row, BoTable.ROOT_FLAG).map(String::valueOf).orElse("");

            BoInfoVo boInfoVo = new BoInfoVo();

            boInfoVo.setApis(loadApis(boId, contextDC));

            List<Row> relationRows = new LinkedList<>();
            //deal relation Classes
            DataSet relDs = contextDC.query()
                    .from("rels")
                    .selectAll()
                    .where("boId")
                    .eq(boId)
                    .execute();

            relationRows.addAll(relDs.toRows());

            List<Row> relsRows = relationRows;

            List<RelationshipVo> relations = relsRows
                    .stream().map(x -> this.toRelationVo(boId, x))
                    .collect(Collectors.toList());

            //append relation field
            List<BoFieldVo> relatedFields = relations.stream().filter(x ->
                    x.getRelationType().equalsIgnoreCase(FieldLikeRelationType.MANY2ONE.getName())
                            || x.getRelationType().equalsIgnoreCase(FieldLikeRelationType.ONE2ONE.getName())
                            || x.getRelationType().equalsIgnoreCase(FieldLikeRelationType.MULTI_VALUES.getName())

            ).map(x -> {
                BoFieldVo boFieldVo = new BoFieldVo();
                boFieldVo.setId(x.getId());
                boFieldVo.setCode(x.getRelationCode().concat(".id"));
                boFieldVo.setSearchable(true);
                if (x.getRelationType().equalsIgnoreCase(FieldLikeRelationType.MULTI_VALUES.getName())) {
                    boFieldVo.setType(FieldType.STRINGS.getType());
                } else {
                    boFieldVo.setType(FieldType.LONG.getType());
                }
                return boFieldVo;
            }).collect(Collectors.toList());

            if (!parentRelationshipVos.isEmpty()) {
                List<RelationshipVo> vos = new ArrayList<>(parentRelationshipVos);
                vos.addAll(relations);
                relations = vos;
            }

            boInfoVo.setRelationshipVos(relations);

            List<BoFieldVo> fields = new ArrayList<>(relatedFields);
            List<BoFieldVo> currentFields = loadBoFields(boId, contextDC);
            fields.addAll(currentFields);
            if (!parentFields.isEmpty()) {
                fields.addAll(parentFields);
            }
            boInfoVo.setFields(fields);
            boInfoVo.setSysType(sysType);
            boInfoVo.setCode(code);
            boInfoVo.setId(Long.parseLong(boId));
            boInfoVo.setName(name);
            boInfoVo.setDomainCode(domainCode);
            boInfoVo.setDomainName(domainName);
            boInfoVo.setRootFlag(rootFlag);

            if (StringUtils.isNoneEmpty(parentId)) {
                boInfoVo.setParentBoId(Long.parseLong(parentId));
            }

            if (StringUtils.isNoneEmpty(domainRootId)) {
                boInfoVo.setDomainRootId(Long.parseLong(domainRootId));
            }


            return Optional.of(boInfoVo);
        });
    }

    private List<BoApiVo> loadApis(String boId, UpdateableDataContext dc) {
        return read(() -> {

            DataSet apis = dc.query()
                    .from(ApiTable.TABLE_NAME)
                    .selectAll().where(ApiTable.BO_ID)
                    .eq(boId).execute();

            return toBoApiVo(boId, apis);
        });
    }

    private List<BoFieldVo> loadBoFields(String boId, UpdateableDataContext dc) {
        return read(() -> {
            DataSet fieldDs = dc.query().from(FieldTable.TABLE_NAME)
                    .selectAll().where(FieldTable.BO_ID).eq(boId).execute();
            return fieldDs.toRows().stream()
                    .map(x -> FieldHelper.toBoFieldVo(boId, x))
                    .collect(Collectors.toList());
        });
    }

    private List<BoApiVo> toBoApiVo(String boId, DataSet apis) {
        List<BoApiVo> boApiVos = new LinkedList<>();
        while (apis.next()) {
            Row row = apis.getRow();
            String code = getRowValue(row, ApiTable.CODE).map(String::valueOf).orElse("");
            String url = getRowValue(row, ApiTable.URL).map(String::valueOf).orElse("");
            String externalUrl = getRowValue(row, ApiTable.EXTERNAL_URL).map(String::valueOf).orElse("");
            String method = getRowValue(row, ApiTable.METHOD).map(String::valueOf).orElse("");
            String param = getRowValue(row, ApiTable.PARAM).map(String::valueOf).orElse("");
            String responseData = getRowValue(row, ApiTable.RESPONSE_DATA).map(String::valueOf).orElse("");
            BoApiVo apiItem = new BoApiVo();

            apiItem.setBoId(Long.parseLong(boId));
            apiItem.setCode(code);
            apiItem.setUrl(url);
            apiItem.setMethod(method);
            apiItem.setParam(param);
            apiItem.setResponseData(responseData);
            boApiVos.add(apiItem);
        }
        return boApiVos;
    }

    private RelationshipVo toRelationVo(String boId, Row row) {

        String id = RowUtils.getRowValue(row, RelationTable.ID).map(String::valueOf).orElse("0");
        String boField = RowUtils.getRowValue(row, RelationTable.BO_FIELD).map(String::valueOf).orElse("");
        String joinField = RowUtils.getRowValue(row, RelationTable.JOIN_FIELD).map(String::valueOf).orElse("");
        String joinBoID = RowUtils.getRowValue(row, RelationTable.JOIN_BO_ID).map(String::valueOf).orElse("");
        String code = RowUtils.getRowValue(row, RelationTable.REL_NAME).map(String::valueOf).orElse("");
        String type = RowUtils.getRowValue(row, RelationTable.REL_TYPE).map(String::valueOf).orElse("");


        RelationshipVo relationshipVo = new RelationshipVo();
        relationshipVo.setBoId(Long.parseLong(boId));
        relationshipVo.setId(Long.parseLong(id));

        if (StringUtils.isNoneEmpty(boField)) {
            relationshipVo.setBoField(Long.parseLong(boField));
        }

        if (StringUtils.isNoneEmpty(joinField)) {
            relationshipVo.setJoinField(Long.parseLong(joinField));
        }

        if (StringUtils.isNoneEmpty(joinBoID)) {
            relationshipVo.setJoinBoId(Long.parseLong(joinBoID));
        }
        relationshipVo.setRelationCode(code);
        relationshipVo.setRelationType(type);

        return relationshipVo;
    }
}
