package com.xforceplus.ultraman.metadata.sync.grpc;

import akka.stream.ActorMaterializer;
import akka.stream.javadsl.AsPublisher;
import akka.stream.javadsl.Sink;
import com.xforceplus.ultraman.config.ConfigurationEngine;
import com.xforceplus.ultraman.config.json.JsonConfigNode;
import com.xforceplus.ultraman.metadata.grpc.Base;
import com.xforceplus.ultraman.metadata.grpc.CheckServiceClient;
import com.xforceplus.ultraman.metadata.grpc.ModuleUpResult;
import com.xforceplus.ultraman.sdk.infra.base.AppIdResolver;
import com.xforceplus.ultraman.sdk.infra.base.AuthConfig;
import com.xforceplus.xplat.galaxy.grpc.client.LongConnect;
import io.reactivex.Observable;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationEventPublisher;

import java.time.Duration;
import java.util.List;

/**
 * module grpc init service
 */
@Slf4j
public class ModuleInitService extends OfflineSupport<List<ModuleUpResult>> implements InitializingBean {

    private Logger logger = LoggerFactory.getLogger(ModuleInitService.class);

    private final CheckServiceClient checkServiceClient;

    private final ActorMaterializer mat;

    private AuthConfig authConfig;

    public ModuleInitService(CheckServiceClient checkServiceClient
            , ActorMaterializer mat, AuthConfig authConfig
            , ApplicationEventPublisher publisher
            , boolean isOffline
    ) {
        super(isOffline);
        this.checkServiceClient = checkServiceClient;
        this.mat = mat;
        this.authConfig = authConfig;
    }

    @Value("${xplat.oqsengine.sdk.init-size:5}")
    private Integer size;

    @Value("${xplat.oqsengine.sdk.init-time:10}")
    private Integer time;

    @Autowired
    private ConfigurationEngine<ModuleUpResult, JsonConfigNode> moduleConfigEngine;

    @Override
    public void afterPropertiesSet() throws Exception {
        Observable<List<ModuleUpResult>> moduleResult = getSource();
        moduleConfigEngine.registerSource(moduleResult.concatMap(Observable::fromIterable));
    }

    @Override
    public Observable<List<ModuleUpResult>> getOnlineSource() {

        String appIds = authConfig.getAppId();

        List<Tuple2<String, String>> appAndEnvTuple = AppIdResolver
                .getAppAndEnvTuple(appIds, authConfig.getEnv());

        return appAndEnvTuple.stream().map(x -> {
            Base.Authorization request = Base.Authorization.newBuilder()
                    .setAppId(x._1())
                    .setEnv(x._2())
                    .setTenantId(authConfig.getTenant())
                    .build();
            return request;
        }).map(req -> {
            Publisher<List<ModuleUpResult>> moduleService = LongConnect.safeSource(2, 20
                    , () -> checkServiceClient.checkStreaming(req))
                    .log("ModuleService")
                    .groupedWithin(size, Duration.ofSeconds(time))
                    .map(x -> {
                        logger.info("Got module size {}", x.size());
                        return x;
                    })
                    .runWith(Sink.asPublisher(AsPublisher.WITH_FANOUT), mat);

            return Observable.fromPublisher(moduleService);
        }).reduce(Observable::mergeWith)
                .orElseGet(() -> {
                    logger.warn("Empty Bo Source");
                    return Observable.empty();
                });
    }
}
