package com.xforceplus.ultraman.metadata.sync.grpc;

import akka.NotUsed;
import akka.stream.ActorMaterializer;
import akka.stream.javadsl.Sink;
import akka.stream.javadsl.Source;
import com.xforceplus.tech.common.utils.JsonHelper;
import com.xforceplus.ultraman.metadata.domain.vo.dto.AppItem;
import com.xforceplus.ultraman.metadata.domain.vo.dto.CurrentVersion;
import com.xforceplus.ultraman.metadata.grpc.NodeServiceClient;
import com.xforceplus.ultraman.metadata.grpc.NodeUp;
import com.xforceplus.ultraman.metadata.repository.MetadataRepository;
import com.xforceplus.ultraman.sdk.infra.base.AuthConfig;
import com.xforceplus.xplat.galaxy.grpc.client.LongConnect;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

/**
 * report node info every config interval
 */
public class NodeReporterInitService implements InitializingBean {

    @Autowired
    private NodeServiceClient nodeServiceClient;

    private Logger logger = LoggerFactory.getLogger(NodeReporterInitService.class);

    @Autowired
    private ActorMaterializer mat;

    private AuthConfig authConfig;

    @Value("${spring.application.name:default}")
    private String applicationName;

    @Autowired(required = false)
    private List<NodeReporterBeforeSendCallback> beforeSendCallbacks;

    @Value("${xplat.oqsengine.sdk.meta.offline.enabled:false}")
    private boolean isOffline;

    @Autowired
    private MetadataRepository metadataRepository;

    public NodeReporterInitService(AuthConfig authConfig) {
        this.authConfig = authConfig;
    }

    @Override
    public void afterPropertiesSet() throws Exception {

        if (!isOffline) {

            //TODO config-able
            Source<NodeUp, NotUsed> reportSource = Source.tick(Duration.ZERO, Duration.ofSeconds(30), "report")
                    .map(x -> fetchCurrentNodeStatus())
                    .map(this::doCallback)
                    .mapMaterializedValue(m -> NotUsed.getInstance());

            LongConnect.safeSource(2, 20
                    , () -> nodeServiceClient.report(reportSource))
                    .log("NodeService")
                    .runWith(Sink.foreach(x -> {
                        logger.debug("[Node-Reporter]Got reply at {}", LocalDateTime.now());
                    }), mat);
        }
    }

    private NodeUp doCallback(NodeUp nodeUp) {
        NodeUp.Builder builder = nodeUp.toBuilder();
        Optional.ofNullable(beforeSendCallbacks).orElseGet(Collections::emptyList).forEach(x -> x.modify(builder));
        return builder.build();
    }

    private NodeUp fetchCurrentNodeStatus() throws SocketException, UnknownHostException {

        Long appId = 0L;
        Long envId = 0L;

        try {
            appId = Long.parseLong(this.authConfig.getAppId());
            envId = Long.parseLong(this.authConfig.getEnv());
        } catch (Exception ex) {

        }

        String sdkVersion = getClass().getPackage().getImplementationVersion();
        logger.info("[Node-Reporter]show version {}", sdkVersion);
        CurrentVersion currentVersion = metadataRepository.currentVersion();
        AppItem currentApp = metadataRepository.getCurrentApp();

        return NodeUp.newBuilder()
                .setAppId(appId)
                .setEnvId(envId)
                .setCode(getNodeName())
                .setName(applicationName)
                .setSdkVersion(Optional.ofNullable(sdkVersion).orElse("UNKNOWN-FROM-TEST"))
                .setCurrentVersion(JsonHelper.toJsonStr(currentVersion))
                .setStatus("OK")
                .setAppCode(Optional.ofNullable(currentApp).map(AppItem::getAppCode).orElse(""))
                .setAppVersion(Optional.ofNullable(currentApp).map(AppItem::getAppVersion).orElse(""))
                .build();
    }

    private String getNodeName() throws SocketException, UnknownHostException {
        InetAddress inetAddress = InetAddress.getLocalHost();
        NetworkInterface byInetAddress = NetworkInterface.getByInetAddress(inetAddress);

        byte[] mac = byInetAddress.getHardwareAddress();

        String macName = null;
        if (mac != null) {
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < mac.length; i++) {
                sb.append(String.format("%02X%s", mac[i], (i < mac.length - 1) ? "-" : ""));
            }

            macName = sb.toString();
        }

        return inetAddress.getHostAddress() + ":" + Optional.ofNullable(macName).orElse("UNKNOWN");
    }
}
