package com.xforceplus.ultraman.metadata.sync.transfer.utils;

import com.fasterxml.jackson.core.type.TypeReference;
import com.xforceplus.metadata.schema.rels.MetadataRelationType;
import com.xforceplus.metadata.schema.typed.*;
import com.xforceplus.metadata.schema.typed.calculator.FieldCalculator;
import com.xforceplus.metadata.schema.typed.dict.DictDetailNode;
import com.xforceplus.metadata.schema.typed.dict.DictNode;
import com.xforceplus.metadata.schema.typed.endpoint.BoHttpEndpoint;
import com.xforceplus.ultraman.metadata.jsonschema.pojo.*;
import com.xforceplus.ultraman.metadata.sync.transfer.configurer.BoConfigurer;
import com.xforceplus.ultraman.metadata.sync.transfer.settings.SdkSettingsHelper;
import com.xforceplus.ultraman.transfer.common.event.SDKMetadataEvent;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nullable;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class DomainConverter {

    private final static TypeReference<Map<String, String>> typeRefer = new TypeReference<Map<String, String>>() {
    };


    public static AppNode toAppNode(SDKMetadataEvent event, String profile, Map<String, List<BoConfigurer>> configurerList) {
        /**
         * using appCodeForDb
         */
        String appCodeForDB = event.getAppCodeForDB();
        AppNode appNode = new AppNode();
        appNode.setCode(appCodeForDB);
        appNode.setVersion(event.getVersion());
        appNode.setId(event.getAppId().toString());
        List<SchemaBo> bos = event.getEntities();
        List<BoNode> bosNodes = Optional.ofNullable(bos).orElseGet(Collections::emptyList).stream().flatMap(x ->
                {
                    List<BoConfigurer> configType = configurerList.get(x.getCode());
                    return DomainConverter.toBoNode(x, configType).stream();
                }
        ).collect(Collectors.toList());
        List<DictNode> dictNodes = Optional.ofNullable(event.getDicts()).orElseGet(Collections::emptyList).stream()
                .flatMap(x -> DomainConverter.toDictNode(x).stream()).collect(Collectors.toList());
        appNode.setBoNodes(bosNodes);
        appNode.setDicts(dictNodes);
        List<SchemaSdkSetting> sdkSettings = event.getSdkSettings();
        if (sdkSettings != null && !sdkSettings.isEmpty()) {
            //multi
            List<DynamicNode> nodes = SdkSettingsHelper.getNode(sdkSettings, profile);
            appNode.setDynamicNodes(nodes);
        }

        return appNode;
    }

    public static List<BoNode> toBoNode(SchemaBo schemaBo, List<BoConfigurer> boConfigurers) {
        List<BoNode> boNodes = new ArrayList<>();

        BoNode boNode = toProfiledBoNode(schemaBo, null, boConfigurers);
        boNodes.add(boNode);

        List<SchemaTenantBo> tenantBos = Optional
                .ofNullable(schemaBo.getTenantBos())
                .orElseGet(Collections::emptyList);

        tenantBos.forEach(x -> {
            BoNode tenantBo = toProfiledBoNode(schemaBo, x, boConfigurers);
            boNodes.add(tenantBo);
        });

        return boNodes;
    }

    /**
     * to tenant bo
     *
     * @param schemaBo
     * @param schemaTenantBo
     * @return
     */
    private static BoNode toProfiledBoNode(SchemaBo schemaBo, @Nullable SchemaTenantBo schemaTenantBo, List<BoConfigurer> configurers) {
        BoNode boNode = new BoNode();
        boNode.setParentId(schemaBo.getParentBoId());
        boNode.setId(schemaBo.getId());
        boNode.setCode(schemaBo.getCode());
        boNode.setName(schemaBo.getName());
        boNode.setBoType(schemaBo.getBoType());

        /**
         * TODO
         */
        String profile = Optional.ofNullable(schemaTenantBo).map(SchemaTenantBo::getTenantCode).orElse(null);

        List<BoRelationship> relationships = new ArrayList<>();

        /**
         * relation
         */
        relationships.addAll(Optional.ofNullable(schemaBo).map(SchemaBo::getBoRelationships)
                .orElseGet(Collections::emptyList).stream().map(DomainConverter::toRelationShip)
                .collect(Collectors.toList()));
        relationships.addAll(Optional.ofNullable(schemaTenantBo).map(SchemaTenantBo::getBoRelationships)
                .orElseGet(Collections::emptyList).stream().map(DomainConverter::toRelationShip)
                .collect(Collectors.toList()));
        boNode.setRelationships(relationships);

        /**
         * field
         */
        List<BoField> fields = new ArrayList<>();

        fields.addAll(Optional.ofNullable(schemaBo.getBoFields())
                .orElseGet(Collections::emptyList).stream().map(x -> toField(boNode.getId(), x, null))
                .collect(Collectors.toList()));
        fields.addAll(Optional.ofNullable(schemaTenantBo).flatMap(x -> {
                    String tenantCode = x.getTenantCode();
                    List<SchemaBoField> boFields = x.getBoFields();
                    return Optional.ofNullable(boFields).map(f -> f.stream().map(fs -> toField(boNode.getId(), fs, tenantCode)).collect(Collectors.toList()));
                })
                .orElseGet(Collections::emptyList));
        boNode.setFields(fields);
        boNode.setBoType(schemaBo.getBoType());
        //TODO profile

        boNode.setProfile(profile);

        /**
         * index
         */
        List<BoIndex> boIndices = new ArrayList<>();
        boIndices.addAll(Optional.ofNullable(schemaBo.getBoIndexes())
                .orElseGet(Collections::emptyList).stream().map(x -> boIndex(x, null))
                .collect(Collectors.toList()));
        boIndices.addAll(Optional.ofNullable(schemaTenantBo).map(SchemaTenantBo::getBoIndexes)
                .orElseGet(Collections::emptyList).stream().map(x -> boIndex(x, profile))
                .collect(Collectors.toList()));

        boNode.setBoIndices(boIndices);


        List<BoHttpEndpoint> apis = new ArrayList<>();
        List<BoHttpEndpoint> mainApis = Optional.ofNullable(schemaBo.getBoApis()).orElseGet(Collections::emptyList).stream().map(x -> toApi(x, boNode.getId(), null))
                .collect(Collectors.toList());
        mainApis.addAll(apis);
        mainApis.addAll(Optional.ofNullable(schemaTenantBo).map(SchemaTenantBo::getBoApis)
                .orElseGet(Collections::emptyList).stream().map(x -> toApi(x, boNode.getId(), profile))
                .collect(Collectors.toList()));
        boNode.setApis(mainApis);


        Optional.ofNullable(configurers).orElseGet(Collections::emptyList).forEach(x -> {
            x.config(boNode);
        });

        return boNode;
    }

    private static BoHttpEndpoint toApi(SchemaBoApi api, String boId, String profile) {
        BoHttpEndpoint boHttpEndpoint = new BoHttpEndpoint();
        boHttpEndpoint.setBoId(boId);
        boHttpEndpoint.setCode(api.getCode());
        boHttpEndpoint.setName(api.getCode());
        boHttpEndpoint.setMethod(api.getMethod());
        boHttpEndpoint.setUrl(api.getUrl());
        boHttpEndpoint.setHeaders(api.getRequestHeader());
        boHttpEndpoint.setParams(api.getParams());
        boHttpEndpoint.setProfile(profile);
        return boHttpEndpoint;
    }

    /**
     * rule is what?
     *
     * @param schemaBoIndex
     * @param profile
     * @return
     */
    private static BoIndex boIndex(SchemaBoIndex schemaBoIndex, String profile) {
        BoIndex boIndex = new BoIndex();
        boIndex.setName(schemaBoIndex.getCode());
        boIndex.setProfile(profile);
        boIndex.setPrimary("primary".equals(schemaBoIndex.getType()));
        boIndex.setUnique("unique".equals(schemaBoIndex.getType()));
        boIndex.setFieldIds(schemaBoIndex.getFieldCodes());
        if(schemaBoIndex.getRule() != null) {
            boIndex.setRule(Integer.parseInt(schemaBoIndex.getRule()));
        }
        return boIndex;
    }

    /**
     * TODO 计算字段缺失 等开发计算字段的时候使用
     *
     * @param schemaBoField
     * @return
     */
    private static BoField toField(String boId, SchemaBoField schemaBoField, String profile) {
        BoField boField = new BoField();
        boField.setName(schemaBoField.getName());
        boField.setCode(schemaBoField.getCode());
        boField.setId(schemaBoField.getId());
        boField.setDynamic(schemaBoField.isDynamic());
        boField.setProfile(StringUtils.isEmpty(profile) ? null : profile);
        boField.setBoId(boId);
        boField.setDefaultValue(schemaBoField.getDefaultValue());

        //TODO
        FieldType fieldType = new FieldType();
        if (!StringUtils.isEmpty(schemaBoField.getValueType())) {
            fieldType.setValueType(schemaBoField.getValueType());
        } else {
            fieldType.setValueType(schemaBoField.getFieldType());
        }
        fieldType.setDictId(schemaBoField.getDictId());
        boField.setFieldType(fieldType);

        Integer length = schemaBoField.getLength();
        Integer decimalPoint = schemaBoField.getDecimalPoint();
        boolean searchable = schemaBoField.isSearchable();
        boolean editable = schemaBoField.isEditable();
        boolean required = schemaBoField.isRequired();

        ExtraConfig extraConfig = new ExtraConfig();
        extraConfig.setLength(length);
        extraConfig.setDecimalPoint(decimalPoint);
        extraConfig.setSearchable(searchable);
        extraConfig.setEditable(editable);
        extraConfig.setRequired(required);

        boField.setExtraConfig(extraConfig);
        
        //calculator
        if("lookup".equalsIgnoreCase(schemaBoField.getFieldType())){
            //lookup
            FieldCalculator calculator = new FieldCalculator();
            SchemaBoFieldCalculator requestCalculator = schemaBoField.getCalculator();
            calculator.setRefFieldId(requestCalculator.getLookupFieldId());
            calculator.setRefBoId(requestCalculator.getLookupBoId());
            calculator.setRefRelId(requestCalculator.getLookupRelationId());
            boField.setCalculator(calculator);
        }

        return boField;
    }
    
    private static BoRelationship toRelationShip(SchemaBoRelationship relationship) {
        BoRelationship boRelationship = new BoRelationship();
        boRelationship.setRelationCode(relationship.getRelationCode());
        boRelationship.setRelationName(relationship.getRelationName());
        if ("OneToMany".equalsIgnoreCase(relationship.getRelationType()) 
                || "ManyToMany".equalsIgnoreCase(relationship.getRelationType())) {
            boRelationship.setRelationType(MetadataRelationType.TO_MANY.name());
        } else if ("ManyToOne".equalsIgnoreCase(relationship.getRelationType())) {
            boRelationship.setRelationType(MetadataRelationType.TO_ONE.name());
        } else if ("MultiValues".equalsIgnoreCase(relationship.getRelationType())) {
            boRelationship.setRelationType(MetadataRelationType.MULTI_VALUES.name());
        }

        //TODO
        boRelationship.setId(relationship.getId());
        boRelationship.setBoId(relationship.getBoId());
        boRelationship.setJoinBoId(relationship.getJoinBoId());
        boRelationship.setJoinBoFieldId(relationship.getJoinField());
        boRelationship.setBoFieldId(relationship.getBoField());
        boRelationship.setStrong("1".equals(relationship.getStrongFlag()));

        return boRelationship;
    }

    public static List<DictNode> toDictNode(SchemaDict schemaDict) {

        List<DictNode> list = new ArrayList<>();
        DictNode dictNode = toProfiledDictNode(schemaDict, null);
        list.add(dictNode);

        List<SchemaTenantDict> tenantDicts = schemaDict.getTenantDicts();

        tenantDicts.forEach(x -> {
            DictNode tenantNode = toProfiledDictNode(schemaDict, x);
            list.add(tenantNode);
        });

        return list;
    }

    private static DictNode toProfiledDictNode(SchemaDict schemaDict, @Nullable SchemaTenantDict dict) {
        DictNode dictNode = new DictNode();

        String profile = Optional.ofNullable(dict).map(SchemaTenantDict::getTenantCode).orElse(null);
        dictNode.setProfile(profile);

        List<SchemaDictDetail> details = null;
        if (dict == null) {
            dictNode.setId(schemaDict.getId());
            dictNode.setPublicId(schemaDict.getId());
            details = schemaDict.getDetails();
        } else {
            dictNode.setId(dict.getId());
            dictNode.setPublicId(Optional.ofNullable(dict.getRefDictId()).orElse(schemaDict.getId()));
            details = dict.getDetails();
        }
        dictNode.setCode(schemaDict.getCode());
        dictNode.setName(schemaDict.getName());
        List<DictDetailNode> items = new ArrayList<>();
        dictNode.setItems(items);

        AtomicInteger index = new AtomicInteger(0);
        details.stream().forEach(x -> {
            DictDetailNode node = new DictDetailNode();
            node.setCode(x.getCode());
            node.setName(x.getName());
            node.setIcon(x.getIcon());
            node.setIndex(index.incrementAndGet());
            items.add(node);
        });
        
        
        
        

        return dictNode;
    }
}