package com.xforceplus.ultraman.metadata.sync.transfer.service.impl;

import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.xforceplus.metadata.schema.dsl.metadata.__;
import com.xforceplus.metadata.schema.runtime.MetadataEngine;
import com.xforceplus.tech.base.core.context.ContextKeys.StringKeys;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.metadata.domain.vo.dto.DictItem;
import com.xforceplus.ultraman.metadata.domain.vo.dto.ResponseList;
import com.xforceplus.ultraman.metadata.service.DictService;
import com.xforceplus.ultraman.sdk.infra.CacheLike;
import com.xforceplus.ultraman.sdk.infra.Refreshable;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import org.apache.commons.lang.StringUtils;
import org.apache.tinkerpop.gremlin.process.traversal.P;
import org.apache.tinkerpop.gremlin.process.traversal.dsl.graph.GraphTraversal;
import org.apache.tinkerpop.gremlin.structure.Vertex;

import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.metadata.schema.dsl.Step.DICT;
import static com.xforceplus.metadata.schema.dsl.utils.MatcherHelper.caseInsensitiveEq;
import static com.xforceplus.metadata.schema.rels.MetadataRelationType.HAS_ITEM;
import static com.xforceplus.metadata.schema.runtime.MetadataEngine.*;

/**
 * TODO
 * move from entityServiceEx
 */
public class DictServiceImpl implements DictService, CacheLike, Refreshable {

    private final ContextService contextService;

    private MetadataEngine metadataEngine;

    private LoadingCache<Tuple2<String, String>, List<DictItem>> dictCache;

    private LoadingCache<Tuple2<String, String>, List<DictItem>> dictCodeCache;

    public DictServiceImpl(ContextService contextService, MetadataEngine metadataEngine) {
        this.metadataEngine = metadataEngine;
        this.contextService = contextService;
        this.dictCache = Caffeine.newBuilder()
                .maximumSize(1000)
                .build(tuple -> {
                    String dictId = tuple._1;
                    String tenantCode = tuple._2;
                    return metadataEngine
                            .raw(g -> {
                                GraphTraversal<Vertex, Vertex> has = g.traversal().V()
                                        .has(LABEL_INDEX, DICT);
                                if (dictId != null) {
                                    has = has.has("publicId", dictId);
                                }

                                GraphTraversal<Vertex, Map<Object, Object>> traversal =
                                        has.has("profile", P.within(null, tenantCode))
                                                .as("a")
                                                .outE(HAS_ITEM.name())
                                                .inV().as("b").select("a", "b").by(__.valueMap()).group().by("a");

                                ResponseList<DictItem> candidateProfile = new ResponseList<>();
                                ResponseList<DictItem> candidateNonProfile = new ResponseList<>();

                                if (traversal.hasNext()) {
                                    Map<Object, Object> next = traversal.next();
                                    if (next != null) {
                                        //if profile next will have two entry
                                        next.forEach((key, value) -> {
                                            Map<String, Object> keyMap = (Map<String, Object>) key;
                                            List<Map<String, Object>> valueList = (List<Map<String, Object>>) value;
                                            String profile = getFromMap(keyMap, PROFILE_INDEX);

                                            List<DictItem> dictItems = toDictItems(keyMap, valueList);

                                            if (!StringUtils.isEmpty(profile)) {
                                                candidateProfile.addAll(dictItems);
                                            } else {
                                                candidateNonProfile.addAll(dictItems);
                                            }
                                        });

                                        if (candidateProfile.isEmpty()) {
                                            return candidateNonProfile;
                                        } else {
                                            return candidateProfile;
                                        }
                                    }
                                }
                                return null;
                            });
                });

        this.dictCodeCache = Caffeine.newBuilder()
                .maximumSize(1000)
                .build(tuple -> {
                    String code = tuple._1;
                    String tenantCode = tuple._2;
                    return metadataEngine
                            .raw(g -> {
                                GraphTraversal<Vertex, Vertex> has = g.traversal().V()
                                        .has(LABEL_INDEX, DICT);
                                if (code != null) {
                                    has = has.has(CODE_INDEX, caseInsensitiveEq(code));
                                }

                                GraphTraversal<Vertex, Map<Object, Object>> traversal =
                                        has.has("profile", P.within(null, tenantCode))
                                                .as("a")
                                                .outE(HAS_ITEM.name())
                                                .inV().as("b").select("a", "b").by(__.valueMap()).group().by("a");

                                ResponseList<DictItem> candidateProfile = new ResponseList<>();
                                ResponseList<DictItem> candidateNonProfile = new ResponseList<>();
                                if (traversal.hasNext()) {
                                    Map<Object, Object> next = traversal.next();
                                    if (next != null) {
                                        next.forEach((key, value) -> {
                                            Map<String, Object> keyMap = (Map<String, Object>) key;
                                            String profile = getFromMap(keyMap, PROFILE_INDEX);
                                            List<Map<String, Object>> valueList = (List<Map<String, Object>>) value;
                                            List<DictItem> dictItems = toDictItems(keyMap, valueList);
                                            if (!StringUtils.isEmpty(profile)) {
                                                candidateProfile.addAll(dictItems);
                                            } else {
                                                candidateNonProfile.addAll(dictItems);
                                            }
                                        });
                                        return Optional.ofNullable(candidateProfile).orElse(candidateNonProfile);
                                    }
                                } 
                                
                                return null;
                            });
                });
    }

    /**
     * find enumCode with tenantCode and enumId and enumCode
     *
     * @param enumCode
     * @param tenantCode
     * @return
     */
    @Override
    public List<DictItem> findDictItems(String dictId, String enumCode, String tenantCode) {
        List<DictItem> dictItems = dictCache.get(Tuple.of(dictId, tenantCode));
        return Optional.ofNullable(dictItems)
                .orElseGet(Collections::emptyList)
                .stream().filter(item -> {
                    if (enumCode != null) {
                        return item.getValue().equals(enumCode);
                    } else {
                        return true;
                    }
                }).collect(Collectors.toCollection(ResponseList::new));
    }

    private String getFromMap(Map<String, Object> main, String key) {
        return Optional.ofNullable(main.get(key))
                .map(x -> (List) x)
                .filter(x -> !x.isEmpty())
                .map(x -> x.get(0))
                .map(Object::toString).orElse(null);
    }

    private List<DictItem> toDictItems(Map<String, Object> main, List<Map<String, Object>> itemMap) {

        String id = getFromMap(main, "id");
        String code = getFromMap(main, "code");
        String name = getFromMap(main, "name");

        return itemMap.stream().map(x -> {
            DictItem item = new DictItem();
            item.setDictName(name);
            item.setDictId(id);
            item.setDictCode(code);

            Object b = x.get("b");
            if (b != null) {
                Map<String, Object> detailMap = (Map<String, Object>) b;
                String detailCode = getFromMap(detailMap, "code");
                String detailName = getFromMap(detailMap, "name");
                String detailIcon = getFromMap(detailMap, "icon");
                String index = getFromMap(detailMap, "index");
                String detailI18n = getFromMap(detailMap, "i18n");

                item.setValue(detailCode);
                item.setText(detailName);
                item.setIcon(detailIcon);
                item.setI18n(detailI18n);
                item.setIndex(Optional.ofNullable(index).map(Integer::parseInt).orElse(0));
            }
            return item;
        }).sorted(Comparator.comparingInt(DictItem::getIndex)).collect(Collectors.toCollection(ResponseList::new));
    }

    @Override
    public List<DictItem> findDictItemsByCode(String code, String enumCode, String tenantCode) {
        return Optional.ofNullable(dictCodeCache.get(Tuple.of(code, tenantCode)))
                .orElseGet(Collections::emptyList)
                .stream().filter(item -> {
                    if (enumCode != null) {
                        return item.getValue().equals(enumCode);
                    } else {
                        return true;
                    }
                }).collect(Collectors.toCollection(ResponseList::new));
    }

    @Override
    public List<DictItem> findAllDictItems(String tenantCode) {
        return findDictItemsByCode(null, null, tenantCode);
    }

    @Override
    public List<DictItem> findAllDictItems() {
        String tenantCode = contextService.get(StringKeys.TENANTCODE_KEY);
        return findAllDictItems(tenantCode);
    }

    @Override
    public Integer findEnumIndex(List<DictItem> dictItems, String value) {
        Optional<DictItem> first = dictItems.stream().filter(x -> x.getValue().equalsIgnoreCase(value)).findFirst();
        if (first.isPresent()) {
            return Double.valueOf(Math.pow(2, first.get().getIndex())).intValue();
        } else {
            return 0;
        }
    }

    @Override
    public List<DictItem> findDictItems(String enumId, String enumCode) {
        String tenantCode = contextService.get(StringKeys.TENANTCODE_KEY);
        return findDictItems(enumId, enumCode, tenantCode);
    }

    @Override
    public List<DictItem> findDictItems(String enumId, String enumCode, Map<String, Object> context) {
        String tenantCode = Optional.ofNullable(context.get(StringKeys.TENANTCODE_KEY.name())).map(x -> x.toString()).orElse(null);
        return findDictItems(enumId, enumCode, tenantCode);
    }

    @Override
    public List<DictItem> findDictItemsByCode(String code, String enumCode) {
        String tenantCode = contextService.get(StringKeys.TENANTCODE_KEY);
        return findDictItemsByCode(code, enumCode, tenantCode);
    }

    @Override
    public void onRefresh(Object payload) {
        refreshCache();
    }

    @Override
    public void refreshCache() {
        this.dictCache.invalidateAll();
        this.dictCodeCache.invalidateAll();
    }
}
