package com.xforceplus.ultraman.metadata.entity;

import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.xforceplus.metadata.schema.typed.BoIndex;

import java.sql.SQLException;
import java.util.Collection;
import java.util.Optional;
import java.util.Set;

/**
 * Entity结构对象
 *
 * @author wangzheng
 * @version 1.0 2020/3/26 15:10
 */
@JsonInclude(JsonInclude.Include.NON_DEFAULT)
@JsonPropertyOrder({"id", "version", "name", "code", "relations", "relatedEntityClasses", "fields"})
public interface IEntityClass {

    /**
     * 获得对象的id
     *
     * @return id
     */
    @JsonGetter("id")
    public long id();

    void setType(int type);

    /**
     * 对象code
     *
     * @return
     */
    @JsonGetter("code")
    public String code();

    /**
     * 对象版本号.
     *
     * @return 版本号.
     */
    @JsonGetter("version")
    public int ver();

    /**
     * 名称.
     *
     * @return
     */
    @JsonGetter("name")
    public String name();

    /**
     * 关系信息的集合
     *
     * @return 根对象默认为Null，OneToOne为OTO，OneToMany为OTM
     */
    @JsonGetter("relations")
    public Collection<IRelation> relations();

    /**
     * 获得本对象的下一级子对象
     *
     * @return 子对象集合 - 子对象目前不继续往下钻
     */
    @JsonGetter("relatedEntityClasses")
    public Collection<IEntityClass> entityClasses();

    @JsonGetter("actions")
    public Set<String> actions();

    /**
     * 获取所有index.
     *
     * @return 返回所有索引集合.
     */
    @JsonGetter("indexes")
    public Collection<BoIndex> indexes();

    /**
     * 获取业务主键index.
     *
     * @return 返回所有业务主键索引集合.
     */
    public Collection<BoIndex> uniqueIndexes();


    /**
     * 继承于当前类型的子类.这里组织成了一个多叉树.
     *                   当前类型
     *                   /    \
     *                继承者A  继承者B
     *  如果继承者A或者B还有继承者,那递归.
     * @return 当前类型继承的直接下一级继承者类型列表.
     */
    @JsonGetter("childEntityClasses")
    public Collection<IEntityClass> childEntityClasses();

    /**
     * 获取当前对象继承的对象类型.
     *
     * @return 继承的目标类型
     */
    @JsonGetter("parentEntityClass")
    public IEntityClass extendEntityClass();

    /**
     * 本对象的属性信息
     *
     * @return 属性集合
     */
    @JsonGetter("fields")
    public Collection<IEntityField> fields();


    /**
     * 本对象的属性信息
     *
     * @return 属性集合
     */
    @JsonGetter("fields")
    Collection<IEntityField> selfFields();

    /**
     *
     * @return
     */
    Collection<IEntityClass> family();

    /**
     *
     * @return
     * @throws SQLException
     */
    Collection<IEntityField> selfWithUnique();

    /**
     *
     * @return
     */
    Collection<IEntityField> uniqueFieldCollection();

    /**
     *
     * @return
     * @throws SQLException
     */
    Collection<IEntityField> selfWithIndex();

    /**
     *
     * @return
     */
    Collection<IEntityField> indexFieldCollection();


    /**
     * 对象的类型
     *
     * @return 属性集合
     */
    @JsonGetter("type")
    default int getType() {
        return 0;
    }

    Collection<IEntityField> withoutRelationFields();

    Collection<IEntityField> selfWithoutRelationFields();

    /**
     * 本地对象指定字段的信息.
     *
     * @param name 字段名称.
     * @return 字段信息.
     */
    public Optional<IEntityField> field(String name);

    /**
     * 使用字段 ID 获取属性信息.
     *
     * @param id 属性ID.
     * @return 属性名称.
     */
    public Optional<IEntityField> field(long id);

    /**
     * 判断是否表示一个通用类型,和 JAVA 中 Object.class 相似的概念.
     *
     * @return true 是通用类型,false 不是通用类型.
     */
    public boolean isAny();

    String indexQueryTable();

    /**
     * 是否是一个动态的类型 默认是
     *
     * @return
     */
    default boolean isDynamic() {
        return true;
    }

    EntityClassType type();

    String masterQueryTable();

    String masterWriteTable();

    public void resetChildEntityClass(Collection<IEntityClass> childEntityClasses);

    String appCode();

    String profile();

    String realProfile();

    Optional<IEntityClass> father();

    int version();

    int level();

    EntityClassRef ref();

    IEntityClass root();

    default boolean needChangLog() {
        return false;
    }
}
