package com.xforceplus.ultraman.metadata.engine.dsl;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.stream.Stream;

/**
 * Resource Path to visit
 * to identify a resource
 */
public class ResourcePath {

    private static Logger logger = LoggerFactory.getLogger(ResourcePath.class);

    private LinkedList<ResourcePart> parts = new LinkedList<>();

    public static ResourcePath parse(String path) {
        return new Parser().from(path);
    }

    public void setRoot(RootResourcePart root) {
        if (parts.isEmpty()) {
            parts.add(root);
        } else {
            logger.warn("Root only can be add once, discard {}", root);
        }
    }

    public Stream<ResourcePart> stream(){
        return parts.stream();
    }

    public Iterator<ResourcePart> iterator(){
        return parts.iterator();
    }

    public boolean push(ResourcePart resourcePart) {
        boolean canAdd = parts.getLast().nextType().stream().anyMatch(x -> x == resourcePart.type());
        if (canAdd) {
            parts.add(resourcePart);
            return true;
        } else {
            logger.warn("Cannot add part {}, last one is {}", resourcePart, parts.getLast());
            return false;
        }
    }

    public boolean isMultiResource(){
        return parts.getLast().isMulti();
    }

    @Override
    public String toString() {
        return "ResourcePath{" +
                "parts=" + parts +
                '}';
    }

    //    public Builder newBuilder(){
//        return new Builder();
//    }

    /**
     * resource type
     */
    enum ResourceType {
        ENTITY_CLASS,

        ENTITY_REF,

        SUB_ENTITY,

        FIELDS;
    }

//TODO builder is not required current
//    static class Builder {
//
//        private List<ResourcePart> parts = new LinkedList<>();
//
//        Builder(){
//        }
//
//        /**
//         * build root
//         * @return
//         */
//        Builder root(RootResourcePart root){
//            if(parts.isEmpty()){
//                parts.add(root);
//            } else {
//                parts.set(0, root);
//            }
//            return this;
//        }
//
//        Builder field(FieldResourcePart)
//    }

    /**
     * parser to parse the string to objects
     */
    public static class Parser {
        /**
         * means current root entity
         */
        public final static String ROOT = "@";

        /**
         * PATH
         */
        public final static String PART_DEL = "#";

        public final static String REF_PREFIX = "_";

        public final static String SUB = "/";

        public final static String STAR = "*";

        /**
         *
         */
        public final static String REF_DEL = ".";

        public final static String REG_REF_DEL = "\\.";

        public ResourcePath from(String path) {
            String[] parts = path.split(PART_DEL);

            ResourcePath resourcePath = new ResourcePath();

            boolean hasRoot = false;

            for (String part : parts) {
                String rawPart = part;
                if (rawPart.startsWith(ROOT)) {
                    if (!hasRoot) {
                        //current part is a root path
                        rawPart = rawPart.substring(1);
                        //root can only be the entityClass
                        String entityCode = rawPart;
                        resourcePath.setRoot(new EntityClassResource(entityCode));
                        hasRoot = true;
                    } else {
                        logger.warn("Can only have one root, discard {}", rawPart);
                    }
                } else if (part.startsWith(REF_PREFIX)) {
                    //part is a related code
                    rawPart = rawPart.substring(1);
                    String entityCode;
                    String relatedCode = null;
                    if (rawPart.contains(REF_DEL)) {
                        //current is a ResourceEnd
                        String[] fieldPart = rawPart.split(REG_REF_DEL, 2);
                        entityCode = fieldPart[0];
                        relatedCode = fieldPart[1];
                    } else {
                        entityCode = rawPart;
                    }

                    if (!StringUtils.isEmpty(entityCode)) {

                        boolean success = resourcePath.push(new RelatedEntityClassResource(entityCode));
                        if (!success) {
                            break;
                        }
                    }

                    String nextPart = relatedCode;
                    if (!StringUtils.isEmpty(nextPart)) {

                        if (nextPart.contains(SUB)) {
                            String[] subFields = nextPart.split(SUB);
                            String subEntityClass = subFields[0];
                            nextPart = subFields[1];
                            boolean success = resourcePath.push(new SubEntityClassResource(subEntityClass));
                            if (!success) {
                                break;
                            }
                        }

                        if (nextPart.equals(STAR)) {
                            boolean success = resourcePath.push(FieldsResource.ins);
                            if (!success) {
                                break;
                            }
                        } else {
                            boolean success = resourcePath.push(new MainFieldResource(nextPart));
                            if (!success) {
                                break;
                            }
                        }
                    }
                } else {

                    if (rawPart.contains(SUB)) {
                        String[] subFields = rawPart.split(SUB);
                        String subEntityClass = subFields[0];
                        rawPart = subFields[1];
                        boolean success = resourcePath.push(new SubEntityClassResource(subEntityClass));
                        if (!success) {
                            break;
                        }
                    }

                    if (rawPart.equals(STAR)) {
                        boolean success = resourcePath.push(FieldsResource.ins);
                        if (!success) {
                            break;
                        }
                    } else {
                        boolean success = resourcePath.push(new MainFieldResource(rawPart));
                        if (!success) {
                            break;
                        }
                    }

                }
            }

            return resourcePath;
        }
    }
}
