package com.xforceplus.ultraman.metadata.engine.impl;

import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.xforceplus.metadata.schema.rels.MetadataRelationType;
import com.xforceplus.metadata.schema.runtime.MetadataEngine;
import com.xforceplus.metadata.schema.typed.BoNode;
import com.xforceplus.ultraman.metadata.domain.vo.dto.BoApiVo;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.engine.dsl.*;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.metadata.entity.impl.LazyEntityClass;
import com.xforceplus.ultraman.metadata.entity.legacy.impl.ColumnField;
import com.xforceplus.ultraman.metadata.repository.MetadataRepository;
import io.opentelemetry.instrumentation.annotations.WithSpan;
import lombok.extern.slf4j.Slf4j;
import org.apache.tinkerpop.gremlin.process.traversal.P;
import org.apache.tinkerpop.gremlin.process.traversal.dsl.graph.__;
import org.janusgraph.core.attribute.Text;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import static com.xforceplus.metadata.schema.dsl.Step.APP;
import static com.xforceplus.metadata.schema.dsl.Step.BO;
import static com.xforceplus.metadata.schema.dsl.utils.MatcherHelper.caseInsensitiveEq;
import static com.xforceplus.metadata.schema.rels.MetadataRelationType.HAS_API;
import static com.xforceplus.metadata.schema.runtime.MetadataEngine.*;
import static org.apache.tinkerpop.gremlin.groovy.jsr223.dsl.credential.__.has;

/**
 * produce engine
 */
@Slf4j
public class EntityClassEngineImpl implements EntityClassEngine {

    private MetadataEngine metadataEngine;

    private Map<UnionKey, UnionKey> mappingKey = new ConcurrentHashMap<>();

    private LoadingCache<UnionKey, IEntityClass> l2Cache;

    private LoadingCache<ProfiledEntityClass, EntityClassGroup> groupCache;

    private List<String> codes;


    @Override
    public void onRefresh(Object payload) {
        this.l2Cache.invalidateAll();
        this.groupCache.invalidateAll();
        codes = null;
        mappingKey.clear();
    }

    class ProfiledEntityClass {

        private String profile;

        private IEntityClass entityClass;

        public ProfiledEntityClass(String profile, IEntityClass entityClass) {
            this.profile = profile;
            this.entityClass = entityClass;
        }

        public String getProfile() {
            return profile;
        }

        public IEntityClass getEntityClass() {
            return entityClass;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            ProfiledEntityClass that = (ProfiledEntityClass) o;
            return Objects.equals(profile, that.profile) &&
                    Objects.equals(entityClass, that.entityClass);
        }

        @Override
        public int hashCode() {
            return Objects.hash(profile, entityClass);
        }
    }

    public EntityClassEngineImpl(MetadataEngine metadataEngine) {
        this.metadataEngine = metadataEngine;
        l2Cache = Caffeine.newBuilder()
                .maximumSize(1000)
                .build(key -> {
                    log.info("[Meta]Loading Entity {} from meta", key);
                    Optional<IEntityClass> optionalEntity = Optional.empty();

                    if (key.id != null) {
                        //loading with id
                        if (key.version != null) {
                            optionalEntity = loadByIdAndVersionInner(String.valueOf(key.id), key.version);
                        } else {
                            optionalEntity = loadByIdAndProfileInner(String.valueOf(key.id), key.profile);
                        }
                    } else if (key.code != null) {
                        if (key.version != null) {
                            optionalEntity = loadByCodeAndVersionInner(key.code, key.version);
                        } else {
                            optionalEntity = loadByCodeAndProfileInner(key.code, key.profile);
                        }
                    } else {
                        return null;
                    }

                    if (optionalEntity.isPresent()) {
                        IEntityClass entityClass = optionalEntity.get();

                        UnionKey callKey = new UnionKey(key.code, key.id, key.version, key.profile);
                        UnionKey fullKey = new UnionKey(entityClass.code(), entityClass.id(), key.version, key.profile);
                        mappingKey.put(fullKey, callKey);
                        return entityClass;
                    } else {
                        return null;
                    }
                });

        groupCache = Caffeine.newBuilder()
                .maximumSize(1000)
                .build(key -> {
                    IEntityClass entityClass = key.entityClass;
                    String profile = key.profile;
                    log.info("[Meta]Construct Reader for Entity {}:{}", entityClass.id(), entityClass.code());
                    return this.describeInner(entityClass, profile);
                });
    }

    private Optional<IEntityClass> loadByCodeAndVersionInner(String boCode, String version) {
        return metadataEngine.read(() -> {
            Optional<Map<String, Object>> boNodeMapOp = metadataEngine.get(__.has(LABEL_INDEX, BO)
                    .has(CODE_INDEX, caseInsensitiveEq(boCode)));

            return boNodeMapOp.map(map -> {
                Optional<Map<String, Object>> appMap = metadataEngine.get(__.has(LABEL_INDEX, APP)
                        .where(__.out(MetadataRelationType.HAS_BO.name())
                                .has(LABEL_INDEX, BO).has(CODE_INDEX, caseInsensitiveEq(boCode))));
                Map<String, Object> appRealMap = appMap.get();
                BoNode boNode = new BoNode(boNodeMapOp.get());
                return new LazyEntityClass(metadataEngine, appRealMap.get(CODE_INDEX).toString(), boNode);
            });
        });
    }

    private Optional<IEntityClass> loadByIdAndVersionInner(String boId, String version) {
        return metadataEngine.read(() -> {
            Optional<Map<String, Object>> boNodeMapOp = metadataEngine.get(__.has(LABEL_INDEX, BO)
                    .has(ID_INDEX, boId));

            Optional<Map<String, Object>> appMapOp = metadataEngine.get(__.has(LABEL_INDEX, APP)
                    .where(__.out(MetadataRelationType.HAS_BO.name())
                            .has(LABEL_INDEX, BO).has(ID_INDEX, boId)));

            return boNodeMapOp.map(map -> {
                Map<String, Object> appMap = appMapOp.get();
                BoNode boNode = new BoNode(boNodeMapOp.get());
                return new LazyEntityClass(metadataEngine, appMap.get(CODE_INDEX).toString(), boNode);
            });
        });
    }

    private Optional<IEntityClass> loadByIdAndProfileInner(String boId, String profile) {
        return metadataEngine.read(() -> {
            List<Map<String, Object>> multi = metadataEngine.getMulti(__.has(LABEL_INDEX, BO)
                    .has(ID_INDEX, boId)
                    .where(has(PROFILE_INDEX, caseInsensitiveEq(profile)).or().hasNot(PROFILE_INDEX)));

            if(multi.isEmpty()) {
                return Optional.empty();
            } else {
                Optional<Map<String, Object>> boNodeMapOp = multi.stream().filter(x -> x.get(PROFILE_INDEX) != null).findFirst();
                Optional<Map<String, Object>> nullProfileOp = multi.stream().filter(x -> x.get(PROFILE_INDEX) == null).findFirst();
                Optional<Map<String, Object>> finalOne = boNodeMapOp.isPresent() ? boNodeMapOp : nullProfileOp;
                return finalOne.map(map -> {
                    Optional<Map<String, Object>> appMap = metadataEngine.get(__.has(LABEL_INDEX, APP)
                            .where(__.out(MetadataRelationType.HAS_BO.name())
                                    .has(LABEL_INDEX, BO).has(ID_INDEX, boId)));
                    Map<String, Object> appRealMap = appMap.get();
                    BoNode boNode = new BoNode(finalOne.get());
                    return new LazyEntityClass(metadataEngine, appRealMap.get(CODE_INDEX).toString(), boNode, profile);
                });
            }
        });
    }

    /**
     * TODO
     *
     * @param boCode
     * @param profile
     * @return
     */
    private Optional<IEntityClass> loadByCodeAndProfileInner(String boCode, String profile) {
        return metadataEngine.read(() -> {
            List<Map<String, Object>> multi = metadataEngine.getMulti(__.has(LABEL_INDEX, BO)
                    .has(CODE_INDEX, caseInsensitiveEq(boCode))
                    .where(has(PROFILE_INDEX, caseInsensitiveEq(profile)).or().hasNot(PROFILE_INDEX)));

            if(multi.isEmpty()) {
                return Optional.empty();
            } else {
                Optional<Map<String, Object>> boNodeMapOp = multi.stream().filter(x -> x.get(PROFILE_INDEX) != null).findFirst();
                Optional<Map<String, Object>> nullProfileOp = multi.stream().filter(x -> x.get(PROFILE_INDEX) == null).findFirst();
                Optional<Map<String, Object>> finalOne = boNodeMapOp.isPresent() ? boNodeMapOp : nullProfileOp;
                return finalOne.map(map -> {
                    Optional<Map<String, Object>> appMap = metadataEngine.get(__.has(LABEL_INDEX, APP)
                            .where(__.out(MetadataRelationType.HAS_BO.name())
                                    .has(LABEL_INDEX, BO).has(CODE_INDEX, caseInsensitiveEq(boCode))));
                    Map<String, Object> appRealMap = appMap.get();
                    BoNode boNode = new BoNode(finalOne.get());
                    return new LazyEntityClass(metadataEngine, appRealMap.get(CODE_INDEX).toString(), boNode, profile);
                });
            }
        });
    }

    /**
     * TODO
     *
     * @return
     */
    @WithSpan
    @Override
    public List<String> codes() {
        if (codes != null) {
            return codes;
        } else {
            List<Map<String, Object>> code = metadataEngine
                    .getMulti(__.has(LABEL_INDEX, BO));
            codes = code.stream().map(x -> x.get(CODE_INDEX).toString())
                    .distinct()
                    .collect(Collectors.toList());
            return codes;
        }
    }

    @Override
    public String appCode() {
        Optional<Map<String, Object>> stringObjectMap = metadataEngine.get(__.has(LABEL_INDEX, APP));
        if(stringObjectMap.isPresent()) {
            return stringObjectMap.get().get("code").toString();
        } else {
            return "";
        }
    }

    @WithSpan
    @Override
    public Optional<IEntityClass> load(String boId, String profile) {
        return loadByUnionKey(new UnionKey(null, Long.parseLong(boId), null, profile));
    }

    @WithSpan
    @Override
    public Optional<IEntityClass> load(String boId, String profile, String version) {
        return loadByUnionKey(new UnionKey(null, Long.parseLong(boId), version, profile));
    }

    @WithSpan
    @Override
    public Optional<IEntityClass> loadByCode(String boCode, String profile) {
        return loadByUnionKey(new UnionKey(boCode, null, null, profile));
    }

    private boolean isEquals(UnionKey a, UnionKey b) {
        if (a != null && b != null) {
            return (
                    ((a.code != null || b.code != null) && Objects.equals(a.code, b.code)) ||
                            ((a.id != null || b.id != null) && Objects.equals(a.id, b.id))) &&
                    Objects.equals(a.version, b.version) &&
                    Objects.equals(a.profile, b.profile);
        }

        return a == null && b == null;
    }

    private Optional<IEntityClass> loadByUnionKey(UnionKey callKey) {

//        Span span = tracer.activeSpan();
//        Span loadStructureSpan = tracer.buildSpan("load structure").asChildOf(span).start();

        Optional<IEntityClass> entityClass;

        UnionKey inMappingKey = mappingKey.get(callKey);

        if (inMappingKey == null) {
            //try
            Optional<UnionKey> inCacheKey = mappingKey.entrySet()
                    .stream()
                    .filter(x -> isEquals(x.getKey(), callKey))
                    .map(Map.Entry::getValue).findFirst();

            UnionKey finalKey = callKey;

            if (inCacheKey.isPresent()) {
                mappingKey.put(callKey, inCacheKey.get());
                finalKey = inCacheKey.get();
            }

            entityClass = Optional.ofNullable(l2Cache.get(finalKey));
        } else {
            entityClass = Optional.ofNullable(l2Cache.get(inMappingKey));
        }

        return entityClass;
    }

    @WithSpan
    @Override
    public Optional<IEntityClass> loadByCode(String boCode, String profile, String version) {
        return Optional.empty();
    }

    @WithSpan
    @Override
    public EntityClassGroup describe(IEntityClass entityClass, String profile) {
        return groupCache.get(new ProfiledEntityClass(profile, entityClass));
    }

    private EntityClassGroup describeInner(IEntityClass entityClass, String profile) {
        return new EntityClassGroupImpl(this, metadataEngine, entityClass, profile);
    }

    @WithSpan
    @Override
    public List<ColumnField> columns(ResourcePath resourcePath, String profile) {
        return columns(null, resourcePath, profile);
    }

    @Override
    public MetadataRepository getRepository() {
        return null;
    }

    @WithSpan
    @Override
    public List<ColumnField> columns(IEntityClass root, ResourcePath resourcePath, String profile) {
        if (!resourcePath.isMultiResource()) {
            log.warn("Cannot not find Multi resource on Single Path {}", resourcePath);
            return Collections.emptyList();
        }

        List<ColumnField> fields = columnsInner(root, resourcePath, profile);
        return fields;
    }

    /**
     * search related entityClass
     *
     * @param current
     * @param relatedCode
     * @return
     */
    @WithSpan
    public IEntityClass searchRelated(IEntityClass current, String relatedCode, String profile) {

        IEntityClass related = current.relations().stream().filter(x -> x.getRelOwnerClassId() == current.id())
                .filter(x -> x.getName().equals(relatedCode))
                .map(x -> load(String.valueOf(x.getEntityClassId()), profile)).filter(Optional::isPresent)
                .map(Optional::get).findFirst().orElse(null);

        if (related == null) {
            //search father
            IEntityClass father = null;
            IEntityClass ptr = current;
            while (ptr.extendEntityClass() != null && related == null) {
                father = ptr.extendEntityClass();
                Optional<IEntityClass> fatherRawOp = load(Long.toString(father.id()), profile);
                if (fatherRawOp.isPresent()) {
                    father = fatherRawOp.get();
                    IEntityClass finalFather = father;
                    related = father.relations().stream().filter(x -> x.getRelOwnerClassId() == finalFather.id())
                            .filter(x -> x.getName().equals(relatedCode))
                            .map(x -> load(String.valueOf(x.getEntityClassId()), profile)).filter(Optional::isPresent)
                            .map(Optional::get).findFirst().orElse(null);
                    ptr = father;
                } else {
                    log.warn("Father is not present while child says is father id:code {}:{}", father.id(), father.code());
                    break;
                }
            }
        }

        return related;
    }

    private List<ColumnField> columnsInner(IEntityClass root, ResourcePath resourcePath, String profile) {

        Iterator<ResourcePart> iterator = resourcePath.iterator();

        IEntityClass ptr = null;

        List<ColumnField> retList = new ArrayList<>();

        while (iterator.hasNext()) {

            ResourcePart nextPart = iterator.next();

            if (nextPart instanceof RelatedEntityClassResource) {
                String relatedCode = ((RelatedEntityClassResource) nextPart).getRelatedCode();
                if (ptr != null) {
                    ptr = searchRelated(ptr, relatedCode, profile);
                }

                if (ptr == null) {
                    break;
                }
            } else if (nextPart instanceof SubEntityClassResource) {
                /**
                 * make sure ptr is not null
                 */
                if (ptr == null) {
                    break;
                }

                String code = ((SubEntityClassResource) nextPart).getCode();
                ptr = searchSub(ptr, code, profile);

                if (ptr == null) {
                    break;
                }

            } else if (nextPart instanceof RootResourcePart) {
                if (root == null) {
                    //TODO root part maybe using injected? by code may cause ambiguous
                    ptr = loadByCode(((RootResourcePart) nextPart).getCode(), profile).orElse(null);
                    if (ptr == null) {
                        break;
                    }
                } else {
                    ptr = root;
                }
            } else if (nextPart instanceof MainFieldResource) {
                String fieldCode = ((MainFieldResource) nextPart).getFieldCode();
                columnRaw(ptr, fieldCode, profile).ifPresent(retList::add);
            } else if (nextPart instanceof FieldsResource) {
                retList.addAll(describe(ptr, profile).columns());
            }
        }

        return retList;
    }

    @WithSpan
    public IEntityClass searchSub(IEntityClass current, String subCode, String profile) {
        Collection<IEntityClass> subClassList = current.childEntityClasses();
        IEntityClass retEntityClass = null;
        if (subClassList != null) {
            for (IEntityClass entityClass : subClassList) {
                if (entityClass.code().equals(subCode)) {
                    retEntityClass = entityClass;
                    break;
                } else {
                    Optional<IEntityClass> sub = this.load(String.valueOf(entityClass.id()), profile);
                    if (sub.isPresent()) {
                        retEntityClass = searchSub(sub.get(), subCode, profile);
                    }
                }
            }
        }

        return retEntityClass;
    }

    private Optional<ColumnField> columnRaw(IEntityClass targetClass, String rawPart, String profile) {
        if (rawPart.contains(ResourcePath.Parser.SUB)) {
            String[] sub = rawPart.split(ResourcePath.Parser.SUB);
            String subCode = sub[0];
            String subField = sub[1];
            //search until find the sub
            IEntityClass subEntityClass = searchSub(targetClass, subCode, profile);
            if (subEntityClass != null) {
                Optional<IEntityField> subFieldOp = subEntityClass.field(subField);
                return subFieldOp.map(x -> new ColumnField(x.name(), x, subEntityClass));
            }
        } else {
            Optional<IEntityField> fieldOp = targetClass.field(rawPart);
            if (!fieldOp.isPresent() && targetClass.extendEntityClass() != null) {
                Optional<IEntityClass> parent = load(Long.valueOf(targetClass.extendEntityClass().id()).toString(), profile);
                if (parent.isPresent()) {
                    return columnRaw(parent.get(), rawPart, profile);
                } else {
                    throw new RuntimeException("Parent " + targetClass.extendEntityClass().code() + " not found");
                }
            } else {
                return fieldOp.map(x -> new ColumnField(x.name(), x, targetClass));
            }
        }

        return Optional.empty();
    }

    @Override
    public Optional<ColumnField> column(ResourcePath resourcePath, String profile) {
        return column(null, resourcePath, profile);
    }

    @WithSpan
    @Override
    public Optional<ColumnField> column(IEntityClass root, ResourcePath resourcePath, String profile) {
        if (resourcePath.isMultiResource()) {
            log.warn("Cannot not find Single resource on Multi Path {}", resourcePath);
            return Optional.empty();
        }

        List<ColumnField> fields = columnsInner(root, resourcePath, profile);

        if (!fields.isEmpty()) {
            return Optional.ofNullable(fields.get(0));
        } else {
            return Optional.empty();
        }
    }

    @WithSpan
    @Override
    public List<IEntityClass> findAllEntities(String profile) {
        return metadataEngine.read(() -> {
            return codes().stream().map(x -> loadByCode(x, profile))
                    .filter(Optional::isPresent)
                    .map(Optional::get).collect(Collectors.toList());
        });
    }

    /**
     * TODO
     *
     * @param id
     * @return
     */
    @Override
    public Set<String> findCustomActionsById(long id) {
        return Collections.emptySet();
    }

    @Override
    public List<BoApiVo> loadApiByCode(String code, String profile) {
        List<Map<String, Object>> multi = metadataEngine.getMulti(
                __.has(CODE_INDEX, caseInsensitiveEq(code)).outE(HAS_API.name()).inV());

        return multi.stream().map(x -> {
            BoApiVo boApiVo = new BoApiVo();
            boApiVo.setCode(String.valueOf(x.get("code")));
            boApiVo.setBoId(Long.parseLong(String.valueOf(x.get("boId"))));
            boApiVo.setMethod(String.valueOf(x.get("method")));
            boApiVo.setUrl(String.valueOf(x.get("url")));
            boApiVo.setName(String.valueOf(x.get("name")));
            boApiVo.setParam(String.valueOf(x.get("params")));
            return boApiVo;
        }).collect(Collectors.toList());
    }

    static class UnionKey {

        private String code;

        private Long id;

        private String version;

        private String profile;

        public UnionKey(String code, Long id, String version, String profile) {
            this.code = code;
            this.id = id;
            this.version = version;
            this.profile = profile;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            UnionKey unionKey = (UnionKey) o;
            return Objects.equals(code, unionKey.code) &&
                    Objects.equals(id, unionKey.id) &&
                    Objects.equals(profile, unionKey.profile) &&
                    Objects.equals(version, unionKey.version);
        }

        @Override
        public int hashCode() {
            return Objects.hash(code, id, version, profile);
        }

        @Override
        public String toString() {
            return "UnionKey{" +
                    "code='" + code + '\'' +
                    ", id=" + id +
                    ", version='" + version + '\'' +
                    ", profile='" + profile + '\'' +
                    '}';
        }
    }
}
