package com.xforceplus.ultraman.metadata.values;

import com.xforceplus.ultraman.metadata.entity.IEntityField;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 表示日期/时间值,储存时时间使用 Asia/Shanghai.
 *
 * @author wangzheng dongbin
 * @version 0.1 2020/2/18 20:54
 * @since 1.8
 */
public class DateTimeValue extends AbstractValue<LocalDateTime> {

    private final Logger logger = LoggerFactory.getLogger(DateTimeValue.class);

    /**
     * 格式化时使用的时区.
     */
    public static final ZoneId ZONE_ID = ZoneId.of("Asia/Shanghai");

    /**
     * 最大值.
     */
    public static final LocalDateTime MIN_DATE_TIME = LocalDateTime.of(LocalDate.of(0, 1, 1), LocalTime.MIN);

    /**
     * 最小值.
     */
    public static final LocalDateTime MAX_DATE_TIME = LocalDateTime.of(LocalDate.of(9999, 1, 1), LocalTime.MAX);

    public DateTimeValue(IEntityField field, LocalDateTime value) {
        super(field, value);
    }

    public DateTimeValue(IEntityField field, LocalDateTime value, Attachment attachment) {
        super(field, value, attachment);
    }

    public DateTimeValue(IEntityField field, LocalDateTime value, String attachemnt) {
        super(field, value, attachemnt);
    }

    @Override
    LocalDateTime fromString(String value) {
        if (value == null) {
            return null;
        }
        try {
            long timestamp = Long.parseLong(value);

            return toLocalDateTime(timestamp);
        } catch (Exception ex) {
            logger.error("{}", ex);
            return null;
        }
    }

    @Override
    public long valueToLong() {
        Instant instant = getValue().atZone(ZONE_ID).toInstant();
        return instant.toEpochMilli();
    }

    @Override
    protected IValue<LocalDateTime> doCopy(IEntityField newField, String attachmentKey, String attachmentValue) {
        Attachment attachment = this.getAttachment().copy();
        attachment.add(attachmentKey, attachmentValue);
        return new DateTimeValue(newField, getValue(), attachment);
    }

    @Override
    protected IValue<LocalDateTime> doCopy(LocalDateTime value) {
        return new DateTimeValue(getField(), value, getAttachment().copy());
    }

    @Override
    public String valueToString() {
        return Long.toString(valueToLong());
    }

    @Override
    public boolean compareByString() {
        return false;
    }

    @Override
    public Long storageValue() {
        return valueToLong();
    }

    /**
     * 默认的timestamp转LocalDateTime.
     */
    public static LocalDateTime toLocalDateTime(long timestamp) {
        return LocalDateTime.ofInstant(Instant.ofEpochMilli(timestamp), ZONE_ID);
    }
}
