package com.xforceplus.ultraman.metadata.helper;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.xforceplus.ultraman.metadata.cdc.OqsEngineEntity;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.entity.EntityClassRef;
import com.xforceplus.ultraman.metadata.entity.FieldType;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.SystemColumn;
import com.xforceplus.ultraman.sdk.infra.utils.JacksonDefaultMapper;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Created by justin.xu on 09/2023.
 *
 * @since 1.8
 */
public class OriginEntityUtils {
    private static final Logger logger = LoggerFactory.getLogger(OriginEntityUtils.class);

    public static OqsEngineEntity generateFromResultSet(ResultSet resultSet, IEntityClass tableEntityClass, EntityClassEngine engine)
            throws SQLException, JsonProcessingException {
        try {
            long entityClassId = resultSet.getLong(SystemColumn.SYS_ENTITY_CLASS);
            String profile = resultSet.getString(SystemColumn.SYS_PROFILE);

            IEntityClass iEntityClass = tableEntityClass;
            IEntityClass relationTableEntityClass = engine.load(String.valueOf(entityClassId), profile).get();
            List<IEntityClass> tableEntityClasses = relationTableEntityClass.getEntityTables().stream().collect(Collectors.toList());

            OqsEngineEntity.Builder builder = new OqsEngineEntity.Builder();
            builder.withEntityClassRef(new EntityClassRef(entityClassId, "", "", profile));
            builder.withRelationTables(tableEntityClass.id(), tableEntityClass.code());
            builder.withTableEntityClass(tableEntityClasses);

            if (entityClassId != tableEntityClass.id()) {
                builder.withFather(tableEntityClass.id());
            } else {
                IEntityClass ec = tableEntityClass.extendEntityClass();
                if (null != ec) {
                    builder.withFather(ec.id());
                } else {
                    builder.withFather(0);
                }
            }

            /**取出元数据**/
            Map<String, IEntityField> iEntityFieldMap = new HashMap<>();
            engine.describe(iEntityClass, profile).getAllFields().stream().forEach(p -> iEntityFieldMap.put(p.name().replace(".", "_"), p));

            ResultSetMetaData resultSetMetaData = resultSet.getMetaData();
            int rowCount = resultSetMetaData.getColumnCount();

            for (int index = 1; index <= rowCount; index++) {
                String columnName = resultSetMetaData.getColumnName(index);
                if (columnName.equals(SystemColumn.ID)) {
                    long id = resultSet.getLong(SystemColumn.ID);
                    builder.withId(id);
                    builder.withAttribute("id", id);
                }
                if (columnName.equals(SystemColumn.SYS_OPERATE_TIME)) {
                    //  update time.
                    long updateTime = resultSet.getLong(index);
                    builder.withUpdateTime(updateTime);
                } else if (columnName.equals(SystemColumn.SYS_VERSION)) {
                    //  update version.
                    builder.withVersion(resultSet.getInt(index));
                } else if (columnName.equals(SystemColumn.SYS_DELETED)) {
                    builder.withDeleted(resultSet.getLong(SystemColumn.SYS_DELETED) != 0);
                } else if (
                        columnName.equals(SystemColumn.SYS_ENTITY_CLASS) ||
                        columnName.equals(SystemColumn.SYS_PROFILE)) {
                    //  do nothing
                } else if (columnName.equals(SystemColumn.DYNAMIC_FIELD)) {
                    builder.withAttributes(
                            parserDynamic(iEntityClass, resultSet.getString(SystemColumn.DYNAMIC_FIELD), !StringUtils.isEmpty(tableEntityClass.realProfile())));
                } else {
                    IEntityField field = iEntityFieldMap.get(columnName);
                    if (null == field) {
                        logger.warn("entityField can not be null, column name {}", columnName);
                        continue;
                    }

                    formatFiledValues(builder, field, columnName, resultSet, index);
                }
            }

            OqsEngineEntity engineEntity = builder.build();
            engineEntity.updateIntact();

            return engineEntity;
        } catch (Exception e) {
            throw e;
        }
    }

    /**
     * 根据字段类型格式化oqsEngineEntity字段值
     *
     * @param builder
     * @param field
     * @param name
     **/
    public static void formatFiledValues(OqsEngineEntity.Builder builder, IEntityField field, String name, ResultSet resultSet, int index) throws SQLException {
        Object r = resultSet.getObject(index);
        switch (field.type()) {
            case LONG:
            case DATETIME:
                builder.withAttribute(name, null != r ? resultSet.getLong(index) : null);
                break;
            case BOOLEAN:
                builder.withAttribute(name, null != r ? resultSet.getLong(index) != 0 : null);
                break;
            case DECIMAL:
                builder.withAttribute(name, null != r ? resultSet.getBigDecimal(index)  : null);
                break;
            case STRING:
                builder.withAttribute(name, null != r ? resultSet.getString(index) : null);
                break;
            case STRINGS:
                try {
                    List<String> multiValues = null;
                    if (null != r) {
                        multiValues =
                                JacksonDefaultMapper.OBJECT_MAPPER.readValue(resultSet.getString(index),
                                        JacksonDefaultMapper.LIST_TYPE_REFERENCE);
                    } else {
                        multiValues = Collections.emptyList();
                    }
                    builder.withAttribute(name, multiValues);
                } catch (Throwable throwable) {
                    try {
                        Object o = resultSet.getObject(index);
                        builder.withAttribute(name, o);
                    } catch (Throwable t) {
                        //  ignore...
                    }
                }
                break;
            default:
                builder.withAttribute(name, resultSet.getObject(index));
                break;
        }
    }


    /**
     * 根据字段类型格式化oqsEngineEntity字段值
     *
     * @param builder
     * @param field
     * @param name
     * @param value
     **/
    public static void formatFiledValues(OqsEngineEntity.Builder builder, IEntityField field, String name, String value) {
        switch (field.type()) {
            case LONG:
            case DATETIME:
                builder.withAttribute(name, Long.parseLong(value));
                break;
            case BOOLEAN:
                long result = Long.parseLong(value);
                builder.withAttribute(name, result != 0);
                break;
            case DECIMAL:
                builder.withAttribute(name, new BigDecimal(value));
                break;
            case STRINGS:
            case ENUMS:    
                try {
                    List<String> multiValues = JacksonDefaultMapper.OBJECT_MAPPER.readValue(value, JacksonDefaultMapper.LIST_TYPE_REFERENCE);
                    builder.withAttribute(name, multiValues);
                } catch (Throwable throwable) {
                    builder.withAttribute(name, value);
                }
                break;
            default:
                builder.withAttribute(name, value);
                break;
        }
    }

    /**
     * 转换attribute.
     *
     * @return 对象键值对.
     */
    public static Map<String, Object> parserDynamic(IEntityClass entityClass, String attrStr, boolean isProfiled) throws JsonProcessingException {

        Map<String, Object> result;
        if (attrStr.isEmpty()) {
            result = new HashMap<>();
        } else {
            result = attributesToMap(attrStr);
        }

        Collection<IEntityField> targetFields;

        if (isProfiled) {
            targetFields = entityClass.selfFields().stream().filter(x -> !StringUtils.isEmpty(x.realProfile())).collect(Collectors.toList());
        } else {
            targetFields = entityClass.selfFields();
        }

        targetFields.stream().filter(IEntityField::isDynamic).forEach(
                f -> {
                    if (!result.containsKey(f.name())) {
                        result.put(f.name(), null);
                    } else if (f.type().equals(FieldType.BOOLEAN)) {
                        result.computeIfPresent(
                                f.name(),
                                (k, s) -> ((int) s) != 0
                        );

                    }
                }
        );
        return result;
    }

    /**
     * 属性字符串表示解析为实际对象列表.
     *
     * @param attrStr 属性的字符串表示.
     * @return 解析结果.
     * @throws JsonProcessingException JSON解析失败.
     */
    public static Map<String, Object> attributesToMap(String attrStr) throws JsonProcessingException {
        return JacksonDefaultMapper.OBJECT_MAPPER.readValue(attrStr, Map.class);
    }
}
