package com.xforceplus.ultraman.metadata.entity.legacy.impl;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.xforceplus.metadata.schema.typed.BoIndex;
import com.xforceplus.ultraman.metadata.entity.*;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 一个元信息定义,OQS内部使用的对象元信息定义.
 *
 * @author xujia 2021/2/18
 * @since 1.8
 */
public class EntityClass implements IEntityClass {

  /*
   * 元数据boId
   */
  @JsonProperty(value = "id")
  private long id;

  private int ver;

  /*
   * 对象名称
   */
  @JsonProperty(value = "name")
  private String name;

  /*
   * 对象code
   */
  @JsonProperty(value = "code")
  private String code;

  /*
   * 所属于的应用code.
   */
  @JsonProperty(value = "appCode")
  private String appCode;

  /*
   * 元数据版本.
   */
  @JsonProperty(value = "version")
  private int version;

  /*
   * 元信息处于的继承层级
   */
  @JsonProperty(value = "level")
  private int level;

  /*
   * profile信息
   */
  @JsonProperty(value = "profile")
  private String profile;

  /*
   * 关系信息
   */
  @JsonProperty(value = "relations")
  private Collection<IRelation> relations;

  /*
   * 继承的对象类型.
   */
  @JsonProperty(value = "father")
  private IEntityClass father;

  /*
   * 对象属性信息
   */
  @JsonProperty(value = "fields")
  private Collection<IEntityField> fields = Collections.emptyList();

  /*
   * entityClass的类型, static/dynamic
   */
  @JsonProperty(value = "type")
  private EntityClassType type;

  @JsonProperty(value = "indexes")
  private Collection<BoIndex> indexes = Collections.emptyList();

  @Override
  public long id() {
    return id;
  }

  @Override
  public void setType(int type) {

  }

  @Override
  public String code() {
    return code;
  }

  @Override
  public int ver() {
    return 0;
  }

  @Override
  public String appCode() {
    return this.appCode;
  }

  @Override
  public String profile() {
    return this.profile;
  }

  @Override
  public String realProfile() {
    return this.profile;
  }

  @Override
  public String name() {
    return name;
  }

  @Override
  public Collection<IRelation> relations() {
    return null;
  }

  @Override
  public Collection<String> getEntityTables() {
    return new ArrayList<>();
  }

  @Override
  public Collection<IEntityClass> entityClasses() {
    return null;
  }

  @Override
  public Set<String> actions() {
    return null;
  }

  @Override
  public Collection<BoIndex> indexes() {
    return indexes;
  }

  @Override
  public Collection<BoIndex> uniqueIndexes() {
    return indexes.stream().filter(BoIndex::isUnique).collect(Collectors.toList());
  }

  @Override
  public Collection<IEntityClass> childEntityClasses() {
    return null;
  }

  @Override
  public IEntityClass extendEntityClass() {
    return null;
  }

  @Override
  public int version() {
    return version;
  }

  @Override
  public int level() {
    return level;
  }

  /**
   * actions
   */
  private Set<String> actions = new HashSet<>();

  /**
   * 子对象结构信息
   */
  private Set<IEntityClass> entityClasses;

  /**
   * 直接一级继承者对象.
   */
  private Set<IEntityClass> childEntityClasses = new HashSet<>();

  private IEntityClass extendEntityClass;

  private EntityClass() {

  }

  @Deprecated
  public EntityClass(Long id,
      String code,
      String name,
      Collection<IRelation> relations,
      Collection<IEntityClass> entityClasses,
      IEntityClass extendEntityClass,
      Collection<IEntityField> fields) {
    this(id, code, relations, entityClasses, extendEntityClass, fields);
    this.name = name;
  }

  /**
   * 构造一个新的entity 类型信息.
   *
   * @param id                类型 id.
   * @param code              类型 code.
   * @param relations         关联对象信息.
   * @param entityClasses     类型关联对象类型信息.
   * @param extendEntityClass 继承对象信息.
   * @param fields            属性列表.
   */
  public EntityClass(Long id,
      int ver,
      String code,
      String name,
      Collection<IRelation> relations,
      Collection<IEntityClass> entityClasses,
      IEntityClass extendEntityClass,
      Collection<IEntityField> fields) {
    this(id, ver, code, relations, entityClasses, extendEntityClass, Collections.emptyList(), fields);
    this.name = name;
  }

  /**
   * 构造一个新的entity 类型信息.
   *
   * @param id                类型 id.
   * @param code              类型 code.
   * @param relations         关联对象信息.
   * @param entityClasses     类型关联对象类型信息.
   * @param extendEntityClass 继承对象信息.
   * @param fields            属性列表.
   */
  @Deprecated
  public EntityClass(long id,
      String code,
      Collection<IRelation> relations,
      Collection<IEntityClass> entityClasses,
      IEntityClass extendEntityClass,
      Collection<IEntityField> fields) {
    this(id, 0, code, relations, entityClasses, extendEntityClass, Collections.emptyList(), fields);

  }

  public EntityClass(long id,
      int ver,
      String code,
      Collection<IRelation> relations,
      Collection<IEntityClass> entityClasses,
      IEntityClass extendEntityClass,
      Collection<IEntityClass> simpleSubEntityClasses,
      Collection<IEntityField> fields) {
    this.id = id;
    this.ver = ver;
    if (this.ver < 0) {
      throw new IllegalArgumentException("The version of Entity Class is invalid and cannot be less than 0.");
    }
    this.code = code;
    if (relations == null) {
      this.relations = Collections.emptyList();
    } else {
      this.relations = new ArrayList<>(relations);
    }
    if (entityClasses == null) {
      this.entityClasses = Collections.emptySet();
    } else {
      this.entityClasses = new HashSet<>(entityClasses);
    }
    if (fields == null) {
      this.fields = Collections.emptyList();
    } else {
      this.fields = new ArrayList<>(fields);
    }

    this.extendEntityClass = extendEntityClass;
    this.childEntityClasses.addAll(simpleSubEntityClasses);
  }


  @Override
  public EntityClassRef ref() {
    return EntityClassRef.Builder.anEntityClassRef()
        .withEntityClassId(id)
        .withEntityClassAppCode(appCode)
        .withEntityClassCode(code)
        .withEntityClassProfile(profile)
        .build();
  }

  @Override
  public IEntityClass root() {
    return null;
  }

  @Override
  public EntityClassType type() {
    return type;
  }

  //TODO
  @Override
  public String masterQueryTable() {
    return "oqs_".concat(appCode).concat("_").concat(code());
  }

  //TODO
  @Override
  public String masterWriteTable(boolean withProfile) {
    return "oqs_".concat(appCode).concat("_").concat(code());
  }

  @Override
  public void resetChildEntityClass(Collection<IEntityClass> childEntityClasses) {

  }

  @Override
  public String indexQueryTable() {
    return null;
  }

  @Override
  public boolean isDynamic() {
    return type == null || type.equals(EntityClassType.DYNAMIC);
  }

  @Override
  public Optional<IEntityClass> father() {
    return Optional.ofNullable(father);
  }

  @Override
  public Collection<IEntityClass> family() {
    List<IEntityClass> familyList = new ArrayList<>(level);
    Optional<IEntityClass> current = Optional.of(this);
    while (current.isPresent()) {
      familyList.add(0, current.get());
      current = current.get().father();
    }

    return familyList;
  }

  @Override
  public Collection<IEntityField> selfWithUnique() {
    List<IEntityField> all = new ArrayList<>(selfFields());

    for (IEntityField entityField : uniqueFieldCollection()) {
      if (!all.contains(entityField)) {
        all.add(entityField);
      }
    }

    return all;
  }

  @Override
  public Collection<IEntityField> uniqueFieldCollection() {
    Map<String, IEntityField> ef = new HashMap<>();
    for (BoIndex index : uniqueIndexes()) {
      String fieldIds = index.getFieldIds();
      String[] fields = fieldIds.split(",");
      for (String name : fields) {
        field(name).ifPresent(entityField -> ef.put(name, entityField));
      }
    }

    return new ArrayList<>(ef.values());
  }

  @Override
  public Collection<IEntityField> selfWithIndex() {
    List<IEntityField> all = new ArrayList<>(selfFields());

    for (IEntityField entityField : indexFieldCollection()) {
      if (!all.contains(entityField)) {
        all.add(entityField);
      }
    }

    return all;
  }

  @Override
  public Collection<IEntityField> indexFieldCollection() {
    Map<String, IEntityField> ef = new HashMap<>();
    for (BoIndex index : indexes()) {
      for (String name : index.getFieldIds().split(",")) {
        field(name).ifPresent(entityField -> ef.put(name, entityField));
      }
    }

    return new ArrayList<>(ef.values());
  }

  @Override
  public Collection<IEntityField> selfFields() {
    return fields;
  }

  @Override
  public Collection<IEntityField> fields() {
    //  获取自己 + 父类的所有IEntityField
    List<IEntityField> entityFields = new ArrayList<>(fields);

    if (null != relations) {
      relations.forEach(
          r -> {
            if (null != r && r.isSelfRelation(id)) {
              if (r.getEntityField() != null) {
                entityFields.add(r.getEntityField());
              }
            }
          }
      );
    }

    if (null != father) {
      entityFields.addAll(father.fields());
    }

    return entityFields;
  }

  @Override
  public Collection<IEntityField> withoutRelationFields() {
    //  获取自己 + 父类的所有IEntityField
    List<IEntityField> entityFields = new ArrayList<>(fields);
    if (null != father) {
      entityFields.addAll(father.withoutRelationFields());
    }

    return entityFields;
  }

  @Override
  public Collection<IEntityField> selfWithoutRelationFields() {
    //  获取自己 + 父类的所有IEntityField
    return new ArrayList<>(fields);
  }

  @Override
  public Optional<IEntityField> field(String name) {
    Optional<IEntityField> entityFieldOp =
        fields.stream().filter(f -> name.equals(f.name())).findFirst();

    //  找到
    if (entityFieldOp.isPresent()) {
      return entityFieldOp;
    } else {
      if (relations != null) {
        //  从关系中找
        for (IRelation relation : relations) {
          if (relation.isSelfRelation(this.id)) {
            if (relation.getEntityField() != null && relation.getEntityField().name().equals(name)) {
              return Optional.of(relation.getEntityField());
            }
          }
        }
      }
    }

    //  从父类找
    if (null != father) {
      return father.field(name);
    }
    return entityFieldOp;
  }

  @Override
  public Optional<IEntityField> field(long id) {
    Optional<IEntityField> entityFieldOp =
        fields.stream().filter(f -> id == f.id()).findFirst();

    if (entityFieldOp.isPresent()) {
      return entityFieldOp;
    } else {
      if (relations != null) {
        //  从关系中找
        for (IRelation relation : relations) {
          if (relation.isSelfRelation(this.id)) {
            if (relation.getEntityField() != null && relation.getEntityField().id() == id) {
              return Optional.of(relation.getEntityField());
            }
          }
        }
      }
    }

    //  从父类找
    if (null != father) {
      return father.field(id);
    }
    return entityFieldOp;
  }

  @Override
  public boolean isAny() {
    return false;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof EntityClass)) {
      return false;
    }
    EntityClass that = (EntityClass) o;
    return id == that.id
        && version == that.version
        && level == that.level
        && Objects.equals(profile, that.profile)
        && Objects.equals(name, that.name)
        && Objects.equals(code, that.code)
        && Objects.equals(appCode, that.appCode)
        && Objects.equals(father, that.father)
        && Objects.equals(relations, that.relations)
        && Objects.equals(fields, that.fields);
  }

  @Override
  public int hashCode() {
    return Objects.hash(id, name, code, version, level, relations, father, fields);
  }

  @Override
  public String toString() {
    final StringBuffer sb = new StringBuffer("EntityClass{");
    sb.append("id=").append(id);
    sb.append(", name='").append(name).append('\'');
    sb.append(", code='").append(code).append('\'');
    sb.append(", profile='").append(profile).append('\'');
    sb.append(", appCode='").append(appCode).append('\'');
    sb.append(", version=").append(version);
    sb.append(", level=").append(level);
    sb.append(", relations=").append(relations);
    sb.append(", father=").append(father);
    sb.append(", fields=").append(fields);
    sb.append('}');
    return sb.toString();
  }

  /**
   * Builder.
   */
  public static final class Builder {

    private long id;
    private String name;
    private String code;
    private String appCode;
    private int version;
    private int level;
    private String profile;
    private Collection<IRelation> relations = Collections.emptyList();
    private IEntityClass father;
    private Collection<IEntityField> fields = Collections.emptyList();
    private EntityClassType type;
    private List<BoIndex> indexes;

    private Builder() {
    }

    public static Builder anEntityClass() {
      return new Builder();
    }


    public Builder withId(long id) {
      this.id = id;
      return this;
    }

    public Builder withProfile(String profile) {
      this.profile = profile;
      return this;
    }

    public Builder withName(String name) {
      this.name = name;
      return this;
    }

    public Builder withCode(String code) {
      this.code = code;
      return this;
    }

    public Builder withAppCode(String appCode) {
      this.appCode = appCode;
      return this;
    }

    public Builder withVersion(int version) {
      this.version = version;
      return this;
    }


    public Builder withIndex(List<BoIndex> indexes) {
      this.indexes = indexes;
      return this;
    }


    public Builder withLevel(int level) {
      this.level = level;
      return this;
    }

    public Builder withRelations(Collection<IRelation> relations) {
      this.relations = relations;
      return this;
    }

    public Builder withFather(IEntityClass father) {
      this.father = father;
      return this;
    }

    public Builder withFields(Collection<IEntityField> fields) {
      this.fields = fields;
      return this;
    }

    public Builder withType(EntityClassType type) {
      this.type = type;
      return this;
    }

    /**
     * 增加新的字段.
     *
     * @param field 目标字段.
     * @return 当前构造器.
     */
    public Builder withField(IEntityField field) {
      if (Collections.emptyList().getClass().equals(this.fields.getClass())) {
        this.fields = new ArrayList<>(fields);
      }

      this.fields.add(field);

      return this;
    }

    /**
     * 构造一个OqsEntityClass 实例.
     *
     * @return 实例.
     */
    public EntityClass build() {
      EntityClass entityClass = new EntityClass();
      entityClass.id = this.id;
      entityClass.code = this.code;
      entityClass.appCode = this.appCode;
      entityClass.name = this.name;
      entityClass.level = this.level;
      entityClass.version = this.version;
      entityClass.father = father;
      entityClass.fields = fields;
      entityClass.relations = this.relations;
      entityClass.type = this.type;
      if (this.profile == null) {
        entityClass.profile = OqsProfile.UN_DEFINE_PROFILE;
      } else {
        entityClass.profile = this.profile;
      }
      entityClass.indexes = (this.indexes == null) ? new HashSet<>() : this.indexes;
      return entityClass;
    }
  }
}
