package com.xforceplus.ultraman.metadata.engine;

import com.xforceplus.ultraman.metadata.domain.record.GeneralRecord;
import com.xforceplus.ultraman.metadata.domain.record.Record;
import com.xforceplus.ultraman.metadata.engine.dsl.ResourcePath;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.metadata.entity.IRelation;
import com.xforceplus.ultraman.metadata.entity.legacy.impl.ColumnField;
import com.xforceplus.ultraman.metadata.entity.legacy.impl.Relation;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * entity-class group
 */
@Slf4j
public class DefaultEntityClassGroup implements EntityClassGroup {

    /**
     * to load more related
     */
    private EntityClassEngine engine;

    /**
     * main entityClass
     */
    private IEntityClass mainEntityClass;

    /**
     * father entityClass
     */
    private List<IEntityClass> fatherEntityClass;

    private List<IEntityClass> childrenEntityClass;

    private String profile;

    public DefaultEntityClassGroup(EntityClassEngine engine, IEntityClass mainEntityClass, List<IEntityClass> fatherEntityClass, List<IEntityClass> childrenEntityClass, String profile) {
        this.engine = engine;
        this.mainEntityClass = mainEntityClass;
        this.fatherEntityClass = fatherEntityClass;
        this.childrenEntityClass = childrenEntityClass;
        this.profile = profile;
    }

//    /**
//     * TODO
//     * zip Value only zip self not consider the children
//     *
//     * @param body
//     * @return
//     */
//    public Stream<Tuple2<IEntityField, Object>> zipValue(Map<String, Object> body, long id) {
//        this.testBody(body, id).forEach((x) -> {
//            log.warn("[{}] is not available in EntityClass [{}]", x, this.mainEntityClass.code());
//        });
//
//        return getAllFields().stream().map(x -> {
//            String key = x.name().toLowerCase();
//            return Tuple.of(x, body.get(key));
//        });
//    }

    public List<IEntityClass> getFatherEntityClass() {
        return Collections.unmodifiableList(fatherEntityClass);
    }

    public List<IEntityClass> getChildrenEntityClass() {
        return Collections.unmodifiableList(childrenEntityClass);
    }

    /**
     * find all keys
     * assume no same code in
     *
     * @return
     */
    private Set<String> getAllKeys() {
        return getAllFields().stream().map(IEntityField::name).collect(Collectors.toSet());
    }

    /**
     * get all relations
     *
     * @return
     */
    public List<IRelation> getAllRelations() {
        List<IRelation> allRelation = new LinkedList<>(mainEntityClass.relations());
        fatherEntityClass.stream().flatMap(x -> x.relations().stream()).forEach(allRelation::add);
        return allRelation;
    }

    /**
     * as all field should have completed fields
     *
     * @return
     */
    public List<IEntityField> getAllFields() {
        List<IEntityField> allFields = new LinkedList<>(mainEntityClass.fields());
        fatherEntityClass.stream().flatMap(x -> x.fields().stream()).forEach(allFields::add);
        //add related
        return allFields;
    }

    public Optional<IEntityField> field(String rawCode) {
        Optional<IEntityField> field = mainEntityClass.field(rawCode);
        if (!field.isPresent()) {
            return fatherEntityClass.stream().map(x -> x.field(rawCode)).filter(Optional::isPresent).map(Optional::get).findFirst();
        } else {
            return field;
        }
    }

    public Optional<IEntityField> field(long fieldId) {
        Optional<IEntityField> field = mainEntityClass.field(fieldId);
        if (!field.isPresent()) {
            return fatherEntityClass.stream().map(x -> x.field(fieldId)).filter(Optional::isPresent).map(Optional::get).findFirst();
        } else {
            return field;
        }
    }

    public Optional<IRelation> relation(long relationId) {
        Optional<IRelation> first = mainEntityClass.relations().stream().filter(x -> x.getId() == relationId).findFirst();
        if (!first.isPresent()) {
            return fatherEntityClass.stream().flatMap(x -> x.relations().stream()).filter(x -> x.getId() == relationId).findFirst();
        } else {
            return first;
        }
    }

    public Optional<IRelation> relation(String relationCode) {
        Optional<IRelation> first = mainEntityClass.relations().stream().filter(x -> x.getName().equalsIgnoreCase(relationCode)).findFirst();
        if (!first.isPresent()) {
            return fatherEntityClass.stream().flatMap(x -> x.relations().stream()).filter(x -> x.getName().equalsIgnoreCase(relationCode)).findFirst();
        } else {
            return first;
        }
    }

    /**
     * find related entityClass by code
     *
     * @param relatedCode
     * @return
     */
    public Optional<IEntityClass> relatedEntityClass(String relatedCode) {
        //find from main to father
        Optional<IRelation> first = mainEntityClass.relations().stream().filter(x -> x.getRelOwnerClassId() == mainEntityClass.id()).filter(x -> x.getName().equals(relatedCode)).findFirst();

        if (first.isPresent()) {
            return engine.load(String.valueOf(first.get().getEntityClassId()), profile);
        } else {
            Optional<IRelation> relationInParent = fatherEntityClass.stream().map(x -> {
                return x.relations().stream()
                        .filter(rel -> rel.getRelOwnerClassId() == x.id())
                        .filter(rel -> rel.getName().equals(relatedCode)).findFirst();
            }).filter(Optional::isPresent).map(Optional::get).findFirst();

            return relationInParent.flatMap(x -> engine.load(String.valueOf(x.getEntityClassId()), profile));
        }
    }

    /**
     * test key
     *
     * @param map
     * @return
     */
    public Set<String> testBody(Map<String, Object> map, long targetId) {
        Set<String> inputKeys = map.keySet();
        Set<String> allKeys = getAllKeys();
        return (Set) inputKeys.stream().filter((x) -> {
            return !allKeys.contains(x);
        }).collect(Collectors.toSet());
    }

    public IEntityClass getEntityClass() {
        return mainEntityClass;
    }

    @Override
    public String profile() {
        return profile;
    }

    @Override
    public String realProfile() {
        return profile;
    }

    @Override
    public EntityClassGroup relatedEntityClassWithRawName(String rawName) {
        return null;
    }

    @Override
    public EntityClassEngine classEngine() {
        return engine;
    }

    /**
     * record only one record
     *
     * @param body
     * @param entityClassId
     * @return
     */
    public Record toRecordNew(Map<Tuple2<String, Long>, Object> body, long entityClassId) {

        IEntityClass targetClass;
        Collection<IEntityField> fieldsList = Collections.emptyList();
        if (mainEntityClass.id() == entityClassId || entityClassId == 0) {
            targetClass = mainEntityClass;
            fieldsList = getAllFields();
        } else {
            Optional<IEntityClass> targetOp = fatherEntityClass.stream().filter(x -> x.id() == entityClassId).findFirst();
            if (targetOp.isPresent()) {
                targetClass = targetOp.get();
                fieldsList = engine.describe(targetClass, profile).getAllFields();
            } else {
                targetOp = childrenEntityClass.stream().filter(x -> x.id() == entityClassId).findFirst();
                if (targetOp.isPresent()) {
                    targetClass = targetOp.get();
                    fieldsList = engine.describe(targetClass, profile).getAllFields();
                }
            }
        }

        //get related fields
        Record record = new GeneralRecord(fieldsList, 0);
        record.setTypeId(entityClassId);

        fieldsList.forEach((x) -> {
            record.set(x, body.get(Tuple.of(x.name(), x.id())));
        });

        Object id = body.get("id");
        if(id != null) {
            record.setId((Long)id);
        }

        return record;
    }

//    public Record toRecord(EntityUp up) {
//        Map<Tuple2<String, Long>, Object> retValue = up.getValuesList().stream()
//                .collect(Collectors.toMap(x -> {
//                            return Tuple.of(x.getName(), x.getFieldId());
//                        }
//                        , ValueUp::getValue));
//
//        long id = up.getId();
//
//        String profile = up.getProfile();
//
//        Record record = toRecordNew(retValue, id);
//
//        if (id > 0) {
//            record.setTypeId(id);
//        }
//
//        long objId = up.getObjId();
//        //reset record field "id" with objId
//        record.set("id", objId);
//        record.setId(objId);
//
//        return record;
//    }

    /**
     * a fieldCode total format
     * <p>
     * A self field  AB X.ID  A:XXX
     * A self sub field
     * A related field _AB.XXX
     * A related sub field _A
     * columne field
     *
     * @param fieldCode
     * @return
     */
    public Optional<ColumnField> column(String fieldCode) {
        return engine.column(this.getEntityClass(), ResourcePath.parse("@".concat(mainEntityClass.code()).concat("#").concat(fieldCode)), profile);
    }

    /**
     * support multi fields list
     *
     * @param fieldCode
     * @return
     */
    public List<ColumnField> columns(String fieldCode) {
        return engine.columns(this.getEntityClass(), ResourcePath.parse("@".concat(mainEntityClass.code()).concat("#").concat(fieldCode)), profile);
    }

    /**
     * return current group fields only from current entityClass to its fathers
     *
     * @return
     */
    public List<ColumnField> columns() {
        Stream<ColumnField> mainFields = mainEntityClass.fields().stream().map(x -> new ColumnField(x.name(), x, mainEntityClass));
        Stream<ColumnField> fatherFields = fatherEntityClass.stream().flatMap(x -> x.fields()
                .stream().map(field -> new ColumnField(x.name(), field, x)));

        return Stream.concat(mainFields, fatherFields).collect(Collectors.toList());
    }

    @Override
    public String getJoinTable(String tableName) {
        return tableName;
    }

    @Override
    public Collection<Tuple2<String, String>> getAllToOneReverseRelations() {
        return Collections.emptyList();
    }
}
