package com.xforceplus.ultraman.metadata.entity.legacy.impl;

import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.metadata.entity.IEntityValue;
import com.xforceplus.ultraman.metadata.entity.impl.EntityValue;
import com.xforceplus.ultraman.metadata.values.IValue;

import java.io.Serializable;
import java.util.*;
import java.util.function.Predicate;

/**
 * a mixed EntityValue can hold multi field from different entity
 */
public class MixedEntityValue implements IEntityValue, Cloneable, Serializable {

    /**
     * 数据id
     */
    private long id;

    /**
     * Entity的值集合
     */
    private Map<String, IValue> values;


    public MixedEntityValue(long id) {
        this.id = id;
    }

    /**
     * make sure entityValue has a complete field
     *
     * @param entityValue
     */
    public MixedEntityValue(IEntityValue entityValue) {
        this.id = entityValue.id();
        entityValue.values().forEach(this::addValue);
    }

    @Override
    public long id() {
        return id;
    }

    @Override
    public void resetId(long id) {
        this.id = id;
    }

    @Override
    public int size() {
        return 0;
    }

    @Override
    public Optional<IValue> getValue(IEntityField field) {
        return Optional.empty();
    }

    @Override
    public Optional<IValue> getValue(String fieldName) {
        return Optional.ofNullable(values.get(fieldName));
    }

    @Override
    public Optional<IValue> getValue(long fieldId) {
        throw new UnsupportedOperationException();
    }

    @Override
    public IEntityValue addValue(IValue value) {
        lazyInit();
        values.put(value.getField().name(), value);
        return this;
    }

    @Override
    public Collection<IValue> values() {
        if (values == null) {
            return Collections.emptyList();
        }
        return values.values();
    }

    @Override
    public IEntityValue addValues(Collection<IValue> values) {
        lazyInit();
        values.stream().forEach(v -> {
            this.values.put(v.getField().name(), v);

        });
        return this;
    }

    @Override
    public Optional<IValue> remove(IEntityField field) {
        lazyInit();
        return Optional.ofNullable(values.remove(field.name()));
    }

    @Override
    public void filter(Predicate<? super IValue> predicate) {
        values.entrySet().removeIf(entry -> !predicate.test(entry.getValue()));
    }

    @Override
    public IEntityValue clear() {
        if (values != null) {
            values.clear();
        }
        return this;
    }

    @Override
    public boolean isDirty() {
        return false;
    }

    @Override
    public void squeezeEmpty() {

    }

    @Override
    public Object clone() throws CloneNotSupportedException {
        EntityValue cloneValue = new EntityValue(id);
        cloneValue.addValues(values());
        return cloneValue;
    }

    private void lazyInit() {
        if (this.values == null) {
            // 这里为了保存顺序为加入顺序.
            this.values = new LinkedHashMap<>();
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof MixedEntityValue)) {
            return false;
        }
        MixedEntityValue that = (MixedEntityValue) o;
        return id == that.id &&
                Objects.equals(values, that.values);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, values);
    }

    @Override
    public String toString() {
        return "MixedEntityValue{" +
                "id=" + id +
                ", values=" + (values != null ? values.toString() : "NULL") +
                '}';
    }
}
