package com.xforceplus.ultraman.metadata.values;

import com.xforceplus.ultraman.metadata.entity.IEntityField;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 附件表示.<br>
 * 附件可以设定多个KEY-VALUE对,但是对于key是有限制的.<br>
 * KEY只允许使用 AttachmentConstraintHelper 中设定的KEY列表中的一个.<br>
 *
 * @author weikai
 * @version 0.1 2022/9/28 09:11
 * @see AttachmentConstraintHelper
 * @since 1.8
 */
public class Attachment {
    // 附件内容
    private Map<String, String> map;

    // 附件所属字段
    private final IEntityField entityField;

    /**
     * 构造器.
     *
     * @param entityField 附件所属字段.
     */
    public Attachment(IEntityField entityField) {
        map = new HashMap<>();
        this.entityField = entityField;
        if(entityField != null) {
            // 所有字段都会有一个固定附件,指标其字段id.
            map.put(AttachmentConstraintHelper.FIELD_ID_ATTACHMENT_KEY, String.valueOf(entityField.id()));
        }
    }

    private void setMap(Map<String, String> map) {
        this.map = map;
    }

    /**
     * 附件中增加kv.
     *
     * @param attachKey   附件key
     * @param attachValue 附件value
     */
    public void add(String attachKey, String attachValue) {
        if (attachKey == null || attachValue == null) {
            //ignore
            return;
        }

        map.put(attachKey, attachValue);
    }

    public Optional<String> getAttachmentValue(String attachKey) {
        return Optional.ofNullable(map.get(attachKey));
    }

    /**
     * 判断是否有自定义附件,除了默认附件.
     *
     * @return true 有, false 没有.
     */
    public boolean haveCustomizeValue() {
        for (String key : this.map.keySet()) {
            if (!AttachmentConstraintHelper.FIELD_ID_ATTACHMENT_KEY.equals(key)) {
                return true;
            }
        }
        return false;
    }

    public Iterator<Map.Entry<String, String>> iterator() {
        return map.entrySet().iterator();
    }

    /**
     * 将附件对象转为实际存储String.
     * key1_value1,key2_value2
     *
     * @return 实际存储String.
     */
    public String toAttachmentString() {
        return doAttachmentKeyValueStrings(true).stream().collect(Collectors.joining());
    }

    /**
     * 获得附件的字符串表示列表.
     *
     * @return 列表.
     */
    public List<String> toAttachmentKeyValueStrings() {
        return doAttachmentKeyValueStrings(false);
    }

    private List<String> doAttachmentKeyValueStrings(boolean close) {
        List<String> values = new ArrayList<>(map.size());
        for (Map.Entry<String, String> entry : map.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();

            values.add(buildAttachmentString(key, value, close));
        }

        return values;
    }

    private String buildAttachmentString(String key, String value, boolean close) {
        if (close) {
            return String.format(
                "%s%s%s%s%s",
                AttachmentConstraintHelper.LEFT_CLOSE,
                key,
                AttachmentConstraintHelper.SPLIT,
                value,
                AttachmentConstraintHelper.RIGHT_CLOSE);
        } else {
            return String.format(
                "%s%s%s",
                key,
                AttachmentConstraintHelper.SPLIT,
                value);
        }
    }

    /**
     * 附件字符串转换为附件对象.
     *
     * @param attachmentString 附件字符串.
     * @param entityField      附件所属字段.
     * @return 附件对象.
     */
    public static Attachment fromAttachmentString(String attachmentString, IEntityField entityField) {
        boolean valueArea = false;
        StringBuilder buff = new StringBuilder();
        Attachment attachment = new Attachment(entityField);
        for (char point : attachmentString.toCharArray()) {
            if (AttachmentConstraintHelper.LEFT_CLOSE == point) {
                valueArea = true;
                buff.delete(0, buff.length());
                continue;
            }

            if (AttachmentConstraintHelper.RIGHT_CLOSE == point) {
                valueArea = false;

                String[] kv = buff.toString().split(Character.toString(AttachmentConstraintHelper.SPLIT));
                if (kv.length != 2) {
                    throw new RuntimeException(
                        String.format("Attachment element: %s can not split to attachmentKey and attachmentValue.",
                            buff));
                }

                if (!kv[0].equals(AttachmentConstraintHelper.FIELD_ID_ATTACHMENT_KEY)) {
                    attachment.add(kv[0], kv[1]);
                }
                continue;
            }

            if (valueArea) {
                buff.append(point);
            }
        }

        return attachment;
    }


    /**
     * 附件copy.
     *
     * @return 新的附件.
     */
    public Attachment copy() {
        Attachment attachment = new Attachment(this.entityField);
        HashMap<String, String> map = new HashMap<>(this.map);
        attachment.setMap(map);
        return attachment;
    }


    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Attachment)) {
            return false;
        }
        Attachment that = (Attachment) o;
        return this.toAttachmentString().equals(that.toAttachmentString());
    }

    @Override
    public int hashCode() {
        String value = this.toAttachmentString();
        if (value == null) {
            return 0;
        } else {
            return Objects.hash(value);
        }
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("Attachment{");
        sb.append("entityField=").append(entityField);
        sb.append(", map=").append(map);
        sb.append('}');
        return sb.toString();
    }
}
