package com.xforceplus.ultraman.oqsengine.plus.common.pool;

import java.nio.ByteBuffer;

/**
 * 一个ByteBuffer的表.<br>
 * 池需要指定一个对齐字节数量.最小为4K.<br>
 * <pre>
 *     4K -> ByteBuffer
 *     8K -> ByteBuffer
 *    12K -> ByteBuffer
 *    .....
 *    nK  -> ByteBuffer
 * </pre>
 * 注意: 每一个大小的实例都只有唯一一个,同时只能有一个线程获取.
 *
 * @author dongbin
 * @version 0.1 2022/9/16 15:49
 * @since 1.8
 */
public class DirectByteBufferTable {

    private static final int MIN_BLOCK_SIZE = 4 * 1024;

    private static int DEFAULT_SIZE = 12;

    private static final DirectByteBufferTable INSTANCE = new DirectByteBufferTable();

    /*
    容量对齐大小.
     */
    private int alignmentBlockSize;
    private DirectByteBuff[] buffers;

    public DirectByteBufferTable() {
        this(MIN_BLOCK_SIZE);
    }

    /**
     * 构造一个新的ByteBufferPool实例.
     */
    public DirectByteBufferTable(int alignmentBlockSize) {
        /*
        由于读取时没有锁定,只对put进行了锁定.
        为了防止HashMap的 get 读取到 put 未完成的状态.
        所以这里使用线程安全的ConcurrentHashMap.
         */
        buffers = new DirectByteBuff[DEFAULT_SIZE];
        this.alignmentBlockSize = alignmentBlockSize;

        if (this.alignmentBlockSize < MIN_BLOCK_SIZE) {
            this.alignmentBlockSize = MIN_BLOCK_SIZE;
        }
    }

    /**
     * 获取实例.
     */
    public static DirectByteBufferTable getInstance() {
        return INSTANCE;
    }

    /**
     * 获取指定大小的ByteBuffer实例.
     *
     * @param cap 需要的字节大小.
     * @return 实例.
     */
    public DirectByteBuff allocate(int cap) {
        int blockSize = calculateBlockSize(cap);
        int blockIndex = calculationBlockIndex(blockSize);

        synchronized (buffers) {
            if (blockIndex > buffers.length - 1) {
                DirectByteBuff[] newBuffers = new DirectByteBuff[buffers.length * 2];
                System.arraycopy(buffers, 0, newBuffers, 0, buffers.length);
                this.buffers = newBuffers;
            }
        }

        DirectByteBuff buff = buffers[blockIndex];
        if (buff == null) {
            synchronized (buffers) {
                buff = buffers[blockIndex];
                if (buff == null) {
                    buff = new DirectByteBuff(ByteBuffer.allocateDirect(blockSize));
                    buffers[blockIndex] = buff;
                }
            }
        } else {
            if (buff.isBusy()) {
                throw new RuntimeException(String.format("A direct buffer of length (%d) is in use.", cap));
            }
        }
        buff.busy();
        return buff;
    }

    // 计算大于cap的最小alignmentBlockSize倍数.
    private int calculateBlockSize(int cap) {
        return cap + (alignmentBlockSize - 1) & ~(alignmentBlockSize - 1);
    }

    // 计算blockSize处于的下标序号.
    private int calculationBlockIndex(int blockSize) {
        return blockSize / alignmentBlockSize - 1;
    }

    /**
     * 池化直接内存.
     */
    public static class DirectByteBuff implements AutoCloseable {

        private ByteBuffer buff;
        private volatile boolean free;

        public DirectByteBuff(ByteBuffer buff) {
            this.buff = buff;
            this.free = true;
        }

        public ByteBuffer buff() {
            return buff;
        }

        protected void busy() {
            this.free = false;
        }

        protected boolean isBusy() {
            return !this.free;
        }

        @Override
        public void close() throws Exception {
            if (isBusy()) {
                this.free = true;
                this.buff.clear();
            }
        }
    }
}
