package com.xforceplus.ultraman.oqsengine.plus.common.datasource;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigObject;
import com.typesafe.config.ConfigValue;
import com.xforceplus.ultraman.oqsengine.plus.common.datasource.log.LoggerDataSource;
import com.xforceplus.ultraman.sdk.core.datasource.PackageInternal;
import com.xforceplus.ultraman.sdk.core.datasource.SupportClientType;
import com.xforceplus.ultraman.sdk.core.datasource.resolver.DataConfigResolver;
import com.xforceplus.ultraman.sdk.infra.base.thread.ExecutorHelper;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import io.micrometer.core.instrument.Metrics;
import io.vavr.Tuple;
import io.vavr.Tuple2;

import java.io.Closeable;
import java.lang.reflect.Method;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

/**
 * master resolver
 */
public class MasterResolver implements DataConfigResolver {

    @Override
    public Tuple2<String, PackageInternal> resolve(Config config) {

        ConfigObject configObject = config.getObject(DATA_SOURCES);
        if (configObject.isEmpty()) {
            throw new RuntimeException("至少配置一个数据源");
        }

        /**
         * TODO show closeable
         */
        PackageInternal masterPackageInternal = null;
        if (config.hasPath(SupportClientType.MASTER_DB.getPath())) {
            masterPackageInternal = buildDataSources(
                    (List<Config>) config.getConfigList(SupportClientType.MASTER_DB.getPath()), true);
            return Tuple.of("master", masterPackageInternal);
        }

        throw new RuntimeException("No Master datasource config");
    }

    private static PackageInternal buildDataSources(List<Config> configs, boolean showSql) {
        Map<Object, Object> ds = new LinkedHashMap<>(configs.size());
        String name = SupportClientType.MASTER_DB.getName() + "-" + 0;
        Closeable first = null;
        for (int i = 0; i < configs.size(); i++) {
            String dsName = SupportClientType.MASTER_DB.getName() + "-" + i;
            Closeable dataSource = buildDataSource(dsName, configs.get(i), showSql);
            if (i == 0) {
                first = dataSource;
            }
            ds.put(dsName, dataSource);
        }
        return new PackageInternal(name, first, ds);
    }

    private static Closeable buildDataSource(String name, Config config, boolean showSql) {
        HikariConfig hikariConfig = new HikariConfig();
        // 默认事务级别为读可提交.
        hikariConfig.setTransactionIsolation("TRANSACTION_READ_COMMITTED");
        hikariConfig.setPoolName(name);
        hikariConfig.setMetricRegistry(Metrics.globalRegistry);

        config.entrySet().stream().forEach(e -> {

            try {
                invokeMethod(hikariConfig, e.getKey(), e.getValue());
            } catch (Exception ex) {
                throw new RuntimeException(
                        String.format("Configuration error, wrong property '%s' '%s'.", e.getKey(), e.getValue()));
            }
        });

        hikariConfig.setThreadFactory(ExecutorHelper.buildNameThreadFactory("jdbc-pool", false));

        if (showSql) {
            return new LoggerDataSource(new HikariDataSource(hikariConfig));
        } else {
            return new HikariDataSource(hikariConfig);
        }
    }

    private static void invokeMethod(HikariConfig hikariConfig, String attrName, ConfigValue value) throws Exception {
        Class clazz = hikariConfig.getClass();
        String methodName = "set" + attrName.toUpperCase(Locale.US).substring(0, 1) + attrName.substring(1);
        Method method = null;
        switch (value.valueType()) {
            case NUMBER: {
                try {
                    method = clazz.getMethod(methodName, Long.TYPE);
                } catch (NoSuchMethodException ex) {
                    method = clazz.getMethod(methodName, Integer.TYPE);
                }
                break;
            }
            case STRING: {
                method = clazz.getMethod(methodName, String.class);
                break;
            }

            case BOOLEAN: {
                method = clazz.getMethod(methodName, Boolean.TYPE);
                break;
            }
            default: {
                throw new NoSuchMethodException(
                    String.format("The '%s' property setting could not be found.", attrName));
            }
        }

        method.invoke(hikariConfig, value.unwrapped());
    }
}
