package com.xforceplus.ultraman.core.impl;

import com.xforceplus.ultraman.core.EntityWriteService;
import com.xforceplus.ultraman.core.pojo.OqsEngineResult;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.sdk.core.pojo.UpdateConfig;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.entity.IEntity;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.MasterStorage;
import com.xforceplus.ultraman.oqsengine.plus.storage.pojo.dto.EntityPackage;
import com.xforceplus.ultraman.sdk.core.rel.legacy.*;
import com.xforceplus.ultraman.sdk.infra.metrics.MetricsDefine;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.annotation.Transactional;

import java.sql.SQLException;
import java.util.*;
import java.util.stream.Collectors;

import static org.springframework.transaction.interceptor.TransactionAspectSupport.currentTransactionStatus;

/**
 * Created by justin.xu on 03/2023.
 *
 * @since 1.8
 */
@Slf4j
@Transactional
public class EntityWriteServiceImpl implements EntityWriteService {

    private MasterStorage masterStorage;

    private EntityClassEngine engine;
    
    private boolean recordUser;

    private final Counter failCountTotal = Metrics.counter(MetricsDefine.FAIL_COUNT_TOTAL);

    public EntityWriteServiceImpl(MasterStorage masterStorage, EntityClassEngine engine, boolean recordUser) {
        this.masterStorage = masterStorage;
        this.engine = engine;
        this.recordUser = recordUser;
    }

    @Override
    public OqsEngineResult<Long> build(IEntity entity, Map<String, Object> context) {

        if (entity.time() == 0) {
            entity.markTime();
        }

        Optional<IEntityClass> entityClassOp =
                engine.load(Long.toString(entity.entityClassRef().getId())
                        , entity.entityClassRef().getProfile());

        if (entityClassOp.isPresent()) {
            try {
                boolean result =
                        masterStorage.build(entity, entityClassOp.get(), context);

                if (result) {
                    return OqsEngineResult.success(entity.id());
                } else {
                    currentTransactionStatus().setRollbackOnly();
                    failCountTotal.increment();
                    return OqsEngineResult.unCreated();
                }
            } catch (SQLException throwable) {
                currentTransactionStatus().setRollbackOnly();
                failCountTotal.increment();
                return OqsEngineResult.panic(throwable);
            }
        }
        return OqsEngineResult.notExistMeta(entity.entityClassRef());
    }

    @Override
    public OqsEngineResult<Long> build(IEntity[] entities, Map<String, Object> context) {
        Map<Tuple2<Long, String>, List<IEntity>> grouped = Arrays.stream(entities)
                .collect(Collectors
                        .groupingBy(x -> Tuple.of(x.entityClassRef().getId()
                                , Optional.ofNullable(x.entityClassRef().getProfile()).orElse(""))));
        try {
            EntityPackage entityPackage = EntityPackage.build();
            
            long start = System.currentTimeMillis();
            for (Map.Entry<Tuple2<Long, String>, List<IEntity>> entityEntry : grouped.entrySet()) {
                Optional<IEntityClass> entityClassOp =
                        engine.load(Long.toString(entityEntry.getKey()._1), entityEntry.getKey()._2);

                entityClassOp.ifPresent(entityClass -> {

                    List<IEntity> entityList = entityEntry.getValue();
                    entityList.forEach(x -> {
                        if (x.time() == 0) {
                            x.markTime();
                        }
                        entityPackage.put(x, entityClass);
                    });
                });
            }

            log.warn("build time {}", System.currentTimeMillis() - start);
            masterStorage.build(entityPackage, context);
            return OqsEngineResult.success(entityPackage.size());
        } catch (Throwable throwable) {
            currentTransactionStatus().setRollbackOnly();
            failCountTotal.increment(entities.length);
            return OqsEngineResult.panic(throwable);
        }
    }

    @Override
    public OqsEngineResult<Long> replace(IEntity entity, UpdateConfig updateConfig, Map<String, Object> context) {
        if (entity.time() == 0) {
            entity.markTime();
        }

        Optional<IEntityClass> entityClassOp =
                engine.load(Long.toString(entity.entityClassRef().getId()), entity.entityClassRef().getProfile());

        if (entityClassOp.isPresent()) {
            try {

                if (updateConfig != null && updateConfig.getVersion() != null && updateConfig.getVersion() >= 0) {
                    entity.resetVersion(updateConfig.getVersion());
                } else {
                    entity.resetVersion(-1);
                }

                if (updateConfig != null && updateConfig.getUseOptimisticLock() != null && updateConfig.getUseOptimisticLock()) {
                    entity.resetOptimizeLock(true);
                } else {
                    entity.resetOptimizeLock(false);
                }

                boolean result =
                        masterStorage.replace(entity, entityClassOp.get(), context);

                if (result) {
                    return OqsEngineResult.success(entity.id());
                } else {
                    currentTransactionStatus().setRollbackOnly();
                    return OqsEngineResult.unReplaced(entity.id());
                }
            } catch (SQLException throwable) {
                currentTransactionStatus().setRollbackOnly();
                failCountTotal.increment();
                return OqsEngineResult.panic(throwable);
            }
        }
        return OqsEngineResult.notExistMeta(entity.entityClassRef());
    }

    @Override
    public OqsEngineResult<Long> replaceConditional(IEntity entity
            , UpdateConfig updateConfig, ExpRel expRel, Map<String, Object> context) {

        Optional<IEntityClass> entityClassOp =
                engine.load(Long.toString(entity.entityClassRef().getId()), entity.entityClassRef().getProfile());

        if (entityClassOp.isPresent()) {
            try {
                if (updateConfig != null && updateConfig.getVersion() != null && updateConfig.getVersion() >= 0) {
                    entity.resetVersion(updateConfig.getVersion());
                } else {
                    entity.resetVersion(-1);
                }

                if (updateConfig != null && updateConfig.getUseOptimisticLock() != null && updateConfig.getUseOptimisticLock()) {
                    entity.resetOptimizeLock(true);
                } else {
                    entity.resetOptimizeLock(false);
                }

                EntityPackage entityPackage = new EntityPackage();
                Map<String, ExpRel> mapping = new HashMap<>();
                mapping.put(entityClassOp.get().code(), expRel);
                entityPackage.setCondition(mapping);
                entityPackage.setExample(entity);
                entityPackage.setTargetEntityClass(entityClassOp.get());
                entityPackage.setTargetCode(entityClassOp.get().code());

                int size = masterStorage.replaceByCondition(entityPackage, context);

                if (size > 0) {
                    return OqsEngineResult.success(entity.id());
                } else {
                    currentTransactionStatus().setRollbackOnly();
                    return OqsEngineResult.unReplaced(entity.id());
                }
            } catch (SQLException throwable) {
                currentTransactionStatus().setRollbackOnly();
                failCountTotal.increment();
                return OqsEngineResult.panic(throwable);
            }
        }
        return OqsEngineResult.notExistMeta(entity.entityClassRef());
    }

    @Override
    public OqsEngineResult<Long> replace(IEntity[] entities, Map<String, Object> context) {
        EntityPackage entityPackage = EntityPackage.build();
        for (IEntity entity : entities) {
            if (entity.time() == 0) {
                entity.markTime();
            }
            Optional<IEntityClass> entityClassOp =
                    engine.load(Long.toString(entity.entityClassRef().getId()), entity.entityClassRef().getProfile());
            entityClassOp.ifPresent(entityClass -> entityPackage.put(entity, entityClass));
        }
        try {
            Boolean[] replace = masterStorage.replace(entityPackage, context);
            long successCount = Arrays.stream(replace).filter(x -> x).count();
            long failedCount = replace.length - successCount;
            
            if(failedCount > 0) {
                return OqsEngineResult.unReplaced(-1L);
            } else {
                return OqsEngineResult.success(Arrays.stream(replace).filter(x -> x).count());
            }
        } catch (SQLException throwable) {
            currentTransactionStatus().setRollbackOnly();
            failCountTotal.increment(entities.length);
            return OqsEngineResult.panic(throwable);
        }
    }

    @Override
    public OqsEngineResult<Long> replaceConditional(IEntity[] entities, Map<String, ExpRel> expRelMapping
            , Map<String, Object> context) {
        EntityPackage entityPackage = EntityPackage.build();
        for (IEntity entity : entities) {
            if (entity.time() == 0) {
                entity.markTime();
            }
            Optional<IEntityClass> entityClassOp =
                    engine.load(Long.toString(entity.entityClassRef().getId()), entity.entityClassRef().getProfile());
            entityClassOp.ifPresent(entityClass -> entityPackage.put(entity, entityClass));
        }
        entityPackage.setCondition(expRelMapping);
        try {
            int ret = masterStorage.replaceByCondition(entityPackage, context);
            long successCount = ret;
            long failedCount = entities.length - successCount;

            if(failedCount > 0) {
                return OqsEngineResult.unReplaced(-1L);
            } else {
                return OqsEngineResult.success(successCount);
            }
        } catch (SQLException throwable) {
            currentTransactionStatus().setRollbackOnly();
            failCountTotal.increment(entities.length);
            return OqsEngineResult.panic(throwable);
        }
    }

    @Override
    public OqsEngineResult<Long> delete(IEntity entity, Map<String, Object> context) {
        if (entity.time() == 0) {
            entity.markTime();
        }
        Optional<IEntityClass> entityClassOp =
                engine.load(Long.toString(entity.entityClassRef().getId()), entity.entityClassRef().getProfile());

        if (entityClassOp.isPresent()) {
            try {
                boolean result =
                        masterStorage.delete(entity, entityClassOp.get(), context);
                Object relation = context.get("relations");
                if(relation instanceof String) {
                    String relationStr = relation.toString();
                    EntityClassGroup describe = engine.describe(entityClassOp.get(), entity.entityClassRef().getProfile());
                    Optional<IEntityClass> relatedEntityClassOp = describe.relatedEntityClass(relationStr);
                    if(relatedEntityClassOp.isPresent()) {
                        IEntityClass relatedEntityClass = relatedEntityClassOp.get();
                        EntityPackage entityPackage = new EntityPackage();
                        entityPackage.setTargetEntityClass(relatedEntityClass);
                        entityPackage.setTargetCode(relatedEntityClass.code());
                        ExpCondition query = ExpCondition.call(ExpOperator.EQUALS, ExpField.field(relationStr.concat(".id")), ExpValue.from(entity.id()));
                        Map<String, ExpRel> mapping = new HashMap<>();
                        mapping.put(relatedEntityClass.code(), new ExpQuery().filters(query).range(1, 10000));
                        entityPackage.setCondition(mapping);
                        masterStorage.deleteByCondition(entityPackage, context);
                    }
                }

                if (result) {
                    return OqsEngineResult.success(entity.id());
                } else {
                    currentTransactionStatus().setRollbackOnly();
                    failCountTotal.increment();
                    return OqsEngineResult.unDeleted(entity.id());
                }
            } catch (SQLException throwable) {
                currentTransactionStatus().setRollbackOnly();
                failCountTotal.increment();
                return OqsEngineResult.panic(throwable);
            }
        }
        currentTransactionStatus().setRollbackOnly();
        failCountTotal.increment();
        return OqsEngineResult.notExistMeta(entity.entityClassRef());
    }

    @Override
    public OqsEngineResult<Long> delete(IEntity[] entities, Map<String, Object> context) {
        EntityPackage entityPackage = EntityPackage.build();
        for (IEntity entity : entities) {
            if (entity.time() == 0) {
                entity.markTime();
            }
            Optional<IEntityClass> entityClassOp =
                    engine.load(Long.toString(entity.entityClassRef().getId()), entity.entityClassRef().getProfile());
            entityClassOp.ifPresent(entityClass -> entityPackage.put(entity, entityClass));
        }
        try {
            Boolean[] delete = masterStorage.delete(entityPackage, context);
            long success = Arrays.stream(delete).filter(x -> x).count();
            return OqsEngineResult.success(success);
        } catch (SQLException throwable) {
            currentTransactionStatus().setRollbackOnly();
            failCountTotal.increment(entities.length);
            return OqsEngineResult.panic(throwable);
        }
    }

    @Override
    public OqsEngineResult<Long> updateByCondition(IEntityClass entityClass, ExpRel expRel, IEntity example, Map<String, Object> context) {
        //entityClass + expRel =>
        EntityPackage entityPackage = new EntityPackage();
        Map<String, ExpRel> mapping = new HashMap<>();
        mapping.put(entityClass.code(), expRel);
        entityPackage.setCondition(mapping);
        entityPackage.setExample(example);
        entityPackage.setTargetCode(entityClass.code());
        entityPackage.setTargetEntityClass(entityClass);
        try {
            //body to set
            int affectedRow = masterStorage.replaceByCondition(entityPackage, context);
            return OqsEngineResult.success(affectedRow);
        } catch (SQLException throwable) {
            currentTransactionStatus().setRollbackOnly();
            return OqsEngineResult.panic(throwable);
        }
    }

    @Override
    public OqsEngineResult<Long> deleteByCondition(IEntityClass entityClass
            , ExpRel expRel
            , Map<String, Object> context) {
        //entityClass + expRel =>
        EntityPackage entityPackage = new EntityPackage();
        Map<String, ExpRel> mapping = new HashMap<>();
        mapping.put(entityClass.code(), expRel);
        entityPackage.setCondition(mapping);
        entityPackage.setTargetCode(entityClass.code());
        entityPackage.setTargetEntityClass(entityClass);
        try {
            //body to set
            int affectedRow = masterStorage.deleteByCondition(entityPackage, context);
            return OqsEngineResult.success(affectedRow);
        } catch (SQLException throwable) {
            currentTransactionStatus().setRollbackOnly();
            return OqsEngineResult.panic(throwable);
        }
    }
}
