package com.xforceplus.ultraman.core.status;

import com.xforceplus.ultraman.sdk.infra.metrics.MetricsDefine;
import com.xforceplus.ultraman.oqsengine.plus.common.mode.OqsMode;
import com.xforceplus.ultraman.sdk.infra.base.thread.ExecutorHelper;
import com.xforceplus.ultraman.sdk.infra.lifecycle.SimpleLifecycle;
import io.micrometer.core.instrument.Metrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * OQS的状态控制器.
 *
 * @author dongbin
 * @version 0.1 2022/6/1 13:04
 * @since 1.8
 */
public class DefaultStateControl implements StateControl, SimpleLifecycle {

    private final Logger logger = LoggerFactory.getLogger(DefaultStateControl.class);

    private final AtomicInteger oqsMode = Metrics.gauge(MetricsDefine.MODE, new AtomicInteger(0));
    private final AtomicInteger oqsModeRease =
            Metrics.gauge(MetricsDefine.READ_ONLEY_MODE_REASE, new AtomicInteger(0));

    private State state = State.NORMAL;
    private StateChecker checker;
    private ScheduledExecutorService worker;

    public DefaultStateControl(StateChecker checker) {
        this.checker = checker;
    }

    @Override
    public void init() throws Exception {
        this.worker = Executors.newScheduledThreadPool(1,
                ExecutorHelper.buildNameThreadFactory("oqs-state-monitor"));

        this.worker.scheduleWithFixedDelay(new Checker(), 12, 12, TimeUnit.SECONDS);
    }

    @Override
    public void destroy() throws Exception {
        ExecutorHelper.shutdownAndAwaitTermination(this.worker);
    }

    public boolean isNormal() {
        return State.NORMAL == state;
    }

    @Override
    public State getLastState() {
        return this.state;
    }

    /*
    检查任务.
     */
    class Checker implements Runnable {

        @Override
        public void run() {
            state = checker.check();

            if (State.UNKNOWN == state) {
                state = State.NORMAL;

                if (logger.isWarnEnabled()) {
                    logger.warn("The status check result is UNKNOWN, which assumes that the status is normal.");
                }
            }

            if (State.NORMAL != state) {
                if (logger.isWarnEnabled()) {
                    logger.warn(
                            "The state checker finds that the current state is abnormal and is {}.", state.name());
                }

                oqsMode.set(OqsMode.READ_ONLY.getValue());
                oqsModeRease.set(state.getSymbol());
            }
        }
    }
}
