package com.xforceplus.ultraman.core;

import com.xforceplus.ultraman.core.pojo.OqsEngineResult;
import com.xforceplus.ultraman.metadata.entity.EntityClassRef;
import com.xforceplus.ultraman.metadata.entity.IEntity;
import com.xforceplus.ultraman.metadata.values.IValue;
import com.xforceplus.ultraman.oqsengine.plus.storage.pojo.dto.select.SelectConfig;

/**
 * Created by justin.xu on 03/2023.
 *
 * @since 1.8
 */
public interface EntitySearchService {

    /**
     * <p>
     * 创建一个新的 entity 实例.<br />
     * 会为新的实例分配一个唯一的ID标识唯一表示此实例.<br />
     * 但是如果设置了一个非0的ID,那么将表示ID由外部指定.<br />
     * <em>注意: 如果指定了ID,那么OQS将不再负责ID的唯一性了.</em><br />
     * 如果OQS管理ID的话,成功创建后会使用"in-place"方式更新实例的ID.<br />
     * <em>注意: 如果是自动事务,任何非成功的响应都将造成事务回滚.</em>
     * </p>
     * 除了通用状态外,还会额外返回 UNCREATED 状态.
     *
     * @param entity 目标 entity 数据.
     * @return 操作结果.如果成功, 其值将是long数值.
     */
    OqsEngineResult<Long> build(IEntity entity);

    /**
     * <p>
     * 这是 build 创建实例的多实例版本.此方法用以在大量写入数据时优化IO.<br />
     * 基本行为和单体的 build 保持了一致.
     * 会为每一个新的实例分配一个唯一的ID标识唯一表示此实例.<br />
     * 但是如果设置了一个非0的ID,那么将表示ID由外部指定.<br />
     * 注意: 如果指定了ID,那么OQS将不再负责ID的唯一性了.<br />
     * 如果OQS管理ID的话,成功创建后会使用"in-place"方式更新实例的ID.<br />
     * <em>注意: 如果是非手动事务,那么任何一个对象的非成功都将造成所有对象失败,并且事务回滚.</em>
     * </p>
     * 除了通用状态外,还会额外返回 UNCREATED 状态.
     *
     * @param entities 目标实体列表.
     * @return 操作结果.如果成功, 返回值中将包含成功的数量.
     */
    OqsEngineResult<Long> build(IEntity[] entities);

    /**
     * <p>
     * 更新一个已经存在对象的字段.对象必须已经存在,不存在的对象将以错误状态返回.<br />
     * 支持部份字段更新,可以只传递指定对象实例的需要更新的部份字段.<br />
     * <em>注意: 在自动事务中实例非成功将造成事务回滚.</em>
     * </p>
     * 除了通用状态外,还会额外返回 UNREPLACE 和 NOT_FOUND 状态.
     *
     * @param entity 目标 entity.
     * @return 更新成功的数量.
     */
    OqsEngineResult<Long> replace(IEntity entity);

    /**
     * <p>
     * 更新存在对象字段的多实例版本.此方法用以在大量写入数据时优化IO.<br />
     * 基本行为和单实例版本的 replace 一致.<br />
     * 支持部份字段更新,可以只传递指定对象实例的需要更新的部份字段.<br />
     * <em>注意: 在自动事务中任意一个实例非成功将造成事务回滚.</em>
     * </p>
     * 除了通用状态外,还会额外返回 UNREPLACE 和 NOT_FOUND 状态.
     *
     * @param entities 目标实体列表.
     * @return 更新成功的数量.
     */
    OqsEngineResult<Long> replace(IEntity[] entities);

    /**
     * <p>
     * 条件更新,所有符合条件的实例都将被更新.
     * 和多实例版本的 replace 的基本行为一致. <br />
     * <em>注意: 在自动事务中任意一个实例非成功将造成事务回滚.</em>
     * </p>
     * 除了通用状态外,还会额外返回 UNREPLACE 状态,但不会返回 NOT_FOUND 状态.<br />
     * 因为指定的是条件,是允许不更新任何实例的.
     *
     * @param newValues      新的字段值.
     * @param entityClassRef 对象信息.
     * @return 更新成功的数量.
     */
    OqsEngineResult<Long> replace(SelectConfig selectConfig, EntityClassRef entityClassRef, IValue... newValues);

    /**
     * <p>
     * 删除一个已经存在的对象实例.<br />
     * 如果要删除的对象实例不存在,将作为一个错误状态返回.<br />
     * <em>注意: 在自动事务中任意一个实例非成功将造成事务回滚.</em>
     * </p>
     * 除了通用状态外,还会额外返回 UNDELETED, NOT_FOUND 状态.<br />
     *
     * @param entity 目标 entity.
     * @return 删除成功的数量.
     */
    OqsEngineResult<Long> delete(IEntity entity);

    /**
     * <p>
     * 删除多个已经存在的对象实例.<br />
     * 基本行为和单实例版本的 delete 基本一致.<br />
     * <em>注意: 在自动事务中任意一个实例非成功将造成事务回滚.</em>
     * </p>
     * 除了通用状态外,还会额外返回 UNDELETED, NOT_FOUND 状态.<br />
     *
     * @param entities 目标 entity 列表.
     * @return 被删除的实例数量.
     */
    OqsEngineResult<Long> delete(IEntity[] entities);

    /**
     * <p>
     * 条件删除, 所有符合结果的对象都将被删除.<br />
     * 删除的基本行为和多实例版本一致,但是如果找不到删除的对象实例不认为是一个错误.将以SUCCESS状态返回.<br />
     * <em>注意: 在自动事务中任意一个实例非成功将造成事务回滚.</em>
     * </p>
     * 除了通用状态外,还会额外返回 UNDELETED, 但不会返回NOT_FOUND 状态.<br />
     *
     * @param entityClassRef 类型信息.
     * @return 被删除的实例数量.
     */
    OqsEngineResult<Long> delete(SelectConfig selectConfig, EntityClassRef entityClassRef);
}
