package com.xforceplus.ultraman.core.impl;

import com.xforceplus.ultraman.core.EntityWriteService;
import com.xforceplus.ultraman.core.pojo.OqsEngineResult;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.entity.EntityClassRef;
import com.xforceplus.ultraman.metadata.entity.IEntity;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.values.IValue;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.MasterStorage;
import com.xforceplus.ultraman.oqsengine.plus.storage.pojo.dto.EntityPackage;
import com.xforceplus.ultraman.oqsengine.plus.storage.pojo.dto.select.SelectConfig;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.annotation.Transactional;

import java.sql.SQLException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import static org.springframework.transaction.interceptor.TransactionAspectSupport.currentTransactionStatus;

/**
 * Created by justin.xu on 03/2023.
 *
 * @since 1.8
 */
@Slf4j
@Transactional
public class EntityWriteServiceImpl implements EntityWriteService {

    private MasterStorage masterStorage;

    private EntityClassEngine engine;

    public EntityWriteServiceImpl(MasterStorage masterStorage, EntityClassEngine engine) {
        this.masterStorage = masterStorage;
        this.engine = engine;
    }

    @Override
    public OqsEngineResult<Long> build(IEntity entity, Map<String, Object> context) {

        if (entity.time() == 0) {
            entity.markTime();
        }

        Optional<IEntityClass> entityClassOp =
                engine.load(Long.toString(entity.entityClassRef().getId()), entity.entityClassRef().getProfile());

        if (entityClassOp.isPresent()) {
            try {
                boolean result =
                        masterStorage.build(entity, entityClassOp.get(), context);

                if (result) {
                    return OqsEngineResult.success(entity.id());
                } else {
                    currentTransactionStatus().setRollbackOnly();
                    return OqsEngineResult.elevateFailed();
                }
            } catch (SQLException throwables) {
                currentTransactionStatus().setRollbackOnly();
                return OqsEngineResult.elevateFailed(throwables.getMessage());
            }
        }
        return OqsEngineResult.elevateFailed("entityClass load failed.");
    }

    @Override
    public OqsEngineResult<Long> build(IEntity[] entities, Map<String, Object> context) {
        EntityPackage entityPackage = EntityPackage.build();
        Map<Tuple2<Long, String>, List<IEntity>> grouped = Arrays.stream(entities)
                .collect(Collectors
                        .groupingBy(x -> Tuple.of(x.entityClassRef().getId()
                                , Optional.ofNullable(x.entityClassRef().getProfile()).orElse(""))));

        try {
            long start = System.currentTimeMillis();
            for (Map.Entry<Tuple2<Long,String>, List<IEntity>> entityEntry : grouped.entrySet()) {
                Optional<IEntityClass> entityClassOp =
                        engine.load(Long.toString(entityEntry.getKey()._1), entityEntry.getKey()._2);

                entityClassOp.ifPresent(entityClass -> {

                    List<IEntity> entityList = entityEntry.getValue();
                    entityList.forEach(x -> {
                        if (x.time() == 0) {
                            x.markTime();
                        }
                        entityPackage.put(x, entityClass);
                    });
                });
            }

            log.warn("build time {}", System.currentTimeMillis() - start);
            masterStorage.build(entityPackage, context);

            return OqsEngineResult.success(entityPackage.size());
        } catch (Throwable throwables) {
            currentTransactionStatus().setRollbackOnly();
            return OqsEngineResult.elevateFailed(throwables.getMessage());
        }
    }

    @Override
    public OqsEngineResult<Long> replace(IEntity entity, Map<String, Object> context) {
        if (entity.time() == 0) {
            entity.markTime();
        }

        Optional<IEntityClass> entityClassOp =
                engine.load(Long.toString(entity.entityClassRef().getId()), entity.entityClassRef().getProfile());

        if (entityClassOp.isPresent()) {
            try {
                boolean result =
                        masterStorage.replace(entity, entityClassOp.get(), context);

                if (result) {
                    return OqsEngineResult.success(entity.id());
                } else {
                    currentTransactionStatus().setRollbackOnly();
                    return OqsEngineResult.elevateFailed();
                }
            } catch (SQLException throwables) {
                currentTransactionStatus().setRollbackOnly();
                return OqsEngineResult.elevateFailed(throwables.getMessage());
            }
        }
        return OqsEngineResult.elevateFailed("entityClass load failed.");

    }

    @Override
    public OqsEngineResult<Long> replace(IEntity[] entities, Map<String, Object> context) {
        EntityPackage entityPackage = EntityPackage.build();
        for (IEntity entity : entities) {
            if (entity.time() == 0) {
                entity.markTime();
            }
            Optional<IEntityClass> entityClassOp =
                    engine.load(Long.toString(entity.entityClassRef().getId()), entity.entityClassRef().getProfile());
            entityClassOp.ifPresent(entityClass -> entityPackage.put(entity, entityClass));
        }
        try {
            masterStorage.replace(entityPackage, context);

            return OqsEngineResult.success(entityPackage.size());
        } catch (SQLException throwables) {
            currentTransactionStatus().setRollbackOnly();
            return OqsEngineResult.elevateFailed(throwables.getMessage());
        }
    }

    @Override
    public OqsEngineResult<Long> replace(SelectConfig selectConfig, EntityClassRef entityClassRef,
                                         Map<String, Object> context, IValue... newValues) {
        return null;
    }

    @Override
    public OqsEngineResult<Long> delete(IEntity entity, Map<String, Object> context) {
        if (entity.time() == 0) {
            entity.markTime();
        }
        Optional<IEntityClass> entityClassOp =
                engine.load(Long.toString(entity.entityClassRef().getId()), entity.entityClassRef().getProfile());

        if (entityClassOp.isPresent()) {
            try {
                boolean result =
                        masterStorage.delete(entity, entityClassOp.get(), context);

                if (result) {
                    return OqsEngineResult.success(entity.id());
                } else {
                    currentTransactionStatus().setRollbackOnly();
                    return OqsEngineResult.elevateFailed();
                }
            } catch (SQLException throwables) {
                currentTransactionStatus().setRollbackOnly();
                return OqsEngineResult.elevateFailed(throwables.getMessage());
            }
        }
        currentTransactionStatus().setRollbackOnly();
        return OqsEngineResult.elevateFailed("entityClass load failed.");
    }

    @Override
    public OqsEngineResult<Long> delete(IEntity[] entities, Map<String, Object> context) {
        EntityPackage entityPackage = EntityPackage.build();
        for (IEntity entity : entities) {
            if (entity.time() == 0) {
                entity.markTime();
            }

            Optional<IEntityClass> entityClassOp =
                    engine.load(Long.toString(entity.entityClassRef().getId()), entity.entityClassRef().getProfile());
            entityClassOp.ifPresent(entityClass -> entityPackage.put(entity, entityClass));
        }
        try {
            masterStorage.delete(entityPackage, context);
            return OqsEngineResult.success(entityPackage.size());
        } catch (SQLException throwables) {
            currentTransactionStatus().setRollbackOnly();
            return OqsEngineResult.elevateFailed(throwables.getMessage());
        }
    }

    @Override
    public OqsEngineResult<Long> delete(SelectConfig selectConfig, EntityClassRef entityClassRef, Map<String, Object> context) {
        return null;
    }
}
