package com.xforceplus.ultraman.core.pojo;


import com.xforceplus.ultraman.core.pojo.exception.OqsEngineException;
import com.xforceplus.ultraman.core.status.State;
import com.xforceplus.ultraman.metadata.entity.EntityClassRef;
import com.xforceplus.ultraman.metadata.entity.IEntity;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.metadata.values.IValue;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.QueryResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

/**
 * OQS的操作响应.<br />
 * 响应的处理方式为传入 OqsEngineResultAction 子类并实现关注的错误处理逻辑.如下.
 * <pre>
 *     {@code
 *     OqsEngineResult<Long> result = ..;
 *     Optional<String> r = result.act(new OqsEngineResultAction<String, Long>() {
 *
 *         // 处理操作成功.
 *         @Override
 *         public Optional<String> success(Optional<String> message, Collection<Hint> hints) {
 *             return Optional.of("success");
 *         }
 *
 *         // 发生冲突.
 *         @Override
 *         public Optional<String> conflict(Optional<String> message, Collection<Hint> hints) {
 *             return Optional.of("conflict");
 *         }
 *
 *         //...其余状态.
 *     });
 *     }
 * </pre>
 * 所有不关注的响应将由默认处理逻辑接管, 默认都将以异常结束.<br />
 * 如果有无法识别的状态,将识别为UNKNOWN状态.
 *
 * @param <V> 正确响应的值.
 * @author dongbin
 * @version 0.1 2022/6/6 13:20
 * @since 1.8
 */
public class OqsEngineResult<V> {

    private static final Logger LOGGER = LoggerFactory.getLogger(OqsEngineResult.class);
    private ResultStatus resultStatus;
    private V value;
    private Throwable ex;
    private Collection<Hint> hints;
    private String message;

    // 由于成功在所有状态检查中都会作为一个结果返回,这里为了降低GC压力使用共享实例.
    private static final OqsEngineResult SUCCESS = new OqsEngineResult(ResultStatus.SUCCESS, "");

    public static OqsEngineResult unknown() {
        return new OqsEngineResult(ResultStatus.UNKNOWN, ResultStatus.UNKNOWN.name());
    }

    public static OqsEngineResult success() {
        return SUCCESS;
    }

    public static OqsEngineResult success(String msg) {
        return new OqsEngineResult(ResultStatus.SUCCESS, msg);
    }

    public static OqsEngineResult<QueryResult> success(QueryResult entity) {
        return new OqsEngineResult(ResultStatus.SUCCESS, entity, null);
    }

    public static OqsEngineResult<QueryResult[]> success(QueryResult[] entities) {
        return new OqsEngineResult(ResultStatus.SUCCESS, entities, null);
    }

    public static OqsEngineResult<Map<IEntity, IValue[]>> success(Map<IEntity, IValue[]> changes) {
        return new OqsEngineResult<>(ResultStatus.SUCCESS, changes, null);
    }

    public static OqsEngineResult<Map.Entry<IEntity, IValue[]>> success(Map.Entry<IEntity, IValue[]> change) {
        return new OqsEngineResult<>(ResultStatus.SUCCESS, change, null);
    }

    public static OqsEngineResult<Long> success(long value) {
        return new OqsEngineResult<>(ResultStatus.SUCCESS, value, null);
    }

    public static OqsEngineResult<Collection<QueryResult>> success(Collection<QueryResult> entities) {
        return new OqsEngineResult<>(ResultStatus.SUCCESS, entities, null);
    }

    public static OqsEngineResult panic(Throwable throwable) {

        LOGGER.error(throwable.getMessage(), throwable);

        return new OqsEngineResult<>(throwable);
    }

    public static OqsEngineResult conflict() {
        return conflict(null);
    }

    public static OqsEngineResult conflict(String msg) {
        return new OqsEngineResult(ResultStatus.CONFLICT, msg);
    }

    public static OqsEngineResult conflict(long entityId) {
        return notFound(String.format("A conflict occurred for entity %d..", entityId));
    }

    public static OqsEngineResult notFound() {
        return notFound(null);
    }

    public static OqsEngineResult notFound(String msg) {
        return new OqsEngineResult(ResultStatus.NOT_FOUND, msg);
    }

    public static OqsEngineResult notFound(long entityId) {
        return notFound(String.format("Entity %d was not found.", entityId));
    }

    public static OqsEngineResult unCreated() {
        return new OqsEngineResult(ResultStatus.UNCREATED, "The entity was not created successfully.");
    }

    /**
     * 构造一个未成功更新的状态结果.
     *
     * @param id 未成功更新的目标对象标识.
     * @return 结果.
     */
    public static OqsEngineResult unReplaced(long id) {
        OqsEngineResult result =
                new OqsEngineResult(ResultStatus.UNREPLACE,
                        String.format("The entity(%d) was not replace successfully.", id));
        result.addHint(new Hint(id));
        return result;
    }

    /**
     * 构造一个未成功删除的状态结果.
     *
     * @param id 未成功更新的目标对象标识.
     * @return 结果.
     */
    public static OqsEngineResult unDeleted(long id) {
        OqsEngineResult result =
                new OqsEngineResult(ResultStatus.UNDELETED,
                        String.format("The entity(%d) was not deleted successfully.", id));
        result.addHint(new Hint(id));
        return result;
    }

    public static OqsEngineResult unAccumulate() {
        return unAccumulate(null);
    }

    public static OqsEngineResult unAccumulate(String msg) {
        return new OqsEngineResult(ResultStatus.UNACCUMULATE, msg);
    }

    public static OqsEngineResult elevateFailed() {
        return elevateFailed(null);
    }

    public static OqsEngineResult elevateFailed(String msg) {
        return new OqsEngineResult(ResultStatus.ELEVATEFAILED, msg);
    }

    public static OqsEngineResult halfSuccess() {
        return halfSuccess(null);
    }

    public static OqsEngineResult halfSuccess(String msg) {
        return new OqsEngineResult(ResultStatus.HALF_SUCCESS, msg);
    }

    /**
     * 缺少必须字段.
     */
    public static OqsEngineResult fieldMust(IEntityField field) {
        OqsEngineResult result =
                new OqsEngineResult(ResultStatus.FIELD_MUST, String.format("The field %s is required.", field.name()));
        result.addHint(new Hint(field));
        return result;
    }

    /**
     * 字段过长.
     */
    public static OqsEngineResult fieldTooLong(IEntityField field) {
        OqsEngineResult result = new OqsEngineResult(ResultStatus.FIELD_TOO_LONG,
                String.format("Field %s is too long. The maximum allowed length is %d.",
                        field.name(), field.config().getLen()));
        result.addHint(new Hint(field));
        return result;
    }

    /**
     * 精度过高.
     */
    public static OqsEngineResult fieldHighPrecision(IEntityField field) {
        OqsEngineResult result = new OqsEngineResult(ResultStatus.FIELD_HIGH_PRECISION,
                String.format("Field %s is too precise. The maximum accuracy allowed is %d.",
                        field.name(), field.config().getPrecision()));
        result.addHint(new Hint(field));
        return result;
    }

    /**
     * 字段不存在.
     */
    public static OqsEngineResult fieldNonExist(IEntityField field) {
        OqsEngineResult result = new OqsEngineResult(ResultStatus.FIELD_NON_EXISTENT,
                String.format("The field %s does not exist.", field.name()));
        result.addHint(new Hint(field));
        return result;
    }

    public static OqsEngineResult notExistMeta() {
        return new OqsEngineResult(ResultStatus.NOT_EXIST_META, "Unexpected meta information.");
    }

    /**
     * 元信息不存在.
     */
    public static OqsEngineResult notExistMeta(EntityClassRef ref) {
        OqsEngineResult result;
        if (ref.getProfile() != null) {
            result = new OqsEngineResult(ResultStatus.NOT_EXIST_META,
                    String.format("Meta message %s-%s does not exist.", ref.getCode(), ref.getProfile()));
        } else {
            result = new OqsEngineResult(ResultStatus.NOT_EXIST_META,
                    String.format("Meta message %s does not exist.", ref.getCode()));
        }

        result.addHint(new Hint(ref));

        return result;
    }

    public static OqsEngineResult oqsStateReadOnly() {
        return new OqsEngineResult(ResultStatus.STATE_READ_ONLY, "The OQS is in read-only mode for unknown reasons.");
    }

    /**
     * 状态只读.
     *
     * @param state 原因.
     * @return 结果.
     */
    public static OqsEngineResult oqsStateReadOnly(State state) {
        OqsEngineResult result =
                new OqsEngineResult(ResultStatus.STATE_READ_ONLY,
                        String.format("The OQS is in read-only mode for %s.", state.name()));
        result.addHint(new Hint(state));
        return result;
    }

    private OqsEngineResult(Throwable ex) {
        this.ex = ex;
        this.resultStatus = ResultStatus.PANIC;
        this.message = ex.getMessage();
    }

    private OqsEngineResult(ResultStatus resultStatus, String message) {
        this(resultStatus, null, message);
    }

    private OqsEngineResult(ResultStatus resultStatus, V value, String message) {
        this.resultStatus = resultStatus;
        this.message = message;
        this.value = value;
    }

    public OqsEngineResult copy(Object value) {
        return new OqsEngineResult(this.resultStatus, value, this.message);
    }

    /**
     * 增加一个提示.
     *
     * @param hint 提示.
     * @return 当前实例.
     */
    public OqsEngineResult addHint(Hint hint) {
        if (hint == null) {
            return this;
        }

        if (this.hints == null) {
            this.hints = new LinkedList();
        }

        this.hints.add(hint);

        if (ResultStatus.SUCCESS == this.resultStatus) {
            this.resultStatus = ResultStatus.HALF_SUCCESS;
        }

        return this;
    }

    /**
     * 增加多个提示.
     *
     * @param hints 提示列表.
     * @return 当前实例.
     */
    public OqsEngineResult addHints(Collection<Hint> hints) {
        if (hints == null || hints.isEmpty()) {
            return this;
        }

        if (this.hints == null) {
            this.hints = new LinkedList();
        }

        this.hints.addAll(hints);

        if (ResultStatus.SUCCESS == this.resultStatus) {
            this.resultStatus = ResultStatus.HALF_SUCCESS;
        }

        return this;
    }

    /**
     * 判断是否成功的响应.
     * <em>注意: ResultStatus.HALF_SUCCESS 也认为是成功的一种.</em>
     *
     * @return true 成功, false非成功.
     */
    public boolean isSuccess() {
        return ResultStatus.SUCCESS == this.resultStatus || ResultStatus.HALF_SUCCESS == this.resultStatus;
    }

    /**
     * 判断是否半成功.
     *
     * @return true 是, false 不是.
     */
    public boolean isHalfSuccess() {
        return ResultStatus.HALF_SUCCESS == this.resultStatus;
    }

    /**
     * 判断是否出现OQS无法处理的错误.
     *
     * @return true 是, false 不是.
     */
    public boolean isPanic() {
        return ResultStatus.PANIC == this.resultStatus;
    }

    /**
     * 不处理任何响应结果.
     * 当发生非成功的状态时会以异常结束.
     */
    public void act() throws Exception {
        act(new OqsEngineReusltAction<>());
    }

    /**
     * 处理响应.需要传入 OqsEngineReusltAction 的实例,一是其子类.<br>
     * 所有非成功的响应默认都以异常处理.
     *
     * @param action 响应动作.
     */
    public <T> Optional<T> act(OqsEngineReusltAction<T, V> action) {
        if (this.hints == null) {
            this.hints = new LinkedList<>();
        }

        Optional<T> value;
        switch (this.resultStatus) {
            case SUCCESS: {
                value = action.success(Optional.ofNullable(this.value), this.hints);
                break;
            }
            case HALF_SUCCESS: {
                value = action.halfSuccess(Optional.ofNullable(this.value), this.hints);
                break;
            }
            case CONFLICT: {
                value = action.conflict(Optional.ofNullable(this.message), this.hints);
                break;
            }
            case NOT_FOUND: {
                value = action.notFound(Optional.ofNullable(this.message), this.hints);
                break;
            }
            case UNCREATED: {
                value = action.unCreated(Optional.ofNullable(this.message), this.hints);
                break;
            }
            case UNREPLACE: {
                value = action.unReplace(Optional.ofNullable(this.message), this.hints);
                break;
            }
            case UNDELETED: {
                value = action.unDeleted(Optional.ofNullable(this.message), this.hints);
                break;
            }
            case UNACCUMULATE: {
                value = action.unAccumulate(Optional.ofNullable(this.message), this.hints);
                break;
            }
            case ELEVATEFAILED: {
                value = action.elevateFailed(Optional.ofNullable(this.message), this.hints);
                break;
            }
            case FIELD_MUST: {
                value = action.fieldMust(Optional.ofNullable(this.message), this.hints);
                break;
            }
            case FIELD_TOO_LONG: {
                value = action.fieldTooLong(Optional.ofNullable(this.message), this.hints);
                break;
            }
            case FIELD_HIGH_PRECISION: {
                value = action.fieldHighPrecision(Optional.ofNullable(this.message), this.hints);
                break;
            }
            case FIELD_NON_EXISTENT: {
                value = action.fieldNonExist(Optional.ofNullable(this.message), this.hints);
                break;
            }
            case NOT_EXIST_META: {
                value = action.notExistMeta(Optional.ofNullable(this.message), this.hints);
                break;
            }
            case STATE_READ_ONLY: {
                value = action.stateReadOnly(Optional.ofNullable(this.message), this.hints);
                break;
            }
            case PANIC: {
                value = action.panic(this.ex, Optional.ofNullable(this.message), this.hints);
                break;
            }
            default: {
                value = action.unknown(Optional.ofNullable(this.message), this.hints);
            }
        }

        /*
        碰到第一个异常提示就以异常结束.
         */
        for (Hint hint : this.hints) {
            if (hint.isException()) {
                throw new OqsEngineException((Throwable) hint.getValue());
            }
        }

        return value;
    }

    public String getMessage() {
        return message;
    }

    /**
     * 获取当前的提示列表.
     *
     * @return 提示列表.
     */
    public Collection<Hint> getHints() {
        if (this.hints == null) {
            this.hints = new LinkedList();
        }

        return hints;
    }

    public Optional<V> getValue() {
        return Optional.ofNullable(value);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof OqsEngineResult)) {
            return false;
        }
        OqsEngineResult<?> result = (OqsEngineResult<?>) o;
        return this.resultStatus == result.resultStatus;
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.resultStatus);
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("OperationResult{");
        sb.append("hints=").append(hints);
        sb.append(", message='").append(message).append('\'');
        sb.append(", resultStatus=").append(resultStatus);
        sb.append(", value=").append(value);
        sb.append('}');
        return sb.toString();
    }
}
