package com.xforceplus.ultraman.oqsengine.plus.master.mysql;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import com.xforceplus.ultraman.metadata.constants.SystemField;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.*;
import com.xforceplus.ultraman.metadata.values.IValue;
import com.xforceplus.ultraman.oqsengine.plus.common.iterator.DataIterator;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.MasterQueryResult;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.MasterStorageEntity;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.executor.ConditionQueryExecutor;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.executor.CreateExecutor;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.executor.DeleteExecutor;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.executor.UpdateExecutor;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.Operation;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.QueryResult;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.SystemColumn;
import com.xforceplus.ultraman.oqsengine.plus.storage.pojo.dto.EntityPackage;
import com.xforceplus.ultraman.oqsengine.plus.storage.pojo.dto.select.SelectConfig;
import com.xforceplus.ultraman.sdk.core.event.*;
import com.xforceplus.ultraman.sdk.core.utils.MasterStorageHelper;
import com.xforceplus.ultraman.sdk.infra.event.EventPublisher;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.apache.calcite.adapter.jdbc.JdbcSchema;
import org.apache.calcite.config.Lex;
import org.apache.calcite.jdbc.CalciteSchema;
import org.apache.calcite.plan.Context;
import org.apache.calcite.schema.Schema;
import org.apache.calcite.schema.SchemaPlus;
import org.apache.calcite.sql.parser.SqlParser;
import org.apache.calcite.tools.FrameworkConfig;
import org.apache.calcite.tools.Frameworks;
import org.apache.calcite.tools.RelBuilder;
import org.apache.commons.lang3.StringUtils;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ConnectionCallback;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.TransactionTemplate;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.metadata.values.DateTimeValue.ZONE_ID;


/**
 * Created by justin.xu on 03/2023.
 *
 * @since 1.8
 */
@Slf4j
public class SQLMasterStorage implements MasterStorage {

    private PlatformTransactionManager manager;

    private TransactionTemplate template;

    private JdbcTemplate jdbcTemplate;

    private DataSource masterDataSource;

    private EntityClassEngine engine;

    private ObjectMapper objectMapper;

    private ExecutorService masterDispatcher;

    //TODO maybe multi
    private SchemaPlus rootSchema;

    @Autowired(required = false)
    public EventPublisher publisher;

    private FrameworkConfig frameworkConfig;

    private Map<String, Schema> mapping = new ConcurrentHashMap<>();

    private long queryTimeout;

    private long refreshTime;

    private long initTime;

    private ScheduledExecutorService scheduler;

    public SQLMasterStorage(PlatformTransactionManager manager, DataSource masterDataSource, ObjectMapper objectMapper, ExecutorService masterDispatcher, EntityClassEngine engine, long queryTimeout, long refreshTime, long initTime) {
        this.manager = manager;
        this.masterDataSource = masterDataSource;
        this.objectMapper = objectMapper;
        this.masterDispatcher = masterDispatcher;
        this.queryTimeout = queryTimeout;
        this.engine = engine;
        scheduler = Executors.newScheduledThreadPool(1);
        this.refreshTime = refreshTime;
        this.initTime = initTime;
        this.template = new TransactionTemplate(manager);
        this.jdbcTemplate = new JdbcTemplate(masterDataSource);
        prepareSchema();

    }

    /**
     * get schema from database or datasource
     */
    private void prepareSchema() {
        rootSchema = CalciteSchema.createRootSchema(true, false).plus();


        JdbcSchema oqs = JdbcSchema.create(rootSchema, "oqs", masterDataSource, null, null);
        rootSchema.add("oqs", oqs);
        mapping.put("oqs", oqs);

        scheduler.scheduleAtFixedRate(() -> {
            Set<String> tableNames = oqs.getTableNames();
            log.info("Refresh JDBC schema");
        }, initTime, refreshTime, TimeUnit.SECONDS);


        Frameworks.ConfigBuilder configBuilder = Frameworks.newConfigBuilder();
        this.frameworkConfig = configBuilder.defaultSchema(rootSchema)
                .parserConfig(SqlParser.config().withLex(Lex.MYSQL).withCaseSensitive(false))
                .context(new Context() {
                    @Override
                    public <C> @Nullable C unwrap(Class<C> aClass) {
                        if (aClass == RelBuilder.Config.class) {
                            return (C) RelBuilder.Config.DEFAULT.withSimplify(false);
                        } else {
                            return null;
                        }
                    }
                })
                .build();
    }

    private void checkId(IEntity entity) throws SQLException {
        if (entity.id() == 0) {
            throw new SQLException("Invalid entity`s id.");
        }
    }

    @Override
    public boolean build(IEntity entity, IEntityClass entityClass, Map<String, Object> context) throws SQLException {
        EntityClassGroup group = engine.describe(entityClass, entityClass.profile());
        return writeProcessing(entity, group, Operation.BUILD, context);
    }

    @Override
    public void build(EntityPackage entityPackage, Map<String, Object> context) throws SQLException {
        packageWriteProcessing(entityPackage, Operation.BUILD, context);
    }

    @Override
    public boolean replace(IEntity entity, IEntityClass entityClass, Map<String, Object> context) throws SQLException {
        EntityClassGroup group = engine.describe(entityClass, entityClass.profile());
        return writeProcessing(entity, group, Operation.REPLACE, context);
    }

    @Override
    public void replace(EntityPackage entityPackage, Map<String, Object> context) throws SQLException {
        packageWriteProcessing(entityPackage, Operation.REPLACE, context);
    }

    @Override
    public boolean delete(IEntity entity, IEntityClass entityClass, Map<String, Object> context) throws SQLException {
        EntityClassGroup group = engine.describe(entityClass, entityClass.profile());
        return writeProcessing(entity, group, Operation.DELETE, context);
    }

    @Override
    public void delete(EntityPackage entityPackage, Map<String, Object> context) throws SQLException {
        packageWriteProcessing(entityPackage, Operation.DELETE, context);
    }

    @Override
    public DataIterator<MasterStorageEntity> iterator(IEntityClass entityClass, long startTime, long endTime, long lastId, boolean useSelfEntityClass) throws SQLException {
        return null;
    }

    @Override
    public DataIterator<MasterStorageEntity> iterator(IEntityClass entityClass, long startTime, long endTime, long lastId, int size, boolean useSelfEntityClass) throws SQLException {
        return null;
    }

    @Override
    public Collection<QueryResult> selectEntities(SelectConfig selectConfig, IEntityClass entityClass) throws SQLException {
        /**
         * TODO
         * current schema
         */
        selectConfig.setScheme(mapping.get("oqs"));
        return template.execute(status -> {
            return jdbcTemplate.execute(new ConnectionCallback<Collection<QueryResult>>() {
                @Override
                public Collection<QueryResult> doInConnection(Connection conn) throws SQLException, DataAccessException {
                    MasterQueryResult result = null;
                    result = ConditionQueryExecutor.build(conn
                            , queryTimeout, masterDataSource, engine.describe(entityClass, selectConfig.getProfile()), frameworkConfig).execute(selectConfig);
                    Collection<QueryResult> entities = new ArrayList<>();
                    if (!result.getValues().isEmpty()) {
                        List<String> fieldsNames = result.getFieldsNames();
                        for (List<Object> value : result.getValues()) {
                            QueryResult queryResult = new QueryResult(null);
                            List<QueryResult.SelectItem> items = new ArrayList<>();
                            queryResult.setSelectItems(items);
                            for (int i = 0; i < fieldsNames.size(); i++) {
                                String field = fieldsNames.get(i);
                                items.add(new QueryResult.SelectItem(field, field, value.get(i)));
                            }
                            entities.add(queryResult);
                        }
                    }
                    return entities;
                }
            });
        });
    }

    //    private QueryResult toQueryResult(MasterQueryEntity masterQueryEntity, IEntityClass entityClass, Object o) {
//        return null;
//    }

    //    @Override
    //    @Override
//    public Collection<QueryResult> selectMultiple(EntityId[] ids) throws SQLException {
//        return template.execute(status -> {
//
//            jdbcTemplate.execute(new ConnectionCallback< Collection<QueryResult>>() {
//
//                @Override
//                public Collection<QueryResult> doInConnection(Connection conn) throws SQLException, DataAccessException {
//                    IEntityClass entityClass = ids[0].getEntityClass();
//                    long[] executions = new long[ids.length];
//                    for (int i = 0; i < ids.length; i++) {
//                        executions[i] = ids[i].getId();
//                    }
//
//                    Collection<MasterQueryEntity> masterQueryEntities = MultiQueryPrimaryExecutor
//                            .build(conn, queryTimeout, entityClass, entityClass.code(), false).execute(executions);
//
//                    Map<String, List<SystemColumn>> middleParts = new HashMap<>();
//                    Map<Long, QueryResult> result = new ConcurrentHashMap<>();
//
//                    for (MasterQueryEntity masterQueryEntity : masterQueryEntities) {
//                        if (masterQueryEntity.getSystemColumn().getEntityClass() != entityClass.id()) {
//                            middleParts.computeIfAbsent(masterQueryEntity.getSystemColumn().getEntityClass() + ":" + masterQueryEntity.getSystemColumn().getProfile(), f -> new ArrayList<>()).add(masterQueryEntity.getSystemColumn());
//                        } else {
//                            result.put(masterQueryEntity.getSystemColumn().getId(), toQueryResult(masterQueryEntity, entityClass, null));
//                        }
//                    }
//
//                    if (!middleParts.isEmpty()) {
//                        CountDownLatch countDownLatch = new CountDownLatch(middleParts.size());
//                        middleParts.forEach((k, v) -> {
//                            masterDispatcher.submit(() -> {
//
//                                long[] parts = new long[v.size()];
//                                for (int i = 0; i < v.size(); i++) {
//                                    executions[i] = v.get(i).getId();
//                                }
//
//                                //  todo get entityClass
//                                IEntityClass realEntityClass = null;
////                                    load(v.get(0).getEntityClass(), v.get(0).getProfile());
//                                String tableName = realEntityClass.code() + "_view";
//
//                                //重新进行view查询.
//                                Collection<MasterQueryEntity> realEntities = null;
//                                try {
//                                    realEntities = (Collection<MasterQueryEntity>) MultiQueryPrimaryExecutor.build(conn, queryTimeout, realEntityClass, tableName, true).execute(parts);
//                                } catch (Exception e) {
//                                    e.printStackTrace();
//                                }
//
//                                if (!realEntities.isEmpty()) {
//                                    for (MasterQueryEntity entity : realEntities) {
//                                        result.put(entity.getSystemColumn().getId(), toQueryResult(entity, entityClass, null));
//                                    }
//                                }
//                                countDownLatch.countDown();
//                            });
//                        });
//                        masterDispatcher.awaitTermination(queryTimeout, TimeUnit.MILLISECONDS);
//
//                        Collection<QueryResult> results = new ArrayList<>(ids.length);
//                        for (EntityId entityId : ids) {
//                            QueryResult queryResult = result.get(entityId.getId());
//                            if (null != queryResult) {
//                                results.add(queryResult);
//                            }
//                        }
//                        return results;
//                    }
//
//                    return Collections.emptyList();
//                }
//            });
//        });
//    }
    private boolean checkResult(boolean[] rs) {
        if (null == rs) {
            return false;
        }

        for (boolean r : rs) {
            if (!r) {
                return false;
            }
        }
        return true;
    }

//    private QueryResult toQueryResult(MasterQueryEntity masterQueryEntity, IEntityClass entityClass, SelectConfig selectConfig) {
//
//        long entityClassId = masterQueryEntity.getSystemColumn().getEntityClass();
//        String profile = masterQueryEntity.getSystemColumn().getProfile();
//        IEntityClass realEntityClass;
//        QueryResult queryResult = new QueryResult(masterQueryEntity.getSystemColumn());
//        if(entityClassId != 0) {
//            if (entityClassId == entityClass.id()) {
//                realEntityClass = entityClass;
//            } else {
//                realEntityClass = engine.load(Long.toString(entityClassId), profile).orElseThrow(() -> new RuntimeException("Not Related class"));
//            }
//
//
//            addSelectItem(queryResult, masterQueryEntity, realEntityClass);
//        } else {
//            addSelectItem(queryResult, masterQueryEntity);
//        }
//
//        return queryResult;
//    }


//    private void addSelectItem(QueryResult queryResult, MasterQueryEntity masterQueryEntity, IEntityClass entityClass) {
//        masterQueryEntity.getBusinessStaticFields().forEach(
//                s -> {
//                    queryResult.getSelectItems().add(s);
//                }
//        );
//
//
//        Map<String, Object> vs = Optional.ofNullable(masterQueryEntity.getBusinessDynamicFields())
//                .map(MasterStorageHelper::buildFromJson)
//                .orElse(Collections.emptyMap());
//
//        for (Map.Entry<String, Object> v : vs.entrySet()) {
//            Optional<IEntityField> rs = entityClass.field(v.getKey());
//            rs.ifPresent(entityField -> queryResult.getSelectItems()
//                    .add(new QueryResult.SelectItem(entityField.name(), entityField.name(), v.getValue())));
//        }
//    }

//    private void addSelectItem(QueryResult queryResult, MasterQueryEntity masterQueryEntity) {
//        masterQueryEntity.getBusinessStaticFields().forEach(
//                s -> {
//                    queryResult.getSelectItems().add(s);
//                }
//        );
//    }

    private void publishAfterCreateEvent(IEntity entity, EntityClassGroup entityClassGroup, Map<String, Object> context) {
        if (publisher != null) {
            try {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass entityClass = entityClassGroup.getEntityClass();
                long id = entity.id();
                String code = entityClass.code();
                publisher.publishEvent(new EntityAfterCreate(code, id, toMap(entity), false, context));
            } catch (Throwable throwable) {
                log.error("", throwable);
            }
        }
    }


    private void publishBeforeCreateEvent(IEntity entity, EntityClassGroup entityClassGroup, Map<String, Object> context) {
        if (publisher != null) {
            try {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass entityClass = entityClassGroup.getEntityClass();
                long id = entity.id();
                String code = entityClass.code();
                publisher.publishEvent(new EntityBeforeCreate(code, id, false, toMap(entity), context));
            } catch (Throwable throwable) {
                log.error("", throwable);
            }
        }
    }

    private void publishAfterUpdateEvent(IEntity entity, EntityClassGroup entityClassGroup, Map<String, Object> context) {
        if (publisher != null) {
            try {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass entityClass = entityClassGroup.getEntityClass();
                long id = entity.id();
                String code = entityClass.code();
                //String code, Long id, Map<String, Object> oldData, Map<String, Object> data
                //            , boolean isInMulti, Map<String, Object> context
                publisher.publishEvent(new EntityAfterUpdate(code, id, Collections.emptyMap(), toMap(entity), false, context));
            } catch (Throwable throwable) {
                log.error("", throwable);
            }
        }
    }


    private void publishBeforeUpdateEvent(IEntity entity, EntityClassGroup entityClassGroup, Map<String, Object> context) {
        if (publisher != null) {
            try {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass entityClass = entityClassGroup.getEntityClass();
                long id = entity.id();
                String code = entityClass.code();
                publisher.publishEvent(new EntityBeforeUpdate(code, id, false, toMap(entity), context));
            } catch (Throwable throwable) {
                log.error("", throwable);
            }
        }
    }

    private void publishAfterDeleteEvent(IEntity entity, EntityClassGroup entityClassGroup, Map<String, Object> context) {
        if (publisher != null) {
            try {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass entityClass = entityClassGroup.getEntityClass();
                long id = entity.id();
                String code = entityClass.code();
                //String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                publisher.publishEvent(new EntityAfterDelete(code, id, Collections.emptyMap(), false, context));
            } catch (Throwable throwable) {
                log.error("", throwable);
            }
        }
    }

    private void publishBeforeDeleteEvent(IEntity entity, EntityClassGroup entityClassGroup, Map<String, Object> context) {
        if (publisher != null) {
            try {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass entityClass = entityClassGroup.getEntityClass();
                long id = entity.id();
                String code = entityClass.code();
                //String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                publisher.publishEvent(new EntityBeforeDelete(code, id, false, context));
            } catch (Throwable throwable) {
                log.error("", throwable);
            }
        }
    }

    private void publishBeforeMultiCreateEvent(EntityPackage entityPackage, Map<String, Object> context) {
        Map<Tuple2<Long, String>, List<Map.Entry<IEntity, IEntityClass>>> grouped = entityPackage.stream().collect(Collectors.groupingBy(x -> Tuple.of(x.getValue().id(), x.getValue().code())));

        grouped.forEach((k, v) -> {
            List<Map<String, Object>> bodies = new ArrayList<>();
            v.stream().forEach(x -> {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass targetEntityClass = x.getValue();
                if(publisher != null) {
                    Map<String, Object> body = toMap(x.getKey());
                    bodies.add(body);
                    publisher.publishEvent(new EntityBeforeCreate(targetEntityClass.code(), x.getKey().id(), true, body, context));
                }
            });

            if(!bodies.isEmpty() && publisher != null) {
                /**
                 * String code, List<Map<String, Object>> dataList, Map<String, Object> context
                 */
                publisher.publishEvent(new EntityBeforeMultiCreate(k._2, bodies, context));
            }
        });
    }

    private void publishAfterMultiCreateEvent(EntityPackage entityPackage, Map<String, Object> context) {
        Map<Tuple2<Long, String>, List<Map.Entry<IEntity, IEntityClass>>> grouped = entityPackage.stream()
                .collect(Collectors.groupingBy(x -> Tuple.of(x.getValue().id(), x.getValue().code())));

        grouped.forEach((k, v) -> {
            List<Map<String, Object>> bodies = new ArrayList<>();
            v.stream().forEach(x -> {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass targetEntityClass = x.getValue();
                if(publisher != null) {
                    Map<String, Object> body = toMap(x.getKey());
                    bodies.add(body);
                    publisher.publishEvent(new EntityAfterCreate(targetEntityClass.code(), x.getKey().id(), body, true, context));
                }
            });

            if(!bodies.isEmpty() && publisher != null) {
                /**
                 * String code, List<Map<String, Object>> dataList, Map<String, Object> context
                 */
                publisher.publishEvent(new EntityAfterMultiCreate(k._2, bodies, context));
            }
        });
    }


    private void publishBeforeMultiUpdateEvent(EntityPackage entityPackage, Map<String, Object> context) {
        Map<Tuple2<Long, String>, List<Map.Entry<IEntity, IEntityClass>>> grouped = entityPackage.stream()
                .collect(Collectors.groupingBy(x -> Tuple.of(x.getValue().id(), x.getValue().code())));

        grouped.forEach((k, v) -> {
            List<Map<String, Object>> bodies = new ArrayList<>();
            v.stream().forEach(x -> {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass targetEntityClass = x.getValue();
                if(publisher != null) {
                    Map<String, Object> body = toMap(x.getKey());
                    bodies.add(body);
                    publisher.publishEvent(new EntityBeforeUpdate(targetEntityClass.code()
                            , x.getKey().id(), true,  body, context));
                }
            });

            if(!bodies.isEmpty() && publisher != null) {
                /**
                 * String code, List<Map<String, Object>> dataList, Map<String, Object> context
                 */
                publisher.publishEvent(new EntityBeforeMultiUpdate(k._2, bodies, context));
            }
        });
    }

    private void publishAfterMultiUpdateEvent(EntityPackage entityPackage, Map<String, Object> context) {
        Map<Tuple2<Long, String>, List<Map.Entry<IEntity, IEntityClass>>> grouped = entityPackage.stream()
                .collect(Collectors.groupingBy(x -> Tuple.of(x.getValue().id(), x.getValue().code())));

        grouped.forEach((k, v) -> {
            List<Map<String, Object>> bodies = new ArrayList<>();
            v.stream().forEach(x -> {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass targetEntityClass = x.getValue();
                if(publisher != null) {
                    Map<String, Object> body = toMap(x.getKey());
                    bodies.add(body);
                    publisher.publishEvent(new EntityAfterUpdate(targetEntityClass.code(), x.getKey().id(), Collections.emptyMap(), body, true, context));
                }
            });

            if(!bodies.isEmpty() && publisher != null) {
                /**
                 * String code, List<Map<String, Object>> dataList, Map<String, Object> context
                 */
                //String code, List<Tuple2<Map<String, Object>, Map<String, Object>>> dataList, Map<String, Object> context
                publisher.publishEvent(new EntityAfterMultiUpdate(k._2, bodies.stream().map(x -> Tuple.of(Collections.<String, Object>emptyMap(), x))
                        .collect(Collectors.toList()), context));
            }
        });
    }


    private void publishAfterMultiDeletedEvent(EntityPackage entityPackage, Map<String, Object> context) {
        Map<Tuple2<Long, String>, List<Map.Entry<IEntity, IEntityClass>>> grouped = entityPackage.stream()
                .collect(Collectors.groupingBy(x -> Tuple.of(x.getValue().id(), x.getValue().code())));

        grouped.forEach((k, v) -> {
            List<Map<String, Object>> bodies = new ArrayList<>();
            v.stream().forEach(x -> {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass targetEntityClass = x.getValue();
                if(publisher != null) {
                    Map<String, Object> body = toMap(x.getKey());
                    bodies.add(body);
                    //String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                    publisher.publishEvent(new EntityAfterDelete(targetEntityClass.code(), x.getKey().id(), Collections.emptyMap(),true, context));
                }
            });

            if(!bodies.isEmpty() && publisher != null) {
                /**
                 * String code, List<Map<String, Object>> dataList, Map<String, Object> context
                 */
                //String code, List<Tuple2<Map<String, Object>, Map<String, Object>>> dataList, Map<String, Object> context
                publisher.publishEvent(new EntityAfterMultiDelete(k._2, bodies, context));
            }
        });
    }

    private void publishBeforeMultiDeleteEvent(EntityPackage entityPackage, Map<String, Object> context) {
        Map<Tuple2<Long, String>, List<Map.Entry<IEntity, IEntityClass>>> grouped = entityPackage.stream()
                .collect(Collectors.groupingBy(x -> Tuple.of(x.getValue().id(), x.getValue().code())));

        grouped.forEach((k, v) -> {
            List<Long> bodies = new ArrayList<>();
            v.stream().forEach(x -> {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass targetEntityClass = x.getValue();
                if(publisher != null) {
//                    Map<String, Object> body = toMap(x.getKey());
                    bodies.add(x.getKey().id());
                    //String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                    publisher.publishEvent(new EntityBeforeDelete(targetEntityClass.code(), x.getKey().id(), true, context));
                }
            });

            if(!bodies.isEmpty() && publisher != null) {
                /**
                 * String code, List<Map<String, Object>> dataList, Map<String, Object> context
                 */
                //String code, List<Tuple2<Map<String, Object>, Map<String, Object>>> dataList, Map<String, Object> context
                publisher.publishEvent(new EntityBeforeMultiDelete(k._2, bodies, context));
            }
        });
    }

    private Map<String, Object> toMap(IEntity entity) {
        IEntityValue iEntityValue = entity.entityValue();
        Map<String, Object> payload = new HashMap<>();
        if (iEntityValue != null) {
            iEntityValue.values().forEach(x -> {
                String name = x.getField().name();
                Object value = x.getValue();
                payload.put(name, value);
            });
        }

        return payload;
    }

    private boolean writeProcessing(IEntity entity, EntityClassGroup entityClass, Operation op, Map<String, Object> context) throws SQLException {
        checkId(entity);

        boolean result = template.execute(status -> {
            return jdbcTemplate.execute(new ConnectionCallback<Boolean>() {
                @Override
                public Boolean doInConnection(Connection con) throws SQLException, DataAccessException {
                    List<MasterStorageEntity> masterStorageEntities = toMasterStorageEntities(entityClass, entity, op);

                    boolean[] rs;
                    switch (op) {
                        case BUILD:
                            publishBeforeCreateEvent(entity, entityClass, context);
                            rs = CreateExecutor.build(con, queryTimeout).execute(masterStorageEntities);
                            publishAfterCreateEvent(entity, entityClass, context);
                            break;
                        case REPLACE:
                            publishBeforeUpdateEvent(entity, entityClass, context);
                            rs = UpdateExecutor.build(con, queryTimeout).execute(masterStorageEntities);
                            publishAfterUpdateEvent(entity, entityClass, context);
                            break;
                        case DELETE:
                            publishBeforeDeleteEvent(entity, entityClass, context);
                            rs = DeleteExecutor.build(con, queryTimeout).execute(masterStorageEntities);
                            publishAfterDeleteEvent(entity, entityClass, context);
                            break;
                        default:
                            rs = null;
                    }

                    if(publisher != null) {
                        Instant instant = LocalDateTime.now().atZone(ZONE_ID).toInstant();
                        long currentTime = instant.toEpochMilli();

                        EntityClassRef entityClassRef = entity.entityClassRef();

                        publisher.publishEvent(new EntityChanged(Collections
                                .singletonList(Tuple.of(entityClassRef.getProfile(), entityClassRef.getId())), currentTime));
                    }

                    return checkResult(rs);

                }
            });
        });

        if (result) {
            entity.neat();
        }


        return result;
    }

    private void packageWriteProcessing(EntityPackage entityPackage, Operation op, Map<String, Object> context) throws SQLException {
        template.executeWithoutResult(status -> {
            jdbcTemplate.execute(new ConnectionCallback<Boolean>() {
                @Override
                public Boolean doInConnection(Connection con) throws SQLException, DataAccessException {
                    List<MasterStorageEntity> executionPackages = new ArrayList<>();
                    entityPackage.stream().forEach(er -> {
                        EntityClassGroup group = engine.describe(er.getValue(), er.getValue().profile());
                        executionPackages.addAll(toMasterStorageEntities(group, er.getKey(), op));
                    });

                    List<List<MasterStorageEntity>> executionPart = Lists.partition(executionPackages, EntityPackage.MAX_SIZE);

                    if(publisher != null) {
                        switch (op) {
                            case BUILD:
                                publishBeforeMultiCreateEvent(entityPackage, context);
                                break;
                            case REPLACE:
                                publishBeforeMultiUpdateEvent(entityPackage, context);
                                break;
                            case DELETE:
                                publishBeforeMultiDeleteEvent(entityPackage, context);
                                break;
                        }
                    }

                    for (List<MasterStorageEntity> parts : executionPart) {
                        switch (op) {
                            case BUILD:
                                CreateExecutor.build(con, queryTimeout).execute(parts);
                                break;
                            case REPLACE:
                                UpdateExecutor.build(con, queryTimeout).execute(parts);
                                break;
                            case DELETE:
                                DeleteExecutor.build(con, queryTimeout).execute(parts);
                                break;
                            default:
                        }
                    }

                    if(publisher != null) {
                        //do publisher
                        List<Tuple2<String, Long>> list = entityPackage.stream().map(entry -> {
                            String profile = entry.getKey().entityClassRef().getProfile();
                            long entityClassId = entry.getKey().entityClassRef().getId();
                            return Tuple.of(profile, entityClassId);
                        }).collect(Collectors.toList());

                        Instant instant = LocalDateTime.now().atZone(ZONE_ID).toInstant();
                        long currentTime = instant.toEpochMilli();

                        publisher.publishEvent(new EntityChanged(list, currentTime));
                    }

                    switch (op) {
                        case BUILD:
                            publishAfterMultiCreateEvent(entityPackage, context);
                            break;
                        case REPLACE:
                            publishAfterMultiUpdateEvent(entityPackage, context);
                            break;
                        case DELETE:
                            publishAfterMultiDeletedEvent(entityPackage, context);
                            break;
                    }
                    return true;
                }
            });
        });
    }

    private List<MasterStorageEntity.TypedStorageValue> extractBocpSystemField(EntityClassGroup group, IEntity entity) {
        Collection<IEntityField> allFields = group.getAllFields();
        List<MasterStorageEntity.TypedStorageValue> values = new ArrayList<>();
        for (IEntityField entityField : allFields) {
            Optional<SystemField> systemField = Arrays.stream(SystemField.values()).filter(x -> x.getName().equalsIgnoreCase(entityField.name())).findFirst();
            if (systemField.isPresent()) {
                Optional<IValue> valueOp = entity.entityValue().getValue(entityField);
                if (valueOp.isPresent()) {
                    values.add(new MasterStorageEntity.TypedStorageValue(entityField, valueOp.get().storageValue()));
                }
            }
        }

        return values;
    }

    private MasterStorageEntity merge(MasterStorageEntity entity, List<MasterStorageEntity.TypedStorageValue> values) {
        MasterStorageEntity masterStorageEntity = new MasterStorageEntity();
        masterStorageEntity.setTableName(entity.getTableName());
        masterStorageEntity.setBusinessDynamicFields(entity.getBusinessDynamicFields());
        masterStorageEntity.setSystemColumn(entity.getSystemColumn());
        List<MasterStorageEntity.TypedStorageValue> mergedValues = new ArrayList<>(values);
        Optional.ofNullable(entity.getBusinessStaticFields())
                .orElseGet(Collections::emptyList).stream().filter(x -> {
                    return values.stream().noneMatch(v -> {
                        return v.getEntityField() == x.getEntityField();
                    });
                }).forEach(x -> {
                    mergedValues.add(x);
                });
        masterStorageEntity.setBusinessStaticFields(mergedValues);
        masterStorageEntity.setBusinessDynamicFieldsRemove(entity.getBusinessDynamicFieldsRemove());
        return masterStorageEntity;
    }

    private List<MasterStorageEntity> toMasterStorageEntities(EntityClassGroup group, IEntity entity, Operation op) {
        List<MasterStorageEntity> masterStorageEntities = new ArrayList<>();

        //extract system field and spread on every object
        List<MasterStorageEntity.TypedStorageValue> values = extractBocpSystemField(group, entity);

        //  加入自己.
        List<MasterStorageEntity> masterStorageEntity = toMasterStorageEntity(group.getEntityClass(), entity, op);

        masterStorageEntity.stream().map(x -> merge(x, values)).forEach(masterStorageEntities::add);

        //  加入父类.
        if (!group.getFatherEntityClass().isEmpty()) {
            Collection<IEntityClass> entityClasses = group.getFatherEntityClass();

            for (IEntityClass e : entityClasses) {
                List<MasterStorageEntity> multiMaster = toMasterStorageEntity(e, entity, op);
                multiMaster.stream().map(x -> merge(x, values)).forEach(masterStorageEntities::add);
            }
        }
        return masterStorageEntities;
    }

    /**
     * if profiled will get multi
     *
     * @param e
     * @param entity
     * @param op
     * @return
     */
    private List<MasterStorageEntity> toMasterStorageEntity(IEntityClass e, IEntity entity, Operation op) {
        String profile = e.realProfile();
        if (StringUtils.isEmpty(profile)) {
            Collection<IEntityField> selfFields = e.selfWithIndex();

            MasterStorageEntity masterStorageEntity = new MasterStorageEntity();
            //  设置系统字段
            SystemColumn systemColumn = new SystemColumn();
            systemColumn.setProfile(e.profile());
            systemColumn.setOperateTime(entity.time());
            systemColumn.setEntityClass(entity.entityClassRef().getId());
            systemColumn.setId(entity.id());
            masterStorageEntity.setSystemColumn(systemColumn);
            //TODO
            masterStorageEntity.setTableName(e.masterWriteTable());

            List<MasterStorageEntity.TypedStorageValue> staticValues = new ArrayList<>();

            //  delete操作不处理任何的字段值.
            if (!op.equals(Operation.DELETE)) {
                List<IValue> dynamicValues = new ArrayList<>();
                for (IEntityField entityField : selfFields) {
                    Optional<IValue> valueOp = entity.entityValue().getValue(entityField);
                    if (valueOp.isPresent()) {

                        if (entityField.isDynamic()) {
                            dynamicValues.add(valueOp.get());
                        } else {
                            staticValues.add(new MasterStorageEntity.TypedStorageValue(entityField, valueOp.get().storageValue()));
                        }
                    }
                }

                masterStorageEntity.setBusinessStaticFields(staticValues);

                //  设置动态.
                Map<String, Object> painValues = MasterStorageHelper.toPainValues(dynamicValues);

                if (op.equals(Operation.REPLACE)) {
                    masterStorageEntity.setBusinessDynamicFields(MasterStorageHelper.buildReplace(painValues));
                    masterStorageEntity.setBusinessDynamicFieldsRemove(MasterStorageHelper.buildRemove(painValues));
                } else {
                    masterStorageEntity.setBusinessDynamicFields(MasterStorageHelper.toBuildJson(painValues));
                }

                systemColumn.setDeleted(false);
            } else {
                systemColumn.setDeleted(true);
            }
            return Collections.singletonList(masterStorageEntity);
        } else {

            List<MasterStorageEntity> masterStorageEntities = new ArrayList<>();


            Collection<IEntityField> selfFields = e.selfWithIndex();

            List<IEntityField> pureSelfFields = new ArrayList<>();
            List<IEntityField> profileFields = new ArrayList<>();
            for (IEntityField selfField : selfFields) {
                String realProfile = selfField.realProfile();
                if (StringUtils.isEmpty(realProfile)) {
                    pureSelfFields.add(selfField);
                } else {
                    profileFields.add(selfField);
                }
            }

            if(!pureSelfFields.isEmpty()) {
                //deal with self

                MasterStorageEntity masterStorageEntity = new MasterStorageEntity();
                //  设置系统字段
                SystemColumn systemColumn = new SystemColumn();
                systemColumn.setProfile(profile);
                systemColumn.setOperateTime(entity.time());
                systemColumn.setEntityClass(e.id());
                systemColumn.setId(entity.id());
                masterStorageEntity.setSystemColumn(systemColumn);
                //TODO
                masterStorageEntity.setTableName(e.masterWriteTable());

                List<MasterStorageEntity.TypedStorageValue> staticValues = new ArrayList<>();

                //  delete操作不处理任何的字段值.
                if (!op.equals(Operation.DELETE)) {
                    List<IValue> dynamicValues = new ArrayList<>();
                    for (IEntityField entityField : pureSelfFields) {
                        Optional<IValue> valueOp = entity.entityValue().getValue(entityField);
                        if (valueOp.isPresent()) {

                            if (entityField.isDynamic()) {
                                dynamicValues.add(valueOp.get());
                            } else {
                                staticValues.add(new MasterStorageEntity.TypedStorageValue(entityField, valueOp.get().storageValue()));
                            }
                        }
                    }

                    masterStorageEntity.setBusinessStaticFields(staticValues);

                    //  设置动态.
                    Map<String, Object> painValues = MasterStorageHelper.toPainValues(dynamicValues);

                    if (op.equals(Operation.REPLACE)) {
                        masterStorageEntity.setBusinessDynamicFields(MasterStorageHelper.buildReplace(painValues));
                        masterStorageEntity.setBusinessDynamicFieldsRemove(MasterStorageHelper.buildRemove(painValues));
                    } else {
                        masterStorageEntity.setBusinessDynamicFields(MasterStorageHelper.toBuildJson(painValues));
                    }

                    systemColumn.setDeleted(false);
                } else {
                    systemColumn.setDeleted(true);
                }
                masterStorageEntities.add(masterStorageEntity);
            }

            if(!profileFields.isEmpty()){
                MasterStorageEntity masterStorageEntity = new MasterStorageEntity();
                //  设置系统字段
                SystemColumn systemColumn = new SystemColumn();
                systemColumn.setProfile(e.profile());
                systemColumn.setOperateTime(entity.time());
                systemColumn.setEntityClass(e.id());
                systemColumn.setId(entity.id());
                masterStorageEntity.setSystemColumn(systemColumn);
                //TODO
                masterStorageEntity.setTableName(e.masterWriteTable().concat("_").concat(profile));

                List<MasterStorageEntity.TypedStorageValue> staticValues = new ArrayList<>();

                //  delete操作不处理任何的字段值.
                if (!op.equals(Operation.DELETE)) {
                    List<IValue> dynamicValues = new ArrayList<>();
                    for (IEntityField entityField : profileFields) {
                        Optional<IValue> valueOp = entity.entityValue().getValue(entityField);
                        if (valueOp.isPresent()) {

                            if (entityField.isDynamic()) {
                                dynamicValues.add(valueOp.get());
                            } else {
                                staticValues.add(new MasterStorageEntity.TypedStorageValue(entityField, valueOp.get().storageValue()));
                            }
                        }
                    }

                    masterStorageEntity.setBusinessStaticFields(staticValues);

                    //  设置动态.
                    Map<String, Object> painValues = MasterStorageHelper.toPainValues(dynamicValues);

                    if (op.equals(Operation.REPLACE)) {
                        masterStorageEntity.setBusinessDynamicFields(MasterStorageHelper.buildReplace(painValues));
                        masterStorageEntity.setBusinessDynamicFieldsRemove(MasterStorageHelper.buildRemove(painValues));
                    } else {
                        masterStorageEntity.setBusinessDynamicFields(MasterStorageHelper.toBuildJson(painValues));
                    }

                    systemColumn.setDeleted(false);
                } else {
                    systemColumn.setDeleted(true);
                }

                masterStorageEntities.add(masterStorageEntity);
            }
            return masterStorageEntities;
        }
    }
}
