package com.xforceplus.ultraman.oqsengine.plus.master.iterator;

import com.xforceplus.ultraman.metadata.cdc.OqsEngineEntity;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.entity.EntityClassRef;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.helper.OriginEntityUtils;
import com.xforceplus.ultraman.oqsengine.plus.common.StringUtils;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.SystemColumn;


import javax.sql.DataSource;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

/**
 * Created by justin.xu on 09/2023.
 *
 * @since 1.8
 */
public class OriginEntityIterator extends EntityUpdateTimeRangeIterator<OqsEngineEntity> {


    private EntityClassEngine engine;

    @Override
    protected OqsEngineEntity buildFromResultSet(ResultSet rs) throws Exception {

        //  去除所有的父类.
        EntityClassRef entityClassRef = new EntityClassRef(
                rs.getLong(SystemColumn.SYS_ENTITY_CLASS), entityClass.appCode(), rs.getString(SystemColumn.SYS_PROFILE));

        if (!entityClass.ref().equals(entityClassRef)) {
            return null;
        }

        return OriginEntityUtils.generateFromResultSet(rs, entityClass, engine);
    }

    @Override
    protected PreparedStatement preparedSQL(String table, String profile) throws SQLException {
        StringBuilder sql = new StringBuilder();
        sql.append("SELECT * ");
        sql.append(" FROM ")
                .append(table)
                .append(" WHERE ")
                .append(SystemColumn.SYS_OPERATE_TIME).append(" >= ").append("?")
                .append(" AND ")
                .append(SystemColumn.SYS_OPERATE_TIME).append(" <= ").append("?")
                .append(" AND ")
                .append(SystemColumn.ID).append(" > ").append("?");

        if (null != profile && !profile.isEmpty()) {
            sql.append(" AND ").append(SystemColumn.SYS_PROFILE).append(" = ").append("?");
        }

        sql.append(" ORDER BY ").append(SystemColumn.ID);

        PreparedStatement ps =
                connection.prepareStatement(sql.toString(), ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        ps.setFetchSize(Integer.MIN_VALUE);
        int pos = 1;

        ps.setLong(pos++, startTime);
        ps.setLong(pos++, endTime);
        ps.setLong(pos++, startId);

        if (null != profile && !profile.isEmpty()) {
            ps.setString(pos++, entityClass.profile());
        }

        return ps;
    }


    /**
     * 构造器.
     */
    public static final class Builder {
        private int buffSize;
        private long startTime;
        private long endTime;
        private String tableName;
        private long startId = 0;

        private EntityClassEngine engine;

        private IEntityClass entityClass;
        private DataSource dataSource;

        private Builder() {
            buffSize = EntityUpdateTimeRangeIterator.DEFAULT_BUFFER_SIZE;
            startTime = -1;
            endTime = -1;
        }

        public static OriginEntityIterator.Builder anEntityIterator() {
            return new OriginEntityIterator.Builder();
        }

        public OriginEntityIterator.Builder witherBuffSize(int size) {
            this.buffSize = size;
            return this;
        }

        public OriginEntityIterator.Builder witherTableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public OriginEntityIterator.Builder withStartTime(long startTime) {
            this.startTime = startTime;
            return this;
        }

        public OriginEntityIterator.Builder withEndTime(long endTime) {
            this.endTime = endTime;
            return this;
        }

        public OriginEntityIterator.Builder withEntityClass(IEntityClass entityClass) {
            this.entityClass = entityClass;
            return this;
        }

        public OriginEntityIterator.Builder withDataSource(DataSource dataSource) {
            this.dataSource = dataSource;
            return this;
        }

        public OriginEntityIterator.Builder withStartId(long startId) {
            this.startId = startId;
            return this;
        }

        public OriginEntityIterator.Builder withEngine(EntityClassEngine engine) {
            this.engine = engine;
            return this;
        }

        /**
         * 构造对象时间范围迭代器.<br>
         * 以下值为必须.<br>
         * <ul>
         *     <li>dataSource</li>
         *     <li>entityClass</li>
         *     <li>startTime</li>
         *     <li>endTime</li>
         *     <li>tableName</li>
         *     <li>metaManager</li>
         * </ul>
         *
         * @return 迭代器实例.
         */
        public OriginEntityIterator build() {
            if (entityClass == null) {
                throw new RuntimeException("Invalid meta information.");
            }

            if (startTime < 0) {
                throw new RuntimeException("Invalid start time.");
            }

            if (endTime < 0) {
                throw new RuntimeException("Invalid end time.");
            }

            if (startTime > endTime) {
                throw new RuntimeException(
                        String.format("The start time cannot be later than the end time.[%d, %d]", startTime, endTime));
            }

            if (dataSource == null) {
                throw new RuntimeException("Invalid data source.");
            }

            if (engine == null) {
                throw new RuntimeException("Invalid engine.");
            }


            if (StringUtils.isEmpty(tableName)) {
                throw new RuntimeException("Invalid table name.");
            }

            OriginEntityIterator entityIterator = new OriginEntityIterator();
            entityIterator.buffSize = this.buffSize;
            entityIterator.startTime = this.startTime;
            entityIterator.endTime = this.endTime;
            entityIterator.dataSource = this.dataSource;
            entityIterator.entityClass = this.entityClass;
            entityIterator.tableName = this.tableName;
            entityIterator.startId = this.startId;
            entityIterator.engine = this.engine;

            try {
                entityIterator.init();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }

            return entityIterator;
        }
    }
}
