package com.xforceplus.ultraman.oqsengine.plus.master.iterator;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.xforceplus.ultraman.metadata.cdc.OqsEngineEntity;
import com.xforceplus.ultraman.metadata.entity.EntityClassRef;
import com.xforceplus.ultraman.metadata.entity.FieldType;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.oqsengine.plus.common.StringUtils;
import com.xforceplus.ultraman.oqsengine.plus.common.iterator.DataIterator;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.SystemColumn;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Created by justin.xu on 07/2023.
 *
 * @since 1.8
 */
public class EntityUpdateTimeRangeIterator implements DataIterator<OqsEngineEntity> {

    private static final int DEFAULT_BUFFER_SIZE = 1000;
    private int buffSize;
    private long startTime;
    private long endTime;
    private String tableName;
    private IEntityClass entityClass;
    private DataSource dataSource;
    private List<OqsEngineEntity> buffer;

    private Connection connection;
    private PreparedStatement ps;
    private ResultSet rs;

    @Override
    public void init() throws Exception {
        buffer = new ArrayList<>(buffSize);

        connection = dataSource.getConnection();
        String sql = buildSQL(tableName, entityClass.selfFields().stream().filter(f -> !f.isDynamic()).collect(Collectors.toList()));
        ps = connection.prepareStatement(sql, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        ps.setFetchSize(Integer.MIN_VALUE);
        int pos = 1;
        ps.setLong(pos++, entityClass.id());
        ps.setLong(pos++, startTime);
        ps.setLong(pos++, endTime);

        rs = ps.executeQuery();
    }

    @Override
    public void destroy() throws Exception {
        if (rs != null) {
            rs.close();
        }
        if (ps != null) {
            ps.close();
        }
        if (connection != null) {
            connection.close();
        }
    }

    @Override
    public boolean hasNext() {
        if (buffer.isEmpty()) {
            load();
        }

        return !buffer.isEmpty();
    }

    @Override
    public OqsEngineEntity next() {
        if (hasNext()) {
            return buffer.remove(0);
        } else {
            return null;
        }
    }

    private void load() throws RuntimeException {
        try {
            OqsEngineEntity entity;
            while (rs.next()) {
                OqsEngineEntity engineEntity = buildFormResultSet(rs);
                if (null != engineEntity) {
                    this.buffer.add(engineEntity);
                    if (this.buffer.size() == buffSize) {
                        break;
                    }
                }
            }
        } catch (Exception ex) {
            throw new RuntimeException(ex.getMessage(), ex);
        }
    }

    private OqsEngineEntity buildFormResultSet(ResultSet rs) throws Exception {

        //  去除所有的父类.
        EntityClassRef entityClassRef = new EntityClassRef(
                rs.getLong(SystemColumn.SYS_ENTITY_CLASS), entityClass.appCode(), rs.getString(SystemColumn.SYS_PROFILE));

        if (!entityClass.ref().equals(entityClassRef)) {
            return null;
        }


        OqsEngineEntity entity = new OqsEngineEntity();
        entity.setId(rs.getLong(SystemColumn.ID));
        entity.setUpdateTime(rs.getLong(SystemColumn.SYS_OPERATE_TIME));
        entity.setDeleted(false);
        entity.setVersion(rs.getInt(SystemColumn.SYS_VERSION));
        entity.setEntityClassRef(new EntityClassRef(entityClass.id(), entityClass.appCode(), rs.getString(SystemColumn.SYS_PROFILE)));

        Map<String, Object> attrs = new HashMap<>();
        attrs.put(SystemColumn.ID, entity.getId());

        String dynamic = rs.getString(SystemColumn.DYNAMIC_FIELD);
        if (!dynamic.isEmpty()) {
            try {
                Map<String, Object> attrDynamics = OqsEngineEntity.addDynamicAttrWithNullValue(entityClass,
                        dynamic);

                attrs.putAll(attrDynamics);
            } catch (JsonProcessingException e) {
                throw new SQLException(e);
            }
        }

        for (IEntityField entityField :
                entityClass.selfFields().stream().filter(f -> !f.isDynamic()).collect(Collectors.toList())) {
            String name = entityField.name().toLowerCase();
            Object value = rs.getObject(name);
            if (value == null || !entityField.type().equals(FieldType.BOOLEAN)) {
                attrs.put(name, value);
            } else {
                attrs.put(name, (int) value > 0);
            }
        }
        entity.setAttributes(attrs);

        return entity;
    }

    private String buildSQL(String table, Collection<IEntityField> entityFields) {
        StringBuilder sql = new StringBuilder();
        sql.append("SELECT ");
        sql.append(String.join(",",
                        SystemColumn.ID,
                        SystemColumn.SYS_ENTITY_CLASS,
                        SystemColumn.SYS_OPERATE_TIME,
                        SystemColumn.SYS_VERSION,
                        SystemColumn.SYS_DELETED,
                        SystemColumn.SYS_PROFILE
                )
        );

        entityFields.stream().filter(e -> {
            return !e.isDynamic();
        }).forEach(
                e -> {
                    sql.append(",").append(e.name().toLowerCase());
                }
        );

        sql.append(SystemColumn.DYNAMIC_FIELD);

        sql.append(" FROM ")
                .append(table)
                .append(" WHERE ")
                .append(SystemColumn.SYS_OPERATE_TIME).append(" >= ").append("?")
                .append(" AND ")
                .append(SystemColumn.SYS_OPERATE_TIME).append(" <= ").append("?")
                .append(") AND ")
                .append(SystemColumn.ID).append(" > ").append("?");

        return sql.toString();
    }

    /**
     * 构造器.
     */
    public static final class Builder {
        private int buffSize;
        private long startTime;
        private long endTime;
        private String tableName;
        private IEntityClass entityClass;
        private DataSource dataSource;

        private Builder() {
            buffSize = EntityUpdateTimeRangeIterator.DEFAULT_BUFFER_SIZE;
            startTime = -1;
            endTime = -1;
        }

        public static Builder anEntityIterator() {
            return new Builder();
        }

        public Builder witherBuffSize(int size) {
            this.buffSize = size;
            return this;
        }

        public Builder witherTableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public Builder withStartTime(long startTime) {
            this.startTime = startTime;
            return this;
        }

        public Builder withEndTime(long endTime) {
            this.endTime = endTime;
            return this;
        }

        public Builder withEntityClass(IEntityClass entityClass) {
            this.entityClass = entityClass;
            return this;
        }

        public Builder withDataSource(DataSource dataSource) {
            this.dataSource = dataSource;
            return this;
        }

        /**
         * 构造对象时间范围迭代器.<br>
         * 以下值为必须.<br>
         * <ul>
         *     <li>dataSource</li>
         *     <li>entityClass</li>
         *     <li>startTime</li>
         *     <li>endTime</li>
         *     <li>tableName</li>
         *     <li>metaManager</li>
         * </ul>
         *
         * @return 迭代器实例.
         */
        public EntityUpdateTimeRangeIterator build() {
            if (entityClass == null) {
                throw new RuntimeException("Invalid meta information.");
            }

            if (startTime < 0) {
                throw new RuntimeException("Invalid start time.");
            }

            if (endTime < 0) {
                throw new RuntimeException("Invalid end time.");
            }

            if (startTime > endTime) {
                throw new RuntimeException(
                        String.format("The start time cannot be later than the end time.[%d, %d]", startTime, endTime));
            }

            if (dataSource == null) {
                throw new RuntimeException("Invalid data source.");
            }


            if (StringUtils.isEmpty(tableName)) {
                throw new RuntimeException("Invalid table name.");
            }

            EntityUpdateTimeRangeIterator entityIterator = new EntityUpdateTimeRangeIterator();
            entityIterator.buffSize = this.buffSize;
            entityIterator.startTime = this.startTime;
            entityIterator.endTime = this.endTime;
            entityIterator.dataSource = this.dataSource;
            entityIterator.entityClass = this.entityClass;
            entityIterator.tableName = this.tableName;

            try {
                entityIterator.init();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }

            return entityIterator;
        }
    }
}
