package com.xforceplus.ultraman.oqsengine.plus.master.iterator;

import com.xforceplus.ultraman.metadata.entity.EntityClassRef;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.plus.common.StringUtils;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.SystemColumn;

import javax.sql.DataSource;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

/**
 * Created by justin.xu on 09/2023.
 *
 * @since 1.8
 */
public class IdIterator extends EntityUpdateTimeRangeIterator<Long> {

    @Override
    protected Long buildFromResultSet(ResultSet rs) throws Exception {

//        //  去除所有的父类.
//        EntityClassRef entityClassRef = new EntityClassRef(
//                rs.getLong(SystemColumn.SYS_ENTITY_CLASS), entityClass.appCode(), entityClass.code(), rs.getString(SystemColumn.SYS_PROFILE));
//
//        if (!entityClass.ref().equals(entityClassRef)) {
//            return null;
//        }

        return rs.getLong(SystemColumn.ID);
    }

    @Override
    protected PreparedStatement preparedSQL(String table, String profile) throws SQLException {
        StringBuilder sql = new StringBuilder();
        sql.append("SELECT ");
        sql.append(SystemColumn.ID).append(" , ").append(SystemColumn.SYS_ENTITY_CLASS).append(" , ").append(SystemColumn.SYS_PROFILE);
        sql.append(" FROM ")
                .append(table)
                .append(" WHERE ")
                .append(SystemColumn.SYS_OPERATE_TIME).append(" >= ").append("?")
                .append(" AND ")
                .append(SystemColumn.SYS_OPERATE_TIME).append(" <= ").append("?")
                .append(" AND ")
                .append(SystemColumn.ID).append(" > ").append("?");

        if (null != profile && !profile.isEmpty()) {
            sql.append(" AND ").append(SystemColumn.SYS_PROFILE).append(" = ").append("?");
        }

        sql.append(" ORDER BY ").append(SystemColumn.ID);

        PreparedStatement ps =
                connection.prepareStatement(sql.toString(), ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
        ps.setFetchSize(Integer.MIN_VALUE);
        int pos = 1;

        ps.setLong(pos++, startTime);
        ps.setLong(pos++, endTime);
        ps.setLong(pos++, startId);

        if (null != profile && !profile.isEmpty()) {
            ps.setString(pos++, entityClass.profile());
        }

        return ps;
    }

    /**
     * 构造器.
     */
    public static final class Builder {
        private int buffSize;
        private long startTime;
        private long endTime;
        private String tableName;
        private long startId = 0;

        private IEntityClass entityClass;
        private DataSource dataSource;

        private Builder() {
            buffSize = EntityUpdateTimeRangeIterator.DEFAULT_BUFFER_SIZE;
            startTime = -1;
            endTime = -1;
        }

        public static Builder anEntityIterator() {
            return new Builder();
        }

        public Builder witherBuffSize(int size) {
            this.buffSize = size;
            return this;
        }

        public Builder witherTableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public Builder withStartTime(long startTime) {
            this.startTime = startTime;
            return this;
        }

        public Builder withEndTime(long endTime) {
            this.endTime = endTime;
            return this;
        }

        public Builder withEntityClass(IEntityClass entityClass) {
            this.entityClass = entityClass;
            return this;
        }

        public Builder withDataSource(DataSource dataSource) {
            this.dataSource = dataSource;
            return this;
        }

        public Builder withStartId(long startId) {
            this.startId = startId;
            return this;
        }

        /**
         * 构造对象时间范围迭代器.<br>
         * 以下值为必须.<br>
         * <ul>
         *     <li>dataSource</li>
         *     <li>entityClass</li>
         *     <li>startTime</li>
         *     <li>endTime</li>
         *     <li>tableName</li>
         *     <li>metaManager</li>
         * </ul>
         *
         * @return 迭代器实例.
         */
        public IdIterator build() {
            if (entityClass == null) {
                throw new RuntimeException("Invalid meta information.");
            }

            if (startTime < 0) {
                throw new RuntimeException("Invalid start time.");
            }

            if (endTime < 0) {
                throw new RuntimeException("Invalid end time.");
            }

            if (startTime > endTime) {
                throw new RuntimeException(
                        String.format("The start time cannot be later than the end time.[%d, %d]", startTime, endTime));
            }

            if (dataSource == null) {
                throw new RuntimeException("Invalid data source.");
            }


            if (StringUtils.isEmpty(tableName)) {
                throw new RuntimeException("Invalid table name.");
            }

            IdIterator entityIterator = new IdIterator();
            entityIterator.buffSize = this.buffSize;
            entityIterator.startTime = this.startTime;
            entityIterator.endTime = this.endTime;
            entityIterator.dataSource = this.dataSource;
            entityIterator.entityClass = this.entityClass;
            entityIterator.tableName = this.tableName;
            entityIterator.startId = this.startId;

            try {
                entityIterator.init();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }

            return entityIterator;
        }
    }
}
