package com.xforceplus.ultraman.oqsengine.plus.master.mysql;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.metadata.cdc.OqsEngineEntity;
import com.xforceplus.ultraman.metadata.constants.SystemField;
import com.xforceplus.ultraman.metadata.domain.vo.dto.ConditionOp;
import com.xforceplus.ultraman.metadata.domain.vo.dto.DictItem;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.*;
import com.xforceplus.ultraman.metadata.service.DictService;
import com.xforceplus.ultraman.metadata.values.EnumsValue;
import com.xforceplus.ultraman.metadata.values.ExpressionValue;
import com.xforceplus.ultraman.metadata.values.IValue;
import com.xforceplus.ultraman.metadata.values.OnDuplicateValue;
import com.xforceplus.ultraman.oqsengine.plus.common.datasource.log.LoggerDataSource;
import com.xforceplus.ultraman.oqsengine.plus.common.iterator.DataIterator;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.ConditionalStorageEntity;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.ConditionalStoragePackage;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.MasterQueryResult;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.MasterStorageEntity;
import com.xforceplus.ultraman.oqsengine.plus.master.iterator.IdIterator;
import com.xforceplus.ultraman.oqsengine.plus.master.iterator.OriginEntityIterator;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.calcite.ConditionalSqlShuttle;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.devops.BatchDeleteExecutor;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.devops.RebuildExecutor;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.executor.*;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.Operation;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.QueryResult;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.SystemColumn;
import com.xforceplus.ultraman.oqsengine.plus.storage.pojo.dto.EntityPackage;
import com.xforceplus.ultraman.oqsengine.plus.storage.pojo.dto.select.SelectConfig;
import com.xforceplus.ultraman.sdk.core.config.ExecutionConfig;
import com.xforceplus.ultraman.sdk.core.datasource.PackageInternal;
import com.xforceplus.ultraman.sdk.core.datasource.route.dynamic.DynamicDataSource;
import com.xforceplus.ultraman.sdk.core.event.*;
import com.xforceplus.ultraman.sdk.core.rel.legacy.*;
import com.xforceplus.ultraman.sdk.core.rel.utils.ExpTreeToRel;
import com.xforceplus.ultraman.sdk.core.utils.MasterStorageHelper;
import com.xforceplus.ultraman.sdk.core.config.CdcConfig;
import com.xforceplus.ultraman.sdk.infra.codec.MySQLCodecCustom;
import com.xforceplus.ultraman.sdk.infra.event.DDLChangedEvent;
import com.xforceplus.ultraman.sdk.infra.event.EventPublisher;
import com.xforceplus.ultraman.sdk.infra.logging.LoggingPattern;
import com.xforceplus.ultraman.sdk.infra.metrics.MetricsDefine;
import com.zaxxer.hikari.HikariDataSource;
import io.micrometer.core.annotation.Timed;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import io.vavr.Tuple3;
import lombok.extern.slf4j.Slf4j;
import org.apache.calcite.adapter.jdbc.JdbcSchema;
import org.apache.calcite.config.Lex;
import org.apache.calcite.jdbc.CalciteSchema;
import org.apache.calcite.plan.Context;
import org.apache.calcite.rel.RelNode;
import org.apache.calcite.schema.Schema;
import org.apache.calcite.schema.SchemaPlus;
import org.apache.calcite.sql.parser.SqlParser;
import org.apache.calcite.tools.FrameworkConfig;
import org.apache.calcite.tools.Frameworks;
import org.apache.calcite.tools.RelBuilder;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.math3.analysis.function.Exp;
import org.apache.shardingsphere.driver.jdbc.core.connection.ShardingSphereConnection;
import org.apache.shardingsphere.driver.jdbc.core.datasource.ShardingSphereDataSource;
import org.owasp.esapi.codecs.Codec;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.event.EventListener;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ConnectionCallback;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.TransactionTemplate;
import org.apache.commons.lang3.tuple.Pair;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.ultraman.sdk.core.utils.MasterStorageHelper.buildFromJson;
import static com.xforceplus.ultraman.sdk.core.utils.MasterStorageHelper.toBuildJson;
import static com.xforceplus.ultraman.sdk.infra.logging.LoggingUtils.logErrorPattern;


/**
 * Created by justin.xu on 03/2023.
 *
 * @since 1.8
 */
@Slf4j
public class SQLMasterStorage implements MasterStorage {

    @Autowired(required = false)
    public EventPublisher publisher;

    @Lazy
    @Autowired
    private FrameworkConfig config;

    private PlatformTransactionManager manager;
    private TransactionTemplate template;
    private JdbcTemplate jdbcTemplate;

    private DynamicDataSource dynamicDataSource;

    private DataSource masterDataSource;

    private EntityClassEngine engine;
    private ObjectMapper objectMapper;
    private Map<String, OQSFrameworkConfig> oqsFrameworkConfigMap = new HashMap<>();
    private SchemaPlus rootSchema;

    private Codec<Character> mySQLCodec = new MySQLCodecCustom(MySQLCodecCustom.Mode.STANDARD);

    private FrameworkConfig frameworkConfig;

    private Map<String, Schema> mapping = new ConcurrentHashMap<>();

    private OQSFrameworkConfig defaultConfig;

    private long queryTimeout;

    private long refreshTime;

    private long initTime;

    private ScheduledExecutorService scheduler;

    private CdcConfig cdcConfig;

    private DictService dictService;

    private ExecutionConfig executionConfig;
    
    
    private ContextService contextService;

    private final Counter inserCountTotal = Metrics.counter(MetricsDefine.WRITE_COUNT_TOTAL, "action", "build");
    private final Counter replaceCountTotal = Metrics.counter(MetricsDefine.WRITE_COUNT_TOTAL, "action", "replace");
    private final Counter deleteCountTotal = Metrics.counter(MetricsDefine.WRITE_COUNT_TOTAL, "action", "delete");
    
    public SQLMasterStorage(PlatformTransactionManager manager, DataSource dataSource
            , ObjectMapper objectMapper
            , EntityClassEngine engine, DictService dictService, CdcConfig cdcConfig
            , ExecutionConfig executionConfig, long queryTimeout, long refreshTime, long initTime, ContextService contextService) {
        this.manager = manager;
        if (dataSource instanceof DynamicDataSource) {
            this.dynamicDataSource = (DynamicDataSource) dataSource;
        } else {
            this.masterDataSource = dataSource;
        }
        this.objectMapper = objectMapper;
        this.queryTimeout = queryTimeout;
        this.engine = engine;
        this.scheduler = Executors.newScheduledThreadPool(1);
        this.refreshTime = refreshTime;
        this.initTime = initTime;
        this.template = new TransactionTemplate(manager);
        this.cdcConfig = cdcConfig;
        this.jdbcTemplate = new JdbcTemplate(dynamicDataSource);
        this.dictService = dictService;
        this.executionConfig = executionConfig;
        this.contextService = contextService;

        prepareSchema();
    }

    public void setPublisher(EventPublisher publisher) {
        this.publisher = publisher;
    }

    public void setConfig(FrameworkConfig config) {
        this.config = config;
    }

    /**
     * get schema from database or datasource
     */
    private void prepareSchema() {
        PackageInternal packageInternal = dynamicDataSource.allMaster();
        packageInternal.getResourceMapping().forEach(
                (k, v) -> {
                    oqsFrameworkConfigMap.put((String) k, new OQSFrameworkConfig((DataSource) v));
                }
        );
        defaultConfig = new OQSFrameworkConfig((DataSource) dynamicDataSource.allMaster().getFirst());

        scheduler.scheduleAtFixedRate(() -> {
            try {
                defaultConfig.getOqsSchema().getTableNames();
            } catch (Throwable throwable) {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "GetDataSourceSchema", throwable);
            }

            oqsFrameworkConfigMap.forEach((k, v) -> {
                try {
                    v.getOqsSchema().getTableNames();
                } catch (Throwable throwable) {
                    logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "RefreshDataSourceSchema", throwable);
                }
            });
            log.debug("Trigger RefreshDataSourceSchema");
        }, initTime, refreshTime, TimeUnit.SECONDS);
    }

    private void checkId(IEntity entity) throws SQLException {
        if (entity.id() == 0) {
            throw new SQLException("Invalid entity`s id.");
        }
    }
    
    @EventListener(DDLChangedEvent.class)
    public void refreshDDL(DDLChangedEvent evt){
        try {
            defaultConfig.getOqsSchema().getTableNames();
        } catch (Throwable throwable) {
            log.error("", throwable);
        }
    }

    @Timed(
            value = MetricsDefine.PROCESS_DELAY_LATENCY_SECONDS,
            percentiles = {0.5, 0.9, 0.99},
            extraTags = {"initiator", "master", "action", "build"})
    @Override
    public boolean build(IEntity entity, IEntityClass entityClass, Map<String, Object> context) throws SQLException {
        EntityClassGroup group = engine.describe(entityClass, entityClass.profile());
        return writeProcessing(entity, group, Operation.BUILD, context);
    }

    @Timed(
            value = MetricsDefine.PROCESS_DELAY_LATENCY_SECONDS,
            percentiles = {0.5, 0.9, 0.99},
            extraTags = {"initiator", "master", "action", "builds"})
    @Override
    public Boolean[] build(EntityPackage entityPackage, Map<String, Object> context) throws SQLException {
        return packageWriteProcessing(entityPackage, Operation.BUILD, context);
    }

    @Timed(
            value = MetricsDefine.PROCESS_DELAY_LATENCY_SECONDS,
            percentiles = {0.5, 0.9, 0.99},
            extraTags = {"initiator", "master", "action", "replace"}
    )
    @Override
    public boolean replace(IEntity entity, IEntityClass entityClass
            , Map<String, Object> context) throws SQLException {
        EntityClassGroup group = engine.describe(entityClass, entityClass.profile());
        return writeProcessing(entity, group, Operation.REPLACE, context);
    }

    @Timed(
            value = MetricsDefine.PROCESS_DELAY_LATENCY_SECONDS,
            percentiles = {0.5, 0.9, 0.99},
            extraTags = {"initiator", "master", "action", "replaces"}
    )
    @Override
    public Boolean[] replace(EntityPackage entityPackage, Map<String, Object> context) throws SQLException {
        return packageWriteProcessing(entityPackage, Operation.REPLACE, context);
    }

    @Override
    public int replaceByCondition(EntityPackage entityPackage, Map<String, Object> context) throws SQLException {
        return conditionalWriteProcessing(entityPackage, Operation.REPLACE_CON, context);
    }

    @Override
    public int deleteByCondition(EntityPackage entityPackage, Map<String, Object> context) throws SQLException {
        return conditionalWriteProcessing(entityPackage, Operation.DELETE_CON, context);
    }

    @Timed(
            value = MetricsDefine.PROCESS_DELAY_LATENCY_SECONDS,
            percentiles = {0.5, 0.9, 0.99},
            extraTags = {"initiator", "master", "action", "delete"})
    @Override
    public boolean delete(IEntity entity, IEntityClass entityClass, Map<String, Object> context) throws SQLException {
        EntityClassGroup group = engine.describe(entityClass, entityClass.profile());
        return writeProcessing(entity, group, Operation.DELETE, context);
    }

    @Timed(
            value = MetricsDefine.PROCESS_DELAY_LATENCY_SECONDS,
            percentiles = {0.5, 0.9, 0.99},
            extraTags = {"initiator", "master", "action", "deletes"}
    )
    @Override
    public Boolean[] delete(EntityPackage entityPackage, Map<String, Object> context) throws SQLException {
        return packageWriteProcessing(entityPackage, Operation.DELETE, context);
    }

    @Override
    public DataIterator<Long> iterator(IEntityClass entityClass, long startTime, long endTime, long lastId, int size, boolean useSelfEntityClass) throws SQLException {

        dynamicDataSource.resetKey(entityClass.profile());

        return IdIterator.Builder.anEntityIterator()
                .withDataSource(dynamicDataSource.currentDataSource())
                .withEntityClass(entityClass)
                .withStartTime(startTime)
                .withEndTime(endTime)
                .witherTableName(entityClass.masterQueryTable())
                .witherBuffSize(size)
                .withStartId(lastId)
                .build();
    }

    @Override
    public DataIterator<OqsEngineEntity> iteratorEntity(IEntityClass entityClass, long startTime, long endTime, long lastId, int size, String profile) throws SQLException {

        dynamicDataSource.resetKey(entityClass.realProfile());

        String table = entityClass.masterQueryTable();

        return OriginEntityIterator.Builder.anEntityIterator()
                .withDataSource(dynamicDataSource.currentDataSource())
                .withEntityClass(entityClass)
                .withStartTime(startTime)
                .withEndTime(endTime)
                .witherTableName(table)
                .witherBuffSize(size)
                .withEngine(engine)
                .withFilterProfile(profile)
                .withStartId(lastId)
                .build();
    }

    @Override
    public DataIterator<OqsEngineEntity> iteratorEntityHistory(IEntityClass entityClass, String profile, String table, long startTime, long endTime,
                                                                  long lastId, int size) throws SQLException {
        dynamicDataSource.resetKey(profile);

        String filterProfile = null == profile || profile.isEmpty() ? entityClass.realProfile() : profile;

        /**
         * 查询主表中的特定租户的数据时，需要将table设置为主表，否则视为查询租户定制表.
         */
        return OriginEntityIterator.Builder.anEntityIterator()
                .withDataSource(dynamicDataSource.currentDataSource())
                .withEntityClass(entityClass)
                .withStartTime(startTime)
                .withEndTime(endTime)
                .witherTableName(table)
                .witherBuffSize(size)
                .withEngine(engine)
                .withStartId(lastId)
                .withFilterProfile(filterProfile)
                .build();
    }

    @Override
    public DataIterator<Long> iteratorIdHistory(IEntityClass entityClass, String profile, String table, long startTime, long endTime, long lastId, int size) throws SQLException {
        dynamicDataSource.resetKey(profile);

        /**
         * 查询主表中的特定租户的数据时，需要将table设置为主表，否则视为查询租户定制表.
         */
        return IdIterator.Builder.anEntityIterator()
                .withDataSource(dynamicDataSource.currentDataSource())
                .withEntityClass(entityClass)
                .withStartTime(startTime)
                .withEndTime(endTime)
                .witherTableName(table)
                .witherBuffSize(size)
                .withStartId(lastId)
                .build();
    }

    @Override
    public boolean rebuildIndex(IEntityClass entityClass, List<Long> ids) throws SQLException {
        dynamicDataSource.resetKey(entityClass.profile());
        return jdbcTemplate.execute(new ConnectionCallback<Boolean>() {
            @Override
            public Boolean doInConnection(Connection con) throws SQLException, DataAccessException {
                Boolean[] rs = RebuildExecutor
                        .build(entityClass, con, 30000)
                        .execute(ids);

                boolean result = true;
                if (null != rs) {
                    for (boolean r : rs) {
                        if (!r) {
                            result = false;
                            break;
                        }
                    }
                }

                return result;
            }
        });
    }
    
    public Map<Long, Integer> selectSysColumn(List<Long> ids, IEntityClass entityClass) throws SQLException {
        OQSFrameworkConfig temp = oqsFrameworkConfigMap.get(dynamicDataSource.currentKey());
        final OQSFrameworkConfig oqsFrameworkConfig = null == temp ? defaultConfig : temp;

        SelectConfig selectConfig = SelectConfig.Builder.anSelectConfig().withIds(ids).build();
        return jdbcTemplate.execute(new ConnectionCallback<Map<Long, Integer>>() {
            @Override
            public Map<Long, Integer> doInConnection(Connection con) throws SQLException, DataAccessException {
                QuerySystemVerExecutor querySystemVerExecutor = new QuerySystemVerExecutor();
                Map<Long, Integer> execute = querySystemVerExecutor.execute(selectConfig, con, entityClass);
                return execute;
            }
        });
    }

    public Map<String, List<Tuple3<Long, String, Long>>> selectSysKindColumn(List<Long> ids, IEntityClass entityClass) throws SQLException {
        OQSFrameworkConfig temp = oqsFrameworkConfigMap.get(dynamicDataSource.currentKey());
        final OQSFrameworkConfig oqsFrameworkConfig = null == temp ? defaultConfig : temp;

        SelectConfig selectConfig = SelectConfig.Builder.anSelectConfig().withIds(ids).build();
        return jdbcTemplate.execute(new ConnectionCallback<Map<String, List<Tuple3<Long, String, Long>>>>() {
            @Override
            public Map<String, List<Tuple3<Long, String, Long>>> doInConnection(Connection con) throws SQLException, DataAccessException {
                QuerySubSystemExecutor querySystemVerExecutor = new QuerySubSystemExecutor(engine);
                Map<String, List<Tuple3<Long, String, Long>>> tableMapping = querySystemVerExecutor.execute(selectConfig, con, entityClass);
                return tableMapping;
            }
        });
    }

    @Override
    public boolean documentation(Collection<OqsEngineEntity> oqsEngineEntities, IEntityClass entityClass, String profile, String table) {
        dynamicDataSource.resetKey(profile);

        boolean result = template.execute(status -> {
            return jdbcTemplate.execute(new ConnectionCallback<Boolean>() {
                @Override
                public Boolean doInConnection(Connection con) throws SQLException, DataAccessException {
                    List<MasterStorageEntity> masterStorageEntities = toMasterDocumentEntity(entityClass, table, oqsEngineEntities);

                    Boolean[] rs = ReplaceExecutor.build(con, queryTimeout).execute(masterStorageEntities);

                    return checkResult(rs);
                }
            });
        });

        return result;
    }

    @Override
    public int batchDelete(Collection<Long> ids, IEntityClass entityClass, long startTime, long endTime, String requestProfile, String table) {
        dynamicDataSource.resetKey(requestProfile);

        return template.execute(status -> {
            return jdbcTemplate.execute(new ConnectionCallback<Integer>() {
                @Override
                public Integer doInConnection(Connection con) throws SQLException, DataAccessException {
                    Boolean[] rs = BatchDeleteExecutor.build(con, queryTimeout, table, startTime, endTime).execute(ids);

                    return checkResultCount(rs);
                }
            });
        });
    }

    @Override
    public Collection<QueryResult> selectEntities(SelectConfig selectConfig, IEntityClass entityClass) throws SQLException {

        OQSFrameworkConfig temp = oqsFrameworkConfigMap.get(dynamicDataSource.currentKey());
        final OQSFrameworkConfig oqsFrameworkConfig = null == temp ? defaultConfig : temp;
        SQLMasterStorage storage = this;
        selectConfig.setScheme(oqsFrameworkConfig.getOqsSchema());
//        return template.execute(status -> {
        return jdbcTemplate.execute(new ConnectionCallback<Collection<QueryResult>>() {
            @Override
            public Collection<QueryResult> doInConnection(Connection conn) throws SQLException, DataAccessException {
                MasterQueryResult result = null;
                result = ConditionQueryExecutor.build(engine, dictService, objectMapper, conn
                        , queryTimeout, engine.describe(entityClass, selectConfig.getProfile()), oqsFrameworkConfig.getFrameworkConfig()
                        , executionConfig.isUseStrictEnum(), executionConfig.getInClauseRewrite(), executionConfig.isLow(), storage).execute(selectConfig);
                Collection<QueryResult> entities = new ArrayList<>();
                if (!result.getValues().isEmpty()) {
                    Map<String, Object> extra = result.getExtra();
                    Object verMapping = extra.get("ver");
                    Map<Long, Integer> versionMappingMap = Collections.emptyMap();
                    if(verMapping instanceof Map) {
                        versionMappingMap = (Map)verMapping;
                        contextService.getAll().put("ver", verMapping);
                    }
                    
                    List<String> fieldsNames = result.getFieldsNames();
                    for (List<Object> value : result.getValues()) {
                        QueryResult queryResult = new QueryResult(new SystemColumn());
                        List<QueryResult.SelectItem> items = new ArrayList<>();
                        queryResult.setSelectItems(items);
                        for (int i = 0; i < fieldsNames.size(); i++) {
                            String field = fieldsNames.get(i);
                            items.add(new QueryResult.SelectItem(field, field, value.get(i)));
                        }
                        entities.add(queryResult);
                    }
                }
                return entities;
            }
        });
//        });
    }

    /**
     * always return true if one is exeucted
     * @param rs
     * @return
     */
    private boolean checkResult(Boolean[] rs) {
        if (null == rs) {
            return false;
        }

        for (boolean r : rs) {
            if (!r) {
                return false;
            }
        }
        return true;
    }

    private int checkResultCount(Boolean[] rs) {
        int count = 0;
        if (null != rs) {
            for (boolean r : rs) {
                if (r) {
                    count++;
                }
            }
        }
        return count;
    }

    private void publishAfterCreateEvent(IEntity entity, EntityClassGroup entityClassGroup, Map<String, Object> context) {
        if (publisher != null) {
            try {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass entityClass = entityClassGroup.getEntityClass();
                long id = entity.id();
                String code = entityClass.code();
                Map<String, Object> body = toMap(entity);
                publisher.publishEvent(new EntityAfterCreate(code, id, toMap(entity), false, context));
                if (!cdcConfig.isEnabled() && !cdcConfig.isDisableEvent()) {
                    //String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                    publisher.publishEvent(new EntityCreated(code, id, body, false, context));
                }
            } catch (Throwable throwable) {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "AfterCreateEvent", throwable);
            }
        }
    }

    private void publishBeforeCreateEvent(IEntity entity, EntityClassGroup entityClassGroup, Map<String, Object> context) {
        if (publisher != null) {
            try {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass entityClass = entityClassGroup.getEntityClass();
                long id = entity.id();
                String code = entityClass.code();
                publisher.publishEvent(new EntityBeforeCreate(code, id, false, toMap(entity), context));
            } catch (Throwable throwable) {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "BeforeCreateEvent", throwable);
            }
        }
    }

    private void publishAfterUpdateEvent(IEntity entity, EntityClassGroup entityClassGroup, Map<String, Object> context) {
        if (publisher != null) {
            try {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass entityClass = entityClassGroup.getEntityClass();
                long id = entity.id();
                String code = entityClass.code();
                //String code, Long id, Map<String, Object> oldData, Map<String, Object> data
                //            , boolean isInMulti, Map<String, Object> context
                Map<String, Object> body = toMap(entity);
                publisher.publishEvent(new EntityAfterUpdate(code, id, Collections.emptyMap(), body, false, context));
                if (!cdcConfig.isEnabled() && !cdcConfig.isDisableEvent()) {
                    //String code, Long id, Map<String, Object> oldData, Map<String, Object> data
                    //            , boolean isInMulti, Map<String, Object> context
                    publisher.publishEvent(new EntityUpdated(code, id, Collections.emptyMap(), body, false, context));
                }
            } catch (Throwable throwable) {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "AfterUpdateEvent", throwable);
            }
        }
    }

    private void publishBeforeUpdateEvent(IEntity entity, EntityClassGroup entityClassGroup, Map<String, Object> context) {
        if (publisher != null) {
            try {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass entityClass = entityClassGroup.getEntityClass();
                long id = entity.id();
                String code = entityClass.code();
                publisher.publishEvent(new EntityBeforeUpdate(code, id, false, toMap(entity), context));
            } catch (Throwable throwable) {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "BeforeUpdateEvent", throwable);
            }
        }
    }

    private void publishAfterDeleteEvent(IEntity entity, EntityClassGroup entityClassGroup, Map<String, Object> context) {
        if (publisher != null) {
            try {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass entityClass = entityClassGroup.getEntityClass();
                long id = entity.id();
                String code = entityClass.code();
                //String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                publisher.publishEvent(new EntityAfterDelete(code, id, Collections.emptyMap(), false, context));
                if (!cdcConfig.isEnabled() && !cdcConfig.isDisableEvent()) {
                    //String code, Long id, Map<String, Object> oldData, Map<String, Object> data
                    //            , boolean isInMulti, Map<String, Object> context
                    publisher.publishEvent(new EntityDeleted(code, id, Collections.emptyMap(), false, context));
                }
            } catch (Throwable throwable) {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "AfterDeleteEvent", throwable);
            }
        }
    }

    private void publishBeforeDeleteEvent(IEntity entity, EntityClassGroup entityClassGroup, Map<String, Object> context) {
        if (publisher != null) {
            try {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass entityClass = entityClassGroup.getEntityClass();
                long id = entity.id();
                String code = entityClass.code();
                //String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                publisher.publishEvent(new EntityBeforeDelete(code, id, false, context));
            } catch (Throwable throwable) {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "BeforeDeleteEvent", throwable);
            }
        }
    }

    private void publishBeforeMultiCreateEvent(EntityPackage entityPackage, Map<String, Object> context) {
        Map<Tuple2<Long, String>, List<Map.Entry<IEntity, IEntityClass>>> grouped = entityPackage.stream().collect(Collectors.groupingBy(x -> Tuple.of(x.getValue().id(), x.getValue().code())));

        grouped.forEach((k, v) -> {
            List<Map<String, Object>> bodies = new ArrayList<>();
            v.stream().forEach(x -> {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass targetEntityClass = x.getValue();
                if (publisher != null) {
                    try {
                        Map<String, Object> body = toMap(x.getKey());
                        bodies.add(body);
                        publisher.publishEvent(new EntityBeforeCreate(targetEntityClass.code(), x.getKey().id(), true, body, context));
                    } catch (Throwable throwable) {
                        logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "BeforeCreateEventInMulti", throwable);
                    }
                }
            });

            if (!bodies.isEmpty() && publisher != null) {
                /**
                 * String code, List<Map<String, Object>> dataList, Map<String, Object> context
                 */
                try {
                    publisher.publishEvent(new EntityBeforeMultiCreate(k._2, bodies, context));
                } catch (Throwable throwable) {
                    logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "BeforeMultiCreateEvent", throwable);
                }
            }
        });
    }

    /**
     * after
     *
     * @param entityPackage
     * @param context
     */
    private void publishAfterMultiCreateEvent(EntityPackage entityPackage, Map<String, Object> context) {
        Map<Tuple2<Long, String>, List<Map.Entry<IEntity, IEntityClass>>> grouped = entityPackage.stream()
                .collect(Collectors.groupingBy(x -> Tuple.of(x.getValue().id(), x.getValue().code())));

        grouped.forEach((k, v) -> {
            List<Map<String, Object>> bodies = new ArrayList<>();
            v.stream().forEach(x -> {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass targetEntityClass = x.getValue();
                if (publisher != null) {

                    Map<String, Object> body = toMap(x.getKey());
                    bodies.add(body);
                    try {
                        publisher.publishEvent(new EntityAfterCreate(targetEntityClass.code(), x.getKey().id(), body, true, context));
                    } catch (Throwable throwable) {
                        logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "AfterCreateEventInMulti", throwable);
                    }
                    if (!cdcConfig.isEnabled() && !cdcConfig.isDisableEvent()) {
                        //String code, Long id, Map<String, Object> oldData, Map<String, Object> data
                        //            , boolean isInMulti, Map<String, Object> context
                        try {
                            publisher.publishEvent(new EntityCreated(targetEntityClass.code(), x.getKey().id(), body, true, context));
                        } catch (Throwable throwable) {
                            logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "CreateEventInMulti", throwable);
                        }
                    }
                }
            });

            if (!bodies.isEmpty() && publisher != null) {
                /**
                 * String code, List<Map<String, Object>> dataList, Map<String, Object> context
                 */
                try {
                    publisher.publishEvent(new EntityAfterMultiCreate(k._2, bodies, context));
                } catch (Throwable throwable) {
                    logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "AfterMultiCreateEvent", throwable);
                }

                if (!cdcConfig.isEnabled() && !cdcConfig.isDisableEvent()) {
                    try {
                        publisher.publishEvent(new EntityMultiCreated(k._2, bodies, context));
                    } catch (Throwable throwable) {
                        logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "MultiCreateEvent", throwable);
                    }
                }
            }
        });
    }

    private void publishBeforeConUpdateEvent(EntityPackage entityPackage, Map<String, Object> context) {
        IEntity requestBody = entityPackage.getExample();
        String targetCode = entityPackage.getTargetCode();
        Map<String, ExpRel> condition = entityPackage.getCondition();
        if (publisher != null) {
            try {
                
                if(requestBody == null) {
                    publisher.publishEvent(new EntityBeforeConUpdate(targetCode, condition, context));
                } else {
                    publisher.publishEvent(new EntityBeforeConUpdate(targetCode, condition, toMap(entityPackage.getExample()), context));
                }
            } catch (Throwable throwable) {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "BeforeConUpdateEvent", throwable);
            }
        }
    }

    private void publishBeforeConDeleteEvent(EntityPackage entityPackage, Map<String, Object> context) {
        String targetCode = entityPackage.getTargetCode();
        Map<String, ExpRel> condition = entityPackage.getCondition();
        if (publisher != null) {
            try {
                publisher.publishEvent(new EntityBeforeConDelete(targetCode, condition, context));
            } catch (Throwable throwable) {
                logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "BeforeConDeleteEvent", throwable);
            }
        }
    }

    private void publishBeforeMultiUpdateEvent(EntityPackage entityPackage, Map<String, Object> context) {
        Map<Tuple2<Long, String>, List<Map.Entry<IEntity, IEntityClass>>> grouped = entityPackage.stream()
                .collect(Collectors.groupingBy(x -> Tuple.of(x.getValue().id(), x.getValue().code())));

        grouped.forEach((k, v) -> {
            List<Map<String, Object>> bodies = new ArrayList<>();
            v.stream().forEach(x -> {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass targetEntityClass = x.getValue();
                if (publisher != null) {
                    try {
                        Map<String, Object> body = toMap(x.getKey());
                        bodies.add(body);
                        publisher.publishEvent(new EntityBeforeUpdate(targetEntityClass.code()
                                , x.getKey().id(), true, body, context));
                    } catch (Throwable throwable) {
                        logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "BeforeUpdateEventInMulti", throwable);
                    }
                }
            });

            if (!bodies.isEmpty() && publisher != null) {
                /**
                 * String code, List<Map<String, Object>> dataList, Map<String, Object> context
                 */
                try {
                    publisher.publishEvent(new EntityBeforeMultiUpdate(k._2, bodies, context));
                } catch (Throwable throwable) {
                    logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "BeforeMultiUpdateEvent", throwable);
                }
            }
        });
    }

    private void publishAfterMultiUpdateEvent(EntityPackage entityPackage, Map<String, Object> context) {
        Map<Tuple2<Long, String>, List<Map.Entry<IEntity, IEntityClass>>> grouped = entityPackage.stream()
                .collect(Collectors.groupingBy(x -> Tuple.of(x.getValue().id(), x.getValue().code())));

        grouped.forEach((k, v) -> {
            List<Map<String, Object>> bodies = new ArrayList<>();
            v.stream().forEach(x -> {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass targetEntityClass = x.getValue();
                if (publisher != null) {
                    Map<String, Object> body = toMap(x.getKey());
                    bodies.add(body);
                    try {
                        publisher.publishEvent(new EntityAfterUpdate(targetEntityClass.code(), x.getKey().id(), Collections.emptyMap(), body, true, context));
                    } catch (Throwable throwable) {
                        logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "AfterUpdateEventInMulti", throwable);
                    }

                    if (!cdcConfig.isEnabled() && !cdcConfig.isDisableEvent()) {
                        //String code, Long id, Map<String, Object> oldData, Map<String, Object> data
                        //            , boolean isInMulti, Map<String, Object> context
                        //String code, List<Map<String, Object>> dataList, Map<String, Object> context
                        try {
                            publisher.publishEvent(new EntityUpdated(targetEntityClass.code(), x.getKey().id(), Collections.emptyMap(), body, true, context));
                        } catch (Throwable throwable) {
                            logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "UpdateEventInMulti", throwable);
                        }
                    }
                }
            });

            if (!bodies.isEmpty() && publisher != null) {
                /**
                 * String code, List<Map<String, Object>> dataList, Map<String, Object> context
                 */
                //String code, List<Tuple2<Map<String, Object>, Map<String, Object>>> dataList, Map<String, Object> context
                try {
                    publisher.publishEvent(new EntityAfterMultiUpdate(k._2, bodies.stream().map(x -> Tuple.of(Collections.<String, Object>emptyMap(), x))
                            .collect(Collectors.toList()), context));
                } catch (Throwable throwable) {
                    logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "AfterMultiUpdateEvent", throwable);
                }
                if (!cdcConfig.isEnabled() && !cdcConfig.isDisableEvent()) {
                    //String code, Long id, Map<String, Object> oldData, Map<String, Object> data
                    //            , boolean isInMulti, Map<String, Object> context
                    //String code, List<Map<String, Object>> dataList, Map<String, Object> context
                    try {
                        publisher.publishEvent(new EntityMultiUpdated(k._2, bodies.stream().map(x -> Tuple.of(Collections.<String, Object>emptyMap(), x))
                                .collect(Collectors.toList()), context));
                    } catch (Throwable throwable) {
                        logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "MultiUpdateEvent", throwable);
                    }
                }
            }
        });
    }

    private void publishAfterMultiDeletedEvent(EntityPackage entityPackage, Map<String, Object> context) {
        Map<Tuple2<Long, String>, List<Map.Entry<IEntity, IEntityClass>>> grouped = entityPackage.stream()
                .collect(Collectors.groupingBy(x -> Tuple.of(x.getValue().id(), x.getValue().code())));

        grouped.forEach((k, v) -> {
            List<Map<String, Object>> bodies = new ArrayList<>();
            v.stream().forEach(x -> {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass targetEntityClass = x.getValue();
                if (publisher != null) {
                    Map<String, Object> body = toMap(x.getKey());
                    bodies.add(body);
                    //String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                    try {
                        publisher.publishEvent(new EntityAfterDelete(targetEntityClass.code(), x.getKey().id(), Collections.emptyMap(), true, context));
                    } catch (Throwable throwable) {
                        logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "AfterDeleteEventInMulti", throwable);
                    }
                    if (!cdcConfig.isEnabled() && !cdcConfig.isDisableEvent()) {
                        try {
                            publisher.publishEvent(new EntityDeleted(targetEntityClass.code(), x.getKey().id(), Collections.emptyMap(), true, context));
                        } catch (Throwable throwable) {
                            logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "DeleteEventInMulti", throwable);
                        }
                    }
                }
            });

            if (!bodies.isEmpty() && publisher != null) {
                /**
                 * String code, List<Map<String, Object>> dataList, Map<String, Object> context
                 */
                //String code, List<Tuple2<Map<String, Object>, Map<String, Object>>> dataList, Map<String, Object> context
                try {
                    publisher.publishEvent(new EntityAfterMultiDelete(k._2, bodies, context));
                } catch (Throwable throwable) {
                    logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "AfterMultiDeleteEvent", throwable);
                }
                if (!cdcConfig.isEnabled() && !cdcConfig.isDisableEvent()) {
                    try {
                        publisher.publishEvent(new EntityMultiDeleted(k._2, bodies, context));
                    } catch (Throwable throwable) {
                        logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "MultiDeleteEvent", throwable);
                    }
                }
            }
        });
    }

    private void publishBeforeMultiDeleteEvent(EntityPackage entityPackage, Map<String, Object> context) {
        Map<Tuple2<Long, String>, List<Map.Entry<IEntity, IEntityClass>>> grouped = entityPackage.stream()
                .collect(Collectors.groupingBy(x -> Tuple.of(x.getValue().id(), x.getValue().code())));

        grouped.forEach((k, v) -> {
            List<Long> bodies = new ArrayList<>();
            v.stream().forEach(x -> {
                /**
                 * String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                 */
                IEntityClass targetEntityClass = x.getValue();
                if (publisher != null) {
//                    Map<String, Object> body = toMap(x.getKey());
                    bodies.add(x.getKey().id());
                    //String code, Long id, Map<String, Object> data, boolean inInMulti, Map<String, Object> context
                    try {
                        publisher.publishEvent(new EntityBeforeDelete(targetEntityClass.code(), x.getKey().id(), true, context));
                    } catch (Throwable throwable) {
                        logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "BeforeDeleteEventInMulti", throwable);
                    }
                }
            });

            if (!bodies.isEmpty() && publisher != null) {
                /**
                 * String code, List<Map<String, Object>> dataList, Map<String, Object> context
                 */
                //String code, List<Tuple2<Map<String, Object>, Map<String, Object>>> dataList, Map<String, Object> context
                try {
                    publisher.publishEvent(new EntityBeforeMultiDelete(k._2, bodies, context));
                } catch (Throwable throwable) {
                    logErrorPattern(log, LoggingPattern.UNKNOWN_ERROR, "BeforeMultiDeleteEvent", throwable);
                }
            }
        });
    }

    private Map<String, Object> toMap(IEntity entity) {
        IEntityValue iEntityValue = entity.entityValue();
        Map<String, Object> payload = new HashMap<>();
        if (iEntityValue != null) {
            iEntityValue.values().forEach(x -> {
                String name = x.getField().name();
                Object value = x.getValue();
                payload.put(name, value);
            });
        }

        return payload;
    }

    private boolean writeProcessing(IEntity entity, EntityClassGroup entityClass, Operation op, Map<String, Object> context) throws SQLException {
        checkId(entity);

        boolean result = template.execute(status -> {
            return jdbcTemplate.execute(new ConnectionCallback<Boolean>() {
                @Override
                public Boolean doInConnection(Connection con) throws SQLException, DataAccessException {
                    List<MasterStorageEntity> masterStorageEntities = toMasterStorageEntities(entityClass, entity, op);

                    Boolean[] rs;
                    switch (op) {
                        case BUILD:
                            publishBeforeCreateEvent(entity, entityClass, context);
                            rs = CreateExecutor.build(con, queryTimeout).execute(masterStorageEntities);
                            publishAfterCreateEvent(entity, entityClass, context);
                            inserCountTotal.increment();
                            break;
                        case REPLACE:
                            publishBeforeUpdateEvent(entity, entityClass, context);
                            rs = UpdateExecutor.build(con, queryTimeout).execute(masterStorageEntities);
                            publishAfterUpdateEvent(entity, entityClass, context);
                            replaceCountTotal.increment();
                            break;
                        case DELETE:
                            publishBeforeDeleteEvent(entity, entityClass, context);
                            Object userCode = context.get("USER_DISPLAYNAME");
                            Object userId = context.get("ID");

                            rs = DeleteExecutor.build(con, queryTimeout, cdcConfig.getRecordUser(), Optional.ofNullable(userCode).map(x -> x.toString())
                                                    .filter(x -> !StringUtils.isEmpty(x))
                                                    .orElseGet(() -> Optional.ofNullable(context.get("USER_LOGINNAME")).map(x -> x.toString()).orElse(""))
                                            , Optional.ofNullable(userId).map(Object::toString).map(Long::parseLong).orElse(null))
                                    .execute(masterStorageEntities);
                            publishAfterDeleteEvent(entity, entityClass, context);
                            deleteCountTotal.increment();
                            break;
                        default:
                            rs = null;
                    }

                    if (publisher != null) {
                        EntityClassRef entityClassRef = entity.entityClassRef();
                        publisher.publishEvent(new EntityChanged(Collections
                                .singletonList(Tuple.of(entityClassRef.getProfile(), entityClassRef.getId())), entity.time()));
                    }
                    return checkResult(rs);

                }
            });
        });

        if (result) {
            entity.neat();
        }


        return result;
    }

    private int conditionalWriteProcessing(EntityPackage entityPackage, Operation op, Map<String, Object> context) {

        OQSFrameworkConfig temp = oqsFrameworkConfigMap.get(dynamicDataSource.currentKey());
        final OQSFrameworkConfig oqsFrameworkConfig = null == temp ? defaultConfig : temp;

        return template.execute(status -> {
            return jdbcTemplate.execute(new ConnectionCallback<Integer>() {

                @Override
                public Integer doInConnection(Connection con) throws SQLException, DataAccessException {

                    if (publisher != null) {
                        switch (op) {
                            case REPLACE_CON:
                                publishBeforeConUpdateEvent(entityPackage, context);
                                break;
                            case DELETE_CON:
                                publishBeforeConDeleteEvent(entityPackage, context);
                                break;
                        }
                    }

                    int affectedRow = 0;
                    int temp = 0;

                    List<ConditionalStoragePackage> storagePackage = toRelatedStatement(entityPackage, oqsFrameworkConfig, context);
                    switch (op) {
                        case REPLACE_CON: {
                            //TODO affectedRow may not accruate
                            temp = ConditionalReplaceExecutor.build(con, queryTimeout, executionConfig.isCheckConditionalLimit())
                                    .execute(storagePackage);
                            if (temp >= affectedRow) {
                                affectedRow = temp;
                            }
                            break;
                        }
                        case DELETE_CON: {
                            Object userCode = context.get("USER_DISPLAYNAME");
                            Object userId = context.get("ID");

                            temp = ConditionalDeleteExecutor.build(con, queryTimeout, executionConfig.isCheckConditionalLimit(), cdcConfig.getRecordUser(), Optional.ofNullable(userCode).map(x -> x.toString())
                                                    .filter(x -> !StringUtils.isEmpty(x))
                                                    .orElseGet(() -> Optional.ofNullable(context.get("USER_LOGINNAME")).map(x -> x.toString()).orElse(""))
                                            , Optional.ofNullable(userId).map(Object::toString).map(Long::parseLong).orElse(null))
                                    .execute(storagePackage);
                            if (temp >= affectedRow) {
                                affectedRow = temp;
                            }
                            break;
                        }
                        default:
                    }
//                    }
                    return affectedRow;
                }
            });
        });
    }

    
    private List<String> getAllRelatedTable(EntityClassGroup group) {
        List<String> tableNames = new ArrayList<>();
        tableNames.add(group.getEntityClass().masterWriteTable(false));
        String profile = group.realProfile();
        if(!StringUtils.isEmpty(profile)) {
            tableNames.add(group.getEntityClass().masterWriteTable(true));
        }
        Collection<IEntityClass> fatherEntityClass = group.getFatherEntityClass();
        if(fatherEntityClass != null && !fatherEntityClass.isEmpty()) {
            fatherEntityClass.stream().forEach(x -> {
                tableNames.add(x.masterWriteTable(false));
                if(!StringUtils.isEmpty(profile)) {
                    tableNames.add(x.masterWriteTable(true));
                }
            });
        }

        Collection<IEntityClass> childrenEntityClass = group.getChildrenEntityClass();
        if(childrenEntityClass != null && !childrenEntityClass.isEmpty()) {
            childrenEntityClass.stream().forEach(x -> {
                tableNames.add(x.masterWriteTable(false));
                if(!StringUtils.isEmpty(profile)) {
                    tableNames.add(x.masterWriteTable(true));
                }
            });
        }

        return tableNames;
    }
    
    private ConditionalStoragePackage buildConditionalStorage(EntityClassGroup group, IEntity entity, OQSFrameworkConfig targetConfig
            , Map<String, ExpRel> conditionalMapping, String targetCode, boolean convertId, Map<String, Object> context) {
        List<MasterStorageEntity> masterStorageEntities = Collections.emptyList();
        //real conditional
        if(entity != null) {
            masterStorageEntities = toMasterStorageEntities(group, entity, Operation.REPLACE);
        }
        ConditionalStoragePackage storagePackage = new ConditionalStoragePackage();
        ExpContext expContext = new ExpContext();
        expContext.setSchema(group);
        expContext.withContext(context);
        ExpRel expRel = conditionalMapping.get(targetCode);
        if(convertId) {
            ExpRel idQuery = new ExpQuery()
                    .filters(ExpCondition.call(ExpOperator.EQUALS, ExpField.ID, ExpValue.from(entity.id())))
                    .range(1, 1);
            expRel = expRel.mergeAnd(idQuery);
        }
        RelNode relTree = ExpTreeToRel.toRelTree(expContext, expRel, config, executionConfig);
        RelBuilder relBuilder = RelBuilder.create(targetConfig.getFrameworkConfig());
        String masterWriteTable = group.getEntityClass().masterWriteTable(false);
        String mainTable = group.getJoinTable(masterWriteTable);
        storagePackage.setMainTableName(mainTable);
        storagePackage.setRawEntityClass(group);
        storagePackage.setRelatedTable(getAllRelatedTable(group));

        ConditionalSqlShuttle conditionalSqlShuttle = new ConditionalSqlShuttle(group, relBuilder
                , masterWriteTable, dictService, executionConfig);
        relTree.accept(conditionalSqlShuttle);

        storagePackage.setLimit(conditionalSqlShuttle.getLimit());
        String conditonaSql = conditionalSqlShuttle.getConditionSql();
        storagePackage.setConditionSql(conditonaSql);
        //map to multi
        List<ConditionalStorageEntity> conditionalStorageList = masterStorageEntities.stream().map(x -> {
            String tableName = x.getTableName();
            String joinTable = group.getJoinTable(tableName);
            ConditionalStorageEntity storageEntity = new ConditionalStorageEntity(tableName, joinTable, conditonaSql, 0, x);
            return storageEntity;
        }).collect(Collectors.toList());

        storagePackage.setConditionalStorageList(conditionalStorageList);
        return storagePackage;
    }
    
    /**
     * one table
     * multi table
     * 
     * entityPackage may have no target
     *
     * @param entityPackage
     * @return
     */
    private List<ConditionalStoragePackage> toRelatedStatement(EntityPackage entityPackage, OQSFrameworkConfig targetConfig, Map<String, Object> context) {

        Map<Tuple3<Long, String, String>, List<Map.Entry<IEntity, IEntityClass>>> grouped = null;
        if(entityPackage.getEntities() != null && !entityPackage.getEntities().isEmpty()) {
            grouped = entityPackage.stream()
                    .collect(Collectors.groupingBy(x -> Tuple.of(x.getValue().id(), x.getValue().code(), x.getValue().realProfile())));
        } else if(entityPackage.getExample() != null){
            grouped = new HashMap<>();
            List<Map.Entry<IEntity, IEntityClass>> list = new ArrayList<>();
            list.add(Pair.of(entityPackage.getExample(), entityPackage.getTargetEntityClass()));
            grouped.put(Tuple.of(entityPackage.getTargetEntityClass().id()
                    , entityPackage.getTargetCode()
                    , entityPackage.getTargetEntityClass().realProfile()), list);
        }

        /**
         * update /multi
         */
        if(grouped != null) {

            Map<String, ExpRel> conditionalMapping = entityPackage.getCondition();
            return grouped.entrySet().stream().flatMap(entry -> {

                Tuple3<Long, String, String> key = entry.getKey();
                Optional<IEntityClass> targetEntityClass = engine.load(Long.toString(key._1), key._3);
                if (targetEntityClass.isPresent()) {
                    EntityClassGroup group = engine.describe(targetEntityClass.get(), targetEntityClass.get().realProfile());


                    if (entityPackage.getExample() != null) {
                        //EntityClassGroup group, IEntity entity, OQSFrameworkConfig targetConfig
                        //            , Map<String, ExpRel> conditionalMapping, String targetCode, Map<String, Object> context
                        return Stream.of(buildConditionalStorage(group, entityPackage.getExample(), targetConfig, conditionalMapping, key._2, false, context));
                    } else {
                        //multi + conditional
                        List<Map.Entry<IEntity, IEntityClass>> value = entry.getValue();
                        List<ConditionalStoragePackage> packages = new ArrayList<>();
                        for (Map.Entry<IEntity, IEntityClass> item : value) {
                            IEntity entity = item.getKey();
                            //build conditional
                            packages.add(buildConditionalStorage(group, entity, targetConfig, conditionalMapping, key._2, true, context));
                        }
                        return packages.stream();
                    }
                }
                return null;
            }).filter(Objects::nonNull).collect(Collectors.toList());
        } else {
            IEntityClass targetEntityClass = entityPackage.getTargetEntityClass();
            EntityClassGroup group = engine.describe(targetEntityClass, targetEntityClass.realProfile());
            Map<String, ExpRel> condition = entityPackage.getCondition();
            return Collections.singletonList(buildConditionalStorage(group, null, targetConfig, condition, group.getEntityClass().code(), false, context));
        }
    }

    private Boolean[] packageWriteProcessing(EntityPackage entityPackage, Operation op, Map<String, Object> context) throws SQLException {
        return template.execute(status -> {
            return jdbcTemplate.execute(new ConnectionCallback<Boolean[]>() {
                @Override
                public Boolean[] doInConnection(Connection con) throws SQLException, DataAccessException {

                    List<MasterStorageEntity> executionPackages = new ArrayList<>();
                    entityPackage.stream().forEach(er -> {
                        EntityClassGroup group = engine.describe(er.getValue(), er.getValue().profile());
                        executionPackages.addAll(toMasterStorageEntities(group, er.getKey(), op));
                    });

                    List<List<MasterStorageEntity>> executionPart = Lists.partition(executionPackages, EntityPackage.MAX_SIZE);

                    if (publisher != null) {
                        switch (op) {
                            case BUILD:
                                publishBeforeMultiCreateEvent(entityPackage, context);
                                break;
                            case REPLACE:
                                publishBeforeMultiUpdateEvent(entityPackage, context);
                                break;
                            case REPLACE_CON:
                                publishBeforeConUpdateEvent(entityPackage, context);
                                break;
                            case DELETE:
                                publishBeforeMultiDeleteEvent(entityPackage, context);
                                break;
                        }
                    }

                    List<Boolean[]> resultSet = new ArrayList<>();
                    for (List<MasterStorageEntity> parts : executionPart) {
                        switch (op) {
                            case BUILD: {
                                Boolean[] execute = CreateExecutor.build(con, queryTimeout).execute(parts);
                                resultSet.add(execute);
                                inserCountTotal.increment(Arrays.stream(execute).filter(x -> x).count());
                                break;
                            }
                            case REPLACE: {
                                Boolean[] execute = UpdateExecutor.build(con, queryTimeout).execute(parts);
                                resultSet.add(execute);
                                replaceCountTotal.increment(Arrays.stream(execute).filter(x -> x).count());
                                break;
                            }
                            case DELETE: {
                                Object userCode = context.get("USER_DISPLAYNAME");
                                Object userId = context.get("ID");
                                Boolean[] execute = DeleteExecutor.build(con, queryTimeout, cdcConfig.getRecordUser(), Optional.ofNullable(userCode).map(x -> x.toString()).orElse(null)
                                        , Optional.ofNullable(userId).map(Object::toString).map(Long::parseLong).orElse(null)).execute(parts);
                                resultSet.add(execute);
                                deleteCountTotal.increment(Arrays.stream(execute).filter(x -> x).count());
                                break;
                            }
                            default:
                        }
                    }

                    if (publisher != null) {
                        try {
                            //do publisher
                            List<Tuple2<String, Long>> list = entityPackage.stream().map(entry -> {
                                String profile = entry.getKey().entityClassRef().getProfile();
                                long entityClassId = entry.getKey().entityClassRef().getId();
                                return Tuple.of(profile, entityClassId);
                            }).collect(Collectors.toList());

                            if (!list.isEmpty()) {
                                publisher.publishEvent(new EntityChanged(list, entityPackage.get(0).get().getKey().time()));
                            }
                        } catch (Throwable throwable) {
                            log.error("{}", throwable);
                        }
                    }

                    switch (op) {
                        case BUILD:
                            publishAfterMultiCreateEvent(entityPackage, context);
                            break;
                        case REPLACE:
                            publishAfterMultiUpdateEvent(entityPackage, context);
                            break;
                        case DELETE:
                            publishAfterMultiDeletedEvent(entityPackage, context);
                            break;
                    }

                    return flatten(resultSet);
                }
            });
        });
    }

    private Boolean[] flatten(List<Boolean[]> resultSet) {
        return resultSet.stream().flatMap(x -> Arrays.stream(x)).toArray(Boolean[]::new);
    }

    private List<MasterStorageEntity.TypedStorageValue> extractBocpSystemField(EntityClassGroup group, IEntity entity) {
        Collection<IEntityField> allFields = group.getAllFields();
        List<MasterStorageEntity.TypedStorageValue> values = new ArrayList<>();
        for (IEntityField entityField : allFields) {
            Optional<SystemField> systemField = Arrays.stream(SystemField.values()).filter(x -> x.getName().equalsIgnoreCase(entityField.name())).findFirst();
            if (systemField.isPresent()) {
                Optional<IValue> valueOp = entity.entityValue().getValue(entityField);
                if (valueOp.isPresent()) {
                    MasterStorageEntity.TypedStorageValue targetValue = new MasterStorageEntity.TypedStorageValue(entityField, valueOp.get().storageValue());
                    IValue iValue = valueOp.get();
                    if(valueOp.get() instanceof ExpressionValue) {
                        targetValue.setKeepRaw(true);
                    } else if(valueOp.get() instanceof OnDuplicateValue){
                        targetValue.setRawValue(valueOp.get());
                        if(((OnDuplicateValue) valueOp.get()).getOnDuplicate() instanceof ExpressionValue) {
                            targetValue.setKeepRawOnDuplicate(true);
                        }
                    }
                    values.add(targetValue);
                }
            }
        }

        return values;
    }

    private MasterStorageEntity merge(MasterStorageEntity entity, List<MasterStorageEntity.TypedStorageValue> values) {
        MasterStorageEntity masterStorageEntity = new MasterStorageEntity();
        masterStorageEntity.setTableName(entity.getTableName());
        masterStorageEntity.setUseOptimisticLock(entity.isUseOptimisticLock());
        masterStorageEntity.setBusinessDynamicFields(entity.getBusinessDynamicFields());
        masterStorageEntity.setSystemColumn(entity.getSystemColumn());
        masterStorageEntity.setProfiled(entity.isProfiled());
        masterStorageEntity.setRawEntityClass(entity.getRawEntityClass());
        masterStorageEntity.setOnlyFather(entity.isOnlyFather());
        List<MasterStorageEntity.TypedStorageValue> mergedValues = new ArrayList<>(values);
        Optional.ofNullable(entity.getBusinessStaticFields())
                .orElseGet(Collections::emptyList).stream().filter(x -> {
                    return values.stream().noneMatch(v -> {
                        //in case System field turn to index
                        return v.getEntityField().id() == (x.getEntityField().id());
                    });
                }).forEach(x -> {
                    mergedValues.add(x);
                });
        masterStorageEntity.setBusinessStaticFields(mergedValues);
        return masterStorageEntity;
    }

    private List<MasterStorageEntity> toMasterStorageEntities(EntityClassGroup group, IEntity entity, Operation op) {
        List<MasterStorageEntity> masterStorageEntities = new ArrayList<>();

        //extract system field and spread on every object
        List<MasterStorageEntity.TypedStorageValue> values = extractBocpSystemField(group, entity);

        boolean hasChild = false;
        if(group.getChildrenEntityClass() != null && !group.getChildrenEntityClass().isEmpty()) {
            //current request entityclass is parent so add father only
            hasChild = true;
        }

        //TODO missing profile
        //  加入自己.
        List<MasterStorageEntity> masterStorageEntity = toMasterStorageEntity(group.getEntityClass(), entity, op, hasChild);

        masterStorageEntity.stream().map(x -> merge(x, values)).forEach(masterStorageEntities::add);
        //  加入父类.
        if (null != group.getFatherEntityClass() && !group.getFatherEntityClass().isEmpty()) {
            Collection<IEntityClass> entityClasses = group.getFatherEntityClass();

            for (IEntityClass e : entityClasses) {
                //in parent logic all 
                List<MasterStorageEntity> multiMaster = toMasterStorageEntity(e, entity, op, false);
                multiMaster.stream().map(x -> merge(x, values)).forEach(masterStorageEntities::add);
            }
        }

        if (masterStorageEntities.size() > 1) {
            //grouped
            if (op == Operation.BUILD) {
                Map<String, Object> attachment = new HashMap<>();
                //add attachment
                attachment.put("grouped", masterStorageEntities.size());
                masterStorageEntities.forEach(x -> {
//                    Map<String, Object> map = buildFromJson(x.getBusinessDynamicFields());
//                    map.put("#a", attachment);
//                    x.setBusinessDynamicFields(toBuildJson(map));
                    x.setAttachment(attachment);
                });
            }
        }

        return masterStorageEntities;
    }

    /**
     * if profiled will get multi
     *
     * @param e
     * @param entity
     * @param op
     * @return
     */
    private List<MasterStorageEntity> toMasterStorageEntity(IEntityClass e, IEntity entity, Operation op, boolean hasChildern) {
        String profile = e.realProfile();
        EntityClassGroup group = engine.describe(e, e.realProfile());
        if (StringUtils.isEmpty(profile)) {
            Collection<IEntityField> selfFields = e.selfWithIndex();

            MasterStorageEntity masterStorageEntity = new MasterStorageEntity();

            Optional<IValue> tenantCode = entity.entityValue().getValue("tenant_code");
            String profileForEntity = e.profile();
            if(StringUtils.isEmpty(profileForEntity)) {
                if(tenantCode.isPresent()) {
                    profileForEntity = tenantCode.get().valueToString();
                }
            }

            //  设置系统字段
            SystemColumn systemColumn = new SystemColumn();
            systemColumn.setProfile(profileForEntity);
            systemColumn.setVersion(entity.version());
            systemColumn.setOperateTime(entity.time());
            systemColumn.setEntityClass(entity.entityClassRef().getId());
            systemColumn.setId(entity.id());
            masterStorageEntity.setSystemColumn(systemColumn);
            masterStorageEntity.setOnlyFather(hasChildern);
            masterStorageEntity.setRawEntityClass(group);
            //TODO
            masterStorageEntity.setTableName(e.masterWriteTable(false));
            masterStorageEntity.setUseOptimisticLock(entity.useOptimizeLock());

            List<MasterStorageEntity.TypedStorageValue> staticValues = new ArrayList<>();
            List<IValue> dynamicValues = new ArrayList<>();

            //  delete操作不处理任何的字段值.
            if (!op.equals(Operation.DELETE)) {
                for (IEntityField entityField : selfFields) {
                    Optional<IValue> valueOp = entity.entityValue().getValue(entityField);
                    if (valueOp.isPresent()) {
                        
                        MasterStorageEntity.TypedStorageValue targetValue;
                        if (entityField.type() == FieldType.ENUMS && executionConfig.isUseStrictEnum()) {
                            String[] values = ((EnumsValue) valueOp.get()).getValue();
                            targetValue = new MasterStorageEntity.TypedStorageValue(entityField, getEnumBit(entityField.dictId(), Arrays.asList(values)), true);
                        } else {
                            targetValue = new MasterStorageEntity.TypedStorageValue(entityField, valueOp.get().storageValue());
                            if(valueOp.get() instanceof ExpressionValue) {
                                targetValue.setKeepRaw(true);
                            } else if(valueOp.get() instanceof OnDuplicateValue){
                                targetValue.setRawValue(valueOp.get());
                                if(((OnDuplicateValue) valueOp.get()).getOnDuplicate() instanceof ExpressionValue) {
                                    targetValue.setKeepRawOnDuplicate(true);
                                }
                            } 
                            
                            if(entityField.isDynamic()) {
                                dynamicValues.add(valueOp.get());
                            } else {
                                staticValues.add(targetValue);
                            }
                        }
                    }
                }

                masterStorageEntity.setBusinessStaticFields(staticValues);
                masterStorageEntity.setBusinessDynamicFields(dynamicValues);
                //  设置动态.
//                Map<String, Object> painValues = MasterStorageHelper.toPainValues(dynamicValues);

//                if (op.equals(Operation.REPLACE)) {
//                    masterStorageEntity.(MasterStorageHelper.buildReplace(painValues, mySQLCodec));
//                    masterStorageEntity.setBusinessDynamicFieldsRemove(MasterStorageHelper.buildRemove(painValues));
//                } else {
//                    masterStorageEntity.setBusinessDynamicFields(MasterStorageHelper.toBuildJson(painValues));
//                }
                
                systemColumn.setDeleted(false);
            } else {
                systemColumn.setDeleted(true);
            }
            return Collections.singletonList(masterStorageEntity);
        } else {

            List<MasterStorageEntity> masterStorageEntities = new ArrayList<>();
            Collection<IEntityField> selfFields = e.selfWithIndex();

            List<IEntityField> pureSelfFields = new ArrayList<>();
            List<IEntityField> profileFields = new ArrayList<>();
            for (IEntityField selfField : selfFields) {
                String realProfile = selfField.realProfile();
                if (StringUtils.isEmpty(realProfile)) {
                    pureSelfFields.add(selfField);
                } else {
                    profileFields.add(selfField);
                }
            }

//            if (!pureSelfFields.isEmpty()) {
            //deal with self

            MasterStorageEntity masterStorageEntity = new MasterStorageEntity();
            //  设置系统字段
            SystemColumn systemColumn = new SystemColumn();
            systemColumn.setProfile(profile);
            systemColumn.setOperateTime(entity.time());
            systemColumn.setEntityClass(e.id());
            systemColumn.setVersion(entity.version());
            systemColumn.setId(entity.id());
            masterStorageEntity.setSystemColumn(systemColumn);
            //TODO
            masterStorageEntity.setTableName(e.masterWriteTable(false));
            masterStorageEntity.setOnlyFather(hasChildern);
            masterStorageEntity.setRawEntityClass(group);
            masterStorageEntity.setUseOptimisticLock(entity.useOptimizeLock());

            List<MasterStorageEntity.TypedStorageValue> staticValues = new ArrayList<>();
            List<IValue> dynamicValues = new ArrayList<>();

            //  delete操作不处理任何的字段值.
            if (!op.equals(Operation.DELETE)) {
                for (IEntityField entityField : pureSelfFields) {
                    Optional<IValue> valueOp = entity.entityValue().getValue(entityField);
                    if (valueOp.isPresent()) {
                        MasterStorageEntity.TypedStorageValue targetValue;
                        if (entityField.type() == FieldType.ENUMS && executionConfig.isUseStrictEnum()) {
                            String[] values = ((EnumsValue) valueOp.get()).getValue();
                            targetValue = new MasterStorageEntity.TypedStorageValue(entityField, getEnumBit(entityField.dictId(), Arrays.asList(values)), true);
                        } else {
                            targetValue = new MasterStorageEntity.TypedStorageValue(entityField, valueOp.get().storageValue());
                            if(valueOp.get() instanceof ExpressionValue) {
                                targetValue.setKeepRaw(true);
                            } else if(valueOp.get() instanceof OnDuplicateValue){
                                targetValue.setRawValue(valueOp.get());
                                if(((OnDuplicateValue) valueOp.get()).getOnDuplicate() instanceof ExpressionValue) {
                                    targetValue.setKeepRawOnDuplicate(true);
                                }
                            }

                            if(entityField.isDynamic()) {
                                dynamicValues.add(valueOp.get());
                            } else {
                                staticValues.add(targetValue);
                            }
                        }                        
                    }
                }

                masterStorageEntity.setBusinessStaticFields(staticValues);
                masterStorageEntity.setBusinessDynamicFields(dynamicValues);
                systemColumn.setDeleted(false);
            } else {
                systemColumn.setDeleted(true);
            }
            masterStorageEntities.add(masterStorageEntity);
            MasterStorageEntity masterStorageEntityProfile = new MasterStorageEntity();
            //  设置系统字段
            SystemColumn systemColumnProfile = new SystemColumn();
            systemColumnProfile.setProfile(e.profile());
            systemColumnProfile.setOperateTime(entity.time());
            systemColumnProfile.setEntityClass(e.id());
            systemColumnProfile.setId(entity.id());
            systemColumnProfile.setVersion(entity.version());
            masterStorageEntityProfile.setSystemColumn(systemColumnProfile);
            masterStorageEntityProfile.setOnlyFather(hasChildern);
            masterStorageEntityProfile.setUseOptimisticLock(entity.useOptimizeLock());
            masterStorageEntityProfile.setProfiled(true);
            masterStorageEntityProfile.setRawEntityClass(group);
            
            //TODO
            masterStorageEntityProfile.setTableName(e.masterWriteTable(true));
            List<MasterStorageEntity.TypedStorageValue> staticValuesProfile = new ArrayList<>();
            List<IValue> dynamicValuesProfile = new ArrayList<>();

            //  delete操作不处理任何的字段值.
            if (!op.equals(Operation.DELETE)) {
                for (IEntityField entityField : profileFields) {
                    Optional<IValue> valueOp = entity.entityValue().getValue(entityField);
                    if (valueOp.isPresent()) {
                        MasterStorageEntity.TypedStorageValue targetValue;
                        if (entityField.type() == FieldType.ENUMS && executionConfig.isUseStrictEnum()) {
                            String[] values = ((EnumsValue) valueOp.get()).getValue();
                            targetValue = new MasterStorageEntity.TypedStorageValue(entityField, getEnumBit(entityField.dictId(), Arrays.asList(values)), true);
                        } else {
                            targetValue = new MasterStorageEntity.TypedStorageValue(entityField, valueOp.get().storageValue());
                            if(valueOp.get() instanceof ExpressionValue) {
                                targetValue.setKeepRaw(true);
                            } else if(valueOp.get() instanceof OnDuplicateValue){
                                targetValue.setRawValue(valueOp.get());
                                if(((OnDuplicateValue) valueOp.get()).getOnDuplicate() instanceof ExpressionValue) {
                                    targetValue.setKeepRawOnDuplicate(true);
                                }
                            }

                            if(entityField.isDynamic()) {
                                dynamicValuesProfile.add(valueOp.get());
                            } else {
                                staticValuesProfile.add(targetValue);
                            }
                        }
                    }
                }

                masterStorageEntityProfile.setBusinessStaticFields(staticValuesProfile);
                masterStorageEntityProfile.setBusinessDynamicFields(dynamicValues);

                //  设置动态.
//                Map<String, Object> painValues = MasterStorageHelper.toPainValues(dynamicValues);

//                if (op.equals(Operation.REPLACE)) {
//                    masterStorageEntityProfile.setBusinessDynamicFields(MasterStorageHelper.buildReplace(painValues, mySQLCodec));
//                    masterStorageEntityProfile.setBusinessDynamicFieldsRemove(MasterStorageHelper.buildRemove(painValues));
//                } else {
//                    masterStorageEntityProfile.setBusinessDynamicFields(MasterStorageHelper.toBuildJson(painValues));
//                }

                systemColumnProfile.setDeleted(false);
            } else {
                systemColumnProfile.setDeleted(true);
            }

            masterStorageEntities.add(masterStorageEntityProfile);
            return masterStorageEntities;
        }
    }

    private List<MasterStorageEntity> toMasterDocumentEntity(IEntityClass e, String table, Collection<OqsEngineEntity> entities) {
        Collection<IEntityField> selfFields = e.selfWithIndex();

        List<MasterStorageEntity> masterStorageEntities = new ArrayList<>();
        for (OqsEngineEntity o : entities) {
            MasterStorageEntity masterStorageEntity = new MasterStorageEntity();

            //  设置系统字段
            SystemColumn systemColumn = new SystemColumn();
            systemColumn.setProfile(o.getEntityClassRef().getProfile());
            systemColumn.setOperateTime(o.getUpdateTime());
            systemColumn.setEntityClass(e.id());
            systemColumn.setId(o.getId());
            masterStorageEntity.setSystemColumn(systemColumn);
            //TODO
            masterStorageEntity.setTableName(table);

            List<MasterStorageEntity.TypedStorageValue> staticValues = new ArrayList<>();
            List<IValue> dynamicValues = new ArrayList<>();

            for (IEntityField entityField : selfFields) {
                
                Object value = o.getAttributes().get(entityField.name());
                MasterStorageEntity.TypedStorageValue targetValue = new MasterStorageEntity.TypedStorageValue(entityField, value);

                //TODO
                if (null != value) {
                        if(value instanceof ExpressionValue) {
                            targetValue.setKeepRaw(true);
                        }
                        staticValues.add(targetValue);
                    }
//                }
            }

            masterStorageEntity.setBusinessDynamicFields(dynamicValues);
            masterStorageEntity.setBusinessStaticFields(staticValues);

            systemColumn.setDeleted(false);

            masterStorageEntities.add(masterStorageEntity);
        }
        return masterStorageEntities;
    }


    private int getEnumBit(String enumId, List<String> values) {
        //TODO profile
        List<DictItem> dictItems = dictService.findDictItems(enumId, null, Collections.emptyMap());
        Optional<Integer> reduce = values.stream().map(x -> dictService.findEnumIndex(dictItems, x)).reduce((a, b) -> a | b);
        return reduce.orElse(0);
    }

    private static final class OQSFrameworkConfig {
        private SchemaPlus rootSchema;
        private JdbcSchema oqsSchema;
        private FrameworkConfig frameworkConfig;

        public OQSFrameworkConfig(DataSource dataSource) {
            rootSchema = CalciteSchema.createRootSchema(true, false).plus();
            if(dataSource instanceof LoggerDataSource) {
                DataSource delegate = ((LoggerDataSource) dataSource).getDelegate();
                if(delegate instanceof HikariDataSource) {
                    oqsSchema = JdbcSchema.create(rootSchema, "oqs", dataSource,  ((HikariDataSource)dataSource).getCatalog(), null);
                } else if(delegate instanceof ShardingSphereDataSource) {
                    try {
                        String catalog = delegate.getConnection().getMetaData().getConnection().getCatalog();
                        oqsSchema = JdbcSchema.create(rootSchema, "oqs", dataSource,  catalog, null);
                    } catch (Throwable throwable) {
                        throwable.printStackTrace();
                    }
                }
            }
            rootSchema.add("oqs", oqsSchema);

            Frameworks.ConfigBuilder configBuilder = Frameworks.newConfigBuilder();
            this.frameworkConfig = configBuilder.defaultSchema(rootSchema)
                    .parserConfig(SqlParser.config().withLex(Lex.MYSQL).withCaseSensitive(false))
                    .context(new Context() {
                        @Override
                        public <C> C unwrap(Class<C> aClass) {
                            if (aClass == RelBuilder.Config.class) {
                                return (C) RelBuilder.Config.DEFAULT.withSimplify(false);
                            } else {
                                return null;
                            }
                        }
                    })
                    .build();

        }

        public JdbcSchema getOqsSchema() {
            return oqsSchema;
        }

        public SchemaPlus getRootSchema() {
            return rootSchema;
        }


        public FrameworkConfig getFrameworkConfig() {
            return frameworkConfig;
        }
    }
}
