package com.xforceplus.ultraman.oqsengine.plus.master.mysql.executor;

import com.google.common.collect.Sets;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.values.IValue;
import com.xforceplus.ultraman.oqsengine.plus.common.StringUtils;
import com.xforceplus.ultraman.oqsengine.plus.common.executor.Executor;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.ConditionalStorageEntity;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.ConditionalStoragePackage;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.MasterStorageEntity;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.StorageType;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.SystemColumn;
import com.xforceplus.ultraman.sdk.core.utils.MasterStorageHelper;
import com.xforceplus.ultraman.sdk.infra.codec.MySQLCodecCustom;
import com.xforceplus.ultraman.sdk.infra.exceptions.InvalidInputsException;
import org.apache.tinkerpop.gremlin.process.traversal.P;
import org.owasp.esapi.ESAPI;
import org.owasp.esapi.codecs.Codec;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * TODO
 */
public class ConditionalReplaceExecutor extends AbstractMasterTaskExecutor<List<ConditionalStoragePackage>, Integer> {

    private boolean isCheckConditionalLimit = false;
    
    private Codec<Character> mySQLCodec = new MySQLCodecCustom(MySQLCodecCustom.Mode.STANDARD);

    private final static String LIMIT_COUNT = "SELECT COUNT(*) from %s where %s";

    public ConditionalReplaceExecutor(Connection connection, long time, boolean isCheckConditionalLimit) {
        super(connection, time);
        this.isCheckConditionalLimit = isCheckConditionalLimit;
    }

    public static Executor<List<ConditionalStoragePackage>, Integer> build(
            Connection connection, long timeout, boolean isCheckConditionalLimit) {
        return new ConditionalReplaceExecutor(connection, timeout, isCheckConditionalLimit);
    }

    @Override
    public Integer execute(List<ConditionalStoragePackage> storagePackage) throws SQLException {
        
        if(storagePackage.isEmpty()) {
            return 0;
        }

        try (Statement st = getConnection().createStatement()) {
            if (storagePackage.size() > 1) {
                storagePackage.forEach(x -> {
                    try {
                        String sql = buildSql(x);
                        st.addBatch(sql);
                    } catch (SQLException e) {
                        throw new RuntimeException(e);
                    }
                });

                int[] ints = st.executeBatch();
                return Arrays.stream(ints).sum();
            } else {
                ConditionalStoragePackage targetStoragePackage = storagePackage.get(0);
                if (isCheckConditionalLimit && targetStoragePackage.getLimit() > 0) {
                    //should check limit
                    String limitSql = String.format(LIMIT_COUNT, targetStoragePackage.getMainTableName(), targetStoragePackage.getConditionSql());
                    ResultSet resultSet = st.executeQuery(limitSql);
                    if (resultSet.next()) {
                        int limit = resultSet.getInt(1);
                        if (limit > targetStoragePackage.getLimit()) {
                            throw new InvalidInputsException(InvalidInputsException.getMsg("Conditional replace exceed the request limit"));
                        }
                    }
                }

                return st.executeUpdate(buildSql(targetStoragePackage));
                
            }
        }
    }

    private String buildSql(ConditionalStoragePackage storagePackage) throws SQLException {

        /**
         * join table
         */
        String mainTableName = storagePackage.getMainTableName();

        //combine multi into one
        StringBuilder base = new StringBuilder();
        base.append("UPDATE ").append(mainTableName).append(" SET ");
        
        String conditionSql = storagePackage.getConditionSql();
        List<ConditionalStorageEntity> conditionalStorageList = storagePackage.getConditionalStorageList();

        boolean isFirst = true;
        long sampleTime = 0L;

        /**
         * TOBE updated
         */
        List<String> relatedTable = storagePackage.getRelatedTable();
        Set<String> todo = new HashSet<>(relatedTable);
        Set<String> done = new HashSet<>();
        
        for (ConditionalStorageEntity storageEntity : conditionalStorageList) {
            if(isFirst) {
                isFirst = false;
            } else {
                base.append(",");
            }
            
            String targetTable = storageEntity.getTargetTable();
            done.add(targetTable);
            String concatColumn = targetTable.concat(".").concat(SystemColumn.SYS_VERSION);
            String concatOperate = targetTable.concat(".").concat(SystemColumn.SYS_OPERATE_TIME);
            base.append(concatColumn).append(" = ").append(concatColumn).append(" +1 ");
            sampleTime = storageEntity.getSystemColumn().getOperateTime();
            base.append(",").append(concatOperate).append(" = ").append(storageEntity.getSystemColumn().getOperateTime());

            for (MasterStorageEntity.TypedStorageValue value : storageEntity.getBusinessStaticFields()) {
                String columnName = value.getEntityField().name();
                if (!SystemColumn.SYSTEM_WORDS.contains(columnName)) {
                    StorageType storageType = StorageType.instance(value.getEntityField().type());
                    if (null == storageType) {
                        throw new SQLException("storageType not found.");
                    }

                    base.append(",")
                            .append(targetTable.concat(".").concat(MasterStorageHelper.toStorageName(columnName, true)))
                            .append("=");

                    if (value.getValue() == null) {
                        base.append("null");
                    } else {
                        Object targetValue;
                        if(value.isKeepRaw()) {
                            targetValue = value.getValue();
                        } else {
                            targetValue = storageType.isNeedSymbol() ?
                                    String.format("'%s'", ESAPI.encoder().encodeForSQL(mySQLCodec, value.getValue().toString()))
                                    : value.getValue();
                        }
                        base.append(targetValue);
                    }
                }
            }

            List<IValue> businessDynamicFields = storageEntity.getBusinessDynamicFields();

            /**
             * TODO
             */
            if(businessDynamicFields != null && !businessDynamicFields.isEmpty()) {
                Map<String, Object> attachment = storageEntity.getAttachment();
                Map<String, Object> painValues = MasterStorageHelper.toPainValues(businessDynamicFields);

                Map<String, Object> dynamicMap = new HashMap<>(painValues);
                dynamicMap.putAll(attachment);

                String replace = MasterStorageHelper.buildReplace(dynamicMap, mySQLCodec, targetTable);
                String remove = MasterStorageHelper.buildRemove(dynamicMap, targetTable);

                if (!replace.isEmpty()) {
                    base.append(",");
                    base.append(targetTable.concat(".").concat(SystemColumn.DYNAMIC_FIELD)).append("=").append(replace);
                }

                if (!remove.isEmpty()) {
                    base.append(",");
                    base.append(targetTable.concat(".").concat(SystemColumn.DYNAMIC_FIELD)).append("=").append(remove);
                }
            }
            
        }

        for (String target : Sets.difference(todo, done)) {
            if(isFirst) {
                isFirst = false;
            } else {
                base.append(",");
            }
            String concatColumn = target.concat(".").concat(SystemColumn.SYS_VERSION);
            String concatOperate = target.concat(".").concat(SystemColumn.SYS_OPERATE_TIME);
            base.append(concatColumn).append(" = ").append(concatColumn).append(" +1 ");
            base.append(",").append(concatOperate).append(" = ").append(sampleTime);
        }
        
        if (!StringUtils.isEmpty(conditionSql)) {
            base.append(" where ").append(conditionSql);
        } else {
            throw new InvalidInputsException(InvalidInputsException.getMsg("Conditional update should always has a range"));
        }
        
    
        

        return base.toString();
    }
}
