package com.xforceplus.ultraman.oqsengine.plus.master.mysql.executor;

import com.xforceplus.metadata.schema.typed.BoIndex;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.values.IValue;
import com.xforceplus.ultraman.metadata.values.OnDuplicateValue;
import com.xforceplus.ultraman.oqsengine.plus.common.executor.Executor;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.Operation;
import com.xforceplus.ultraman.sdk.infra.codec.MySQLCodecCustom;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.MasterStorageEntity;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.StorageType;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.SystemColumn;
import com.xforceplus.ultraman.sdk.core.utils.MasterStorageHelper;
import org.owasp.esapi.ESAPI;
import org.owasp.esapi.codecs.Codec;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.*;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.Operation.REPLACE;

/**
 * Created by justin.xu on 03/2023.
 *
 * @since 1.8
 */
public class CreateExecutor extends AbstractMasterTaskExecutor<List<MasterStorageEntity>, Boolean[]> {

    /**
     * TODO
     * current static inited
     */
    private Codec<Character> mySQLCodec = new MySQLCodecCustom(MySQLCodecCustom.Mode.STANDARD);

    public CreateExecutor(Connection connection, long timeout) {
        super(connection, timeout);
    }

    public static Executor<List<MasterStorageEntity>, Boolean[]> build(Connection connection, long timeout) {
        return new CreateExecutor(connection, timeout);
    }

    @Override
    public Boolean[] execute(List<MasterStorageEntity> collection) throws SQLException {
        try (Statement st = getConnection().createStatement()) {

            checkTimeout(st);

            boolean isBatch = collection.size() > 1;
            if (isBatch) {
                for (int i = 0; i < collection.size(); i++) {
                    //  建SQL.
                    String sql = buildSql(collection.get(i));
                    st.addBatch(sql);
                }

                return executedUpdate(st, true);
            } else {
                String sql = buildSql(collection.get(0));
                return new Boolean[]{st.executeUpdate(sql) > 0};
            }
        }
    }

    protected String buildSql(MasterStorageEntity masterStorageEntity) throws SQLException {
        StringBuilder base = new StringBuilder();
        base.append("INSERT INTO ").append(masterStorageEntity.getTableName()).append(" (");

        //  systemField.
        base.append(String.join(",", SystemColumn.ID, SystemColumn.SYS_ENTITY_CLASS, SystemColumn.SYS_VERSION, SystemColumn.SYS_OPERATE_TIME, SystemColumn.SYS_DELETED, SystemColumn.SYS_PROFILE));

        //  static field
        if (!masterStorageEntity.getBusinessStaticFields().isEmpty()) {
            for (MasterStorageEntity.TypedStorageValue value : masterStorageEntity.getBusinessStaticFields()) {
                String columnName = value.getEntityField().name().toLowerCase();
                if (!SystemColumn.SYSTEM_WORDS.contains(columnName)) {
                    base.append(",").append(MasterStorageHelper.toStorageName(columnName, true));
                }
            }
        }

        //  dynamic field
        //if (!masterStorageEntity.getBusinessDynamicFields().isEmpty()) {
        base.append(",").append(SystemColumn.DYNAMIC_FIELD);
        //}

        base.append(") VALUES (");

        base.append(masterStorageEntity.getSystemColumn().getId()).append(",").append(masterStorageEntity.getSystemColumn().getEntityClass()).append(",").append(masterStorageEntity.getSystemColumn().getVersion()).append(",").append(masterStorageEntity.getSystemColumn().getOperateTime()).append(",").append(masterStorageEntity.getSystemColumn().getDeleted()).append(",").append("'").append(masterStorageEntity.getSystemColumn().getProfile()).append("'");

        //  static field
        if (!masterStorageEntity.getBusinessStaticFields().isEmpty()) {
            for (MasterStorageEntity.TypedStorageValue value : masterStorageEntity.getBusinessStaticFields()) {
                if (!SystemColumn.SYSTEM_WORDS.contains(value.getEntityField().name())) {
                    StorageType storageType = StorageType.instance(value.getEntityField().type());
                    if (null == storageType) {
                        throw new SQLException("storageType not found.");
                    }

                    if (value.getValue() != null) {
                        Object targetValue;
                        if(value.isKeepRaw()) {
                            targetValue = value.getValue();
                        } else {
                            targetValue = storageType.isNeedSymbol() ? String.format("'%s'", ESAPI.encoder().encodeForSQL(mySQLCodec, value.getValue().toString())) : value.getValue();
                        }
                        base.append(",").append(targetValue);
                    } else {
                        base.append(",").append("null");
                    }
                }
                
            }
        }

        //  dynamic field
        if (!masterStorageEntity.getBusinessDynamicFields().isEmpty()) {
            List<IValue> businessDynamicFields = masterStorageEntity.getBusinessDynamicFields();
            Map<String, Object> attachment = masterStorageEntity.getAttachment();
            Map<String, Object> painValues = MasterStorageHelper.toPainValues(businessDynamicFields);
            
            Map<String, Object> dynamicMap = new HashMap<>(painValues);
            dynamicMap.putAll(attachment);
            
            base.append(",").append("'").append(MasterStorageHelper.toBuildJson(dynamicMap)).append("'");
        } else {
            base.append(",").append("'").append("{}").append("'");
        }

        base.append(") ");
        boolean hasOnDuplicated = checkHashOnDuplicated(masterStorageEntity);
        if(hasOnDuplicated) {
            base.append(" ON DUPLICATE KEY UPDATE ");
            base.append(SystemColumn.SYS_VERSION).append("=").append(SystemColumn.SYS_VERSION).append("+1");
            base.append(",").append(SystemColumn.SYS_OPERATE_TIME).append("=").append(masterStorageEntity.getSystemColumn().getOperateTime());

            if (!masterStorageEntity.getBusinessStaticFields().isEmpty()) {
                for (MasterStorageEntity.TypedStorageValue value : masterStorageEntity.getBusinessStaticFields()) {
                    String columnName = value.getEntityField().name();
                    if(value.getRawValue() instanceof OnDuplicateValue) {
                        if (!SystemColumn.SYSTEM_WORDS.contains(columnName)) {
                            StorageType storageType = StorageType.instance(value.getEntityField().type());
                            if (null == storageType) {
                                throw new SQLException("storageType not found.");
                            }

                            base.append(",")
                                    .append(MasterStorageHelper.toStorageName(columnName, true))
                                    .append("=");

                            Object o = ((OnDuplicateValue) value.getRawValue()).storageValue(REPLACE);
                            if (o == null) {
                                base.append("null");
                            } else {
                                Object targetValue = o;
                                if (value.isKeepRawOnDuplicate()) {
                                    targetValue = o;
                                } else {
                                    targetValue = storageType.isNeedSymbol() ?
                                            String.format("'%s'", ESAPI.encoder().encodeForSQL(mySQLCodec, targetValue.toString()))
                                            : targetValue;
                                }
                                base.append(targetValue);
                            }
                        }
                    } else if (SystemColumn.UPDATE_WORDS.contains(columnName)) {
                        StorageType storageType = StorageType.instance(value.getEntityField().type());
                        if (null == storageType) {
                            throw new SQLException("storageType not found.");
                        }

                        base.append(",")
                                .append(MasterStorageHelper.toStorageName(columnName, true))
                                .append("=");

                        if (value.getValue() != null) {
                            Object targetValue;
                            if(value.isKeepRaw()) {
                                targetValue = value.getValue();
                            } else {
                                targetValue = storageType.isNeedSymbol() ? String.format("'%s'", ESAPI.encoder().encodeForSQL(mySQLCodec, value.getValue().toString())) : value.getValue();
                            }
                            base.append(targetValue);
                        } else {
                            base.append("null");
                        }
                    }
                }
            }

            List<IValue> businessDynamicFields = masterStorageEntity.getBusinessDynamicFields();
            Map<String, Object> attachment = masterStorageEntity.getAttachment();
            List<OnDuplicateValue> onDuplicatedValues = businessDynamicFields.stream().filter(x -> x instanceof OnDuplicateValue)
                    .map(x -> (OnDuplicateValue)x).collect(Collectors.toList());

            Map<String, Object> painValues = MasterStorageHelper.toPainOnDuplicateValues(onDuplicatedValues);

            Map<String, Object> dynamicMap = new HashMap<>(painValues);
            dynamicMap.putAll(attachment);

            String replace = MasterStorageHelper.buildReplace(dynamicMap, mySQLCodec, null);
            String remove = MasterStorageHelper.buildRemove(dynamicMap, null);

            if (!replace.isEmpty()) {
                base.append(",");
                base.append(SystemColumn.DYNAMIC_FIELD).append("=").append(replace);
            }

            if (!remove.isEmpty()) {
                base.append(",");
                base.append(SystemColumn.DYNAMIC_FIELD).append("=").append(remove);
            }
        }

        return base.toString();
    }
    
    boolean checkHashOnDuplicated(MasterStorageEntity masterStorageEntity) {
        EntityClassGroup rawEntityClass = masterStorageEntity.getRawEntityClass();
        Collection<BoIndex> indexes = rawEntityClass.getEntityClass().indexes();
        List<BoIndex> onDuplicated = indexes.stream().filter(x -> x.getRule() == 1).collect(Collectors.toList());
        return !onDuplicated.isEmpty();
    }
}
