package com.xforceplus.ultraman.oqsengine.plus.master.mysql.executor;

import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.values.IValue;
import com.xforceplus.ultraman.oqsengine.plus.common.executor.Executor;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.MasterEntity;
import com.xforceplus.ultraman.sdk.infra.codec.MySQLCodecCustom;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.MasterStorageEntity;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.StorageType;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.SystemColumn;
import com.xforceplus.ultraman.sdk.core.utils.MasterStorageHelper;
import com.xforceplus.ultraman.sdk.infra.exceptions.SQLRelatedException;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import org.owasp.esapi.ESAPI;
import org.owasp.esapi.codecs.Codec;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 * Created by justin.xu on 03/2023.
 *
 * @since 1.8
 */
public class UpdateExecutor extends AbstractMasterTaskExecutor<List<MasterStorageEntity>, Boolean[]> {

    /**
     * TODO
     * current static inited
     */
    private Codec<Character> mySQLCodec = new MySQLCodecCustom(MySQLCodecCustom.Mode.STANDARD);

    public UpdateExecutor(Connection resource, long timeout) {
        super(resource, timeout);
    }

    public static Executor<List<MasterStorageEntity>, Boolean[]> build(
            Connection resource, long timeout) {
        return new UpdateExecutor(resource, timeout);
    }

    @Override
    public Boolean[] execute(List<MasterStorageEntity> masterStorageEntities) throws SQLException {
        boolean isBatch = masterStorageEntities.size() > 1;
        
        List<Integer> focusRetIndex = new ArrayList<>();
        AtomicInteger index = new AtomicInteger(0);
        try (Statement st = getConnection().createStatement()) {

            Map<String, List<MasterStorageEntity>> groupedMapping = masterStorageEntities.stream().filter(x -> x.isUseOptimisticLock() || x.getSystemColumn().getVersion() >  -1)
                    .collect(Collectors.groupingBy(MasterEntity::getTableName));

            Map<Tuple2<String, Long>, Integer> versionMap = new HashMap<>();

            if (!groupedMapping.isEmpty()) {
                //get the current every storage version
                groupedMapping.forEach((k, list) -> {
                    String querySql = buildVersionQuerySql(k, list);
                    try {
                        ResultSet resultSet = st.executeQuery(querySql);
                        while (resultSet.next()) {
                            long id = resultSet.getLong("id");
                            Object sysVer = resultSet.getObject("_sys_ver");
                            if (sysVer != null) {
                                versionMap.put(Tuple.of(k, id)
                                        , resultSet.getInt("_sys_ver"));
                            }
                        }
                    } catch (SQLException e) {
                        throw new SQLRelatedException(e);
                    }
                });
            }

            masterStorageEntities.stream()
                    .filter(x -> x.isUseOptimisticLock() && x.getSystemColumn().getVersion() >= 0)
                    .forEach(masterStorageEntity -> {
                        String tableName = masterStorageEntity.getTableName();
                        long id = masterStorageEntity.getSystemColumn().getId();
                        versionMap.put(Tuple.of(tableName, id), Long.valueOf(masterStorageEntity.getSystemColumn().getVersion()).intValue());
                    });

            if (isBatch) {
                for (MasterStorageEntity entity : masterStorageEntities) {

                    Integer ver = versionMap.get(Tuple.of(entity.getTableName(), entity.getSystemColumn().getId()));
                    //  建SQL.
                    String sql = buildSql(
                            entity,
//                            entity.getBusinessDynamicFields(),
//                            entity.getBusinessDynamicFieldsRemove(),
                            ver
                    );

                    st.addBatch(sql);
                    focusRetIndex.add(index.getAndIncrement());

                    if(entity.isOnlyFather()) {
                        /**
                         * should build sql for children
                         * current omit child
                         */
                        List<String> sqlList = buildChildSql(entity);
                        for(String childSql : sqlList) {
                            st.addBatch(childSql);
                            index.incrementAndGet();
                        }
                    }
                }

                Boolean[] booleans = executedUpdate(st, true);
                //mapping focus 
                return focusRetIndex.stream().map(x -> booleans[x]).toArray(Boolean[]::new);
            } else {
                //when single can do optimize
                MasterStorageEntity entity = masterStorageEntities.get(0);
                Integer ver = versionMap.get(Tuple.of(entity.getTableName(), entity.getSystemColumn().getId()));

                //  建SQL.
                String sql = buildSql(
                        entity,
//                        entity.getBusinessDynamicFields(),
//                        entity.getBusinessDynamicFieldsRemove(),
                        ver
                );
                focusRetIndex.add(index.getAndIncrement());
                st.addBatch(sql);

                if(entity.isOnlyFather()) {

                    /**
                     * should build sql for children
                     */
                    List<String> sqlList = buildChildSql(entity);
                    sqlList.stream().forEach(targetSql -> {
                        try {
                            st.addBatch(targetSql);
                            index.incrementAndGet();
                        } catch (SQLException e) {
                            e.printStackTrace();
                        }
                    });
                }

                Boolean[] booleans = executedUpdate(st, true);
                return focusRetIndex.stream().map(x -> booleans[x]).toArray(Boolean[]::new);
            }
        }
    }


    protected List<String> buildChildSql(MasterStorageEntity masterStorageEntity) {

        EntityClassGroup rawEntityClass = masterStorageEntity.getRawEntityClass();
        Collection<IEntityClass> childrenEntityClass = rawEntityClass.getChildrenEntityClass();
        List<String> childrenSql = new ArrayList<>();
        for (IEntityClass entityClass : childrenEntityClass) {
            String targetTable = entityClass.masterWriteTable(masterStorageEntity.isProfiled());
            StringBuilder base = new StringBuilder();
            base.append(String.format("UPDATE %s SET ", targetTable));
            base.append(SystemColumn.SYS_VERSION).append("=").append(SystemColumn.SYS_VERSION).append("+1");
            base.append(",").append(SystemColumn.SYS_OPERATE_TIME).append("=")
                    .append(masterStorageEntity.getSystemColumn().getOperateTime());

            base.append(" ").append("where id").append("=").append(masterStorageEntity.getSystemColumn().getId());
            String targetSql = base.toString();
            childrenSql.add(targetSql);
        }
        return childrenSql;
    }

    protected String buildVersionQuerySql(String tableName, List<MasterStorageEntity> entities) {
        StringBuilder base = new StringBuilder();
        base.append("SELECT id, _sys_ver FROM ")
                .append(tableName)
                .append(" WHERE id in (")
                .append(entities.stream()
                        .map(x -> String.valueOf(x.getSystemColumn().getId()))
                        .collect(Collectors.joining(",")))
                .append(")");
        return base.toString();
    }

    protected String buildSql(MasterStorageEntity entity, Integer ver) throws SQLException {
        StringBuilder base = new StringBuilder();
        base.append(String.format("UPDATE %s SET ", entity.getTableName()));
        base.append(SystemColumn.SYS_VERSION).append("=").append(SystemColumn.SYS_VERSION).append("+1");
        base.append(",").append(SystemColumn.SYS_OPERATE_TIME).append("=").append(entity.getSystemColumn().getOperateTime());

        if (!entity.getBusinessStaticFields().isEmpty()) {
            for (MasterStorageEntity.TypedStorageValue value : entity.getBusinessStaticFields()) {

                String columnName = value.getEntityField().name();
                if (!SystemColumn.SYSTEM_WORDS.contains(columnName)) {
                    StorageType storageType = StorageType.instance(value.getEntityField().type());
                    if (null == storageType) {
                        throw new SQLException("storageType not found.");
                    }

                    base.append(",")
                            .append(MasterStorageHelper.toStorageName(columnName, true))
                            .append("=");

                    if (value.getValue() == null) {
                        base.append("null");
                    } else {
                        Object targetValue;
                        if(value.isKeepRaw()) {
                            targetValue = value.getValue();
                        } else {
                            targetValue = storageType.isNeedSymbol() ?
                                    String.format("'%s'", ESAPI.encoder().encodeForSQL(mySQLCodec, value.getValue().toString()))
                                    : value.getValue();
                        }
                        base.append(targetValue);
                        
                    }
                }
            }
        }

        List<IValue> businessDynamicFields = entity.getBusinessDynamicFields();
        
        if(businessDynamicFields != null && !businessDynamicFields.isEmpty()) {
            Map<String, Object> attachment = entity.getAttachment();
            Map<String, Object> painValues = MasterStorageHelper.toPainValues(businessDynamicFields);

            Map<String, Object> dynamicMap = new HashMap<>(painValues);
            dynamicMap.putAll(attachment);

            String replace = MasterStorageHelper.buildReplace(dynamicMap, mySQLCodec, null);
            String remove = MasterStorageHelper.buildRemove(dynamicMap, null);

            if (!replace.isEmpty()) {
                base.append(",");
                base.append(SystemColumn.DYNAMIC_FIELD).append("=").append(replace);
            }
    
            if (!remove.isEmpty()) {
                base.append(",");
                base.append(SystemColumn.DYNAMIC_FIELD).append("=").append(remove);
            }
        }

        base.append(" ").append("where id").append("=").append(entity.getSystemColumn().getId());
        base.append(" AND ").append(SystemColumn.SYS_DELETED).append("=").append(false);
        if (ver != null) {
            base.append(" AND ");
            base.append(SystemColumn.SYS_VERSION).append("=").append(ver);
        }

        return base.toString();
    }
}
