package com.xforceplus.ultraman.oqsengine.plus.master.mysql.query;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Range;
import com.google.common.collect.RangeSet;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.metadata.helper.ConvertHelper;
import com.xforceplus.ultraman.metadata.service.DictService;
import com.xforceplus.ultraman.oqsengine.plus.common.StringUtils;
import com.xforceplus.ultraman.oqsengine.plus.master.calcite.visitor.RemoveSortWithLimitVisitor;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.MasterQueryEntity;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.MasterQueryResult;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.MysqlSqlDialectEx;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.SQLMasterStorage;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.executor.AbstractMasterTaskExecutor;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.SystemColumn;
import com.xforceplus.ultraman.oqsengine.plus.storage.pojo.dto.select.SelectConfig;
import com.xforceplus.ultraman.sdk.core.utils.MasterStorageHelper;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import io.vavr.Tuple3;
import lombok.extern.slf4j.Slf4j;
import org.apache.calcite.DataContext;
import org.apache.calcite.plan.RelOptUtil;
import org.apache.calcite.plan.hep.HepPlanner;
import org.apache.calcite.plan.hep.HepProgram;
import org.apache.calcite.rel.RelFieldCollation;
import org.apache.calcite.rel.RelNode;
import org.apache.calcite.rel.core.AggregateCall;
import org.apache.calcite.rel.rel2sql.RelToSqlConverter;
import org.apache.calcite.rel.type.RelDataType;
import org.apache.calcite.rel.type.RelDataTypeField;
import org.apache.calcite.rel.type.StructKind;
import org.apache.calcite.rex.*;
import org.apache.calcite.sql.SqlNode;
import org.apache.calcite.sql.SqlUtil;
import org.apache.calcite.sql.dialect.MysqlSqlDialect;
import org.apache.calcite.sql.fun.SqlStdOperatorTable;
import org.apache.calcite.sql.type.SqlTypeName;
import org.apache.calcite.tools.FrameworkConfig;
import org.apache.calcite.tools.RelBuilder;
import org.apache.calcite.util.Pair;
import org.apache.calcite.util.Sarg;
import org.apache.calcite.util.Util;
import org.springframework.beans.factory.annotation.Autowired;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * Created by justin.xu on 03/2023.
 *
 * @since 1.8
 */
@Slf4j
public abstract class AbstractMasterQueryExecutor<T, R> extends AbstractMasterTaskExecutor<T, R> {

    private static ZoneId zoneId = ZoneId.of("Asia/Shanghai");
    private static DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    private String schemaName = "oqsengine";
    private List<Map.Entry<String, Tuple2<StructKind, Class>>> targetFields;

    private String QUERY_RAW_POLY = "SELECT id, _sys_profile, _sys_entityclass from %s where id in (%s) and (_sys_profile != ? or _sys_entityclass != ?)";

    private String QUERY_RAW_VER = "SELECT id, _sys_ver from %s where id in (%s)";
    private ObjectMapper mapper;
    
    private EntityClassEngine engine;
    
    private boolean useStrictEnum;
    
    private DictService dictService;

    private Map<String, List<String>> rewriteMapping;
    
    private boolean isLow = true;

    private TypeReference<List<String>> typeReference = new TypeReference<List<String>>() {
    };

    private SQLMasterStorage storage;

    public AbstractMasterQueryExecutor(EntityClassEngine engine, DictService dictService, ObjectMapper mapper
            , Connection connection, long timeoutMs, boolean useStrictEnum, Map<String, List<String>> rewriteMapping
            , boolean isLow, SQLMasterStorage storage                       
    ) {
        super(connection, timeoutMs);
        this.mapper = mapper;
        this.engine = engine;
        this.useStrictEnum = useStrictEnum;
        this.dictService = dictService;
        this.rewriteMapping = rewriteMapping;
        this.isLow = isLow;
        this.storage = storage;
    }

    private static String nameConvert(String originName, IEntityClass entityClass) {
        originName = originName.toLowerCase();
        Optional<IEntityField> fieldOp = entityClass.field(originName);
        if (fieldOp.isPresent()) {
            IEntityField field = fieldOp.get();

            //  表示查询了一个dynamic字段 && !field.isIndex()
            if (field.isDynamic()) {
                return SystemColumn.DYNAMIC_FIELD + ".$" + originName;
            }
        }
        return originName;
    }

//    private void genProjectFromAgg(Pair<RexNode, String> projectPair, RelDataType dataType
//            , List<AggregateCall> aggs, IEntityClass entityClass, Map<String, String> finalMapping) {
//        RexNode inputRef = projectPair.getKey();
//        if (inputRef instanceof RexInputRef) {
//            int index = ((RexInputRef) inputRef).getIndex();
//            AggregateCall aggregateCall = aggs.get(index - 1);
//            String originField = dataType.getFieldList().get().getName();
//            String field = nameConvert(originField, entityClass);
//            finalMapping.put(field, projectPair.getValue());
//        }
//    }

    private void genProjectFromType(Pair<RexNode, String> projectPair
            , RelDataType dataType, IEntityClass entityClass
            , List<Pair<String, String>> fieldMapping) {
        RexNode inputRef = projectPair.getKey();
        if (inputRef instanceof RexInputRef) {
            String originField = dataType.getFieldList().get(((RexInputRef) inputRef).getIndex()).getName();
            String field = nameConvert(originField, entityClass);
            fieldMapping.add(new Pair<>(field, projectPair.getValue()));
        }
    }

    protected MasterQueryResult executeQuery(
            SelectConfig selectConfig
            , EntityClassGroup entityClassGroup
            , FrameworkConfig config) throws SQLException {

        Connection resource = getConnection();

        List<MasterQueryEntity> queryEntities = new ArrayList<>();

        try {
            RelBuilder relBuilder = makeRelBuilder(config);
            RelNode rawTree = selectConfig.getRawTree();
            String plan = RelOptUtil.toString(rawTree);
            log.debug("old plan {}", plan);
            //transform a new tree with new scan with old one
            CopyCustomShuttle copyCustomShuttle = new CopyCustomShuttle(relBuilder, dictService, entityClassGroup, new HashMap<>(), useStrictEnum, rewriteMapping);
            rawTree.accept(copyCustomShuttle);

            RelNode finalNode = relBuilder.build();
            HepProgram program = HepProgram.builder().build();
            HepPlanner planner = new HepPlanner(program);
            planner.setRoot(finalNode);
            RelNode optimizedNode = planner.findBestExp();

            /**
             * params
             */
            DataContext dataContext = selectConfig.getDataContext();
            List<RexDynamicParam> params = copyCustomShuttle.getParams();
            MasterQueryResult result = new MasterQueryResult();
            if (!onlyCount(selectConfig)) {

                String newPlan = RelOptUtil.toString(optimizedNode);
                log.debug("new plan {}", newPlan);

                SqlNode sqlNode = new RelToSqlConverter(isLow ? MysqlSqlDialectEx.DEFAULT_LOW : MysqlSqlDialectEx.DEFAULT_HIGH).visitRoot(optimizedNode).asStatement();
                String s = Util.toLinux(sqlNode.toSqlString(isLow ? MysqlSqlDialectEx.DEFAULT_LOW : MysqlSqlDialectEx.DEFAULT_HIGH).getSql()).replaceAll("\n", " ");
                log.debug("final sql {}", s);

                //final last
                List<Map.Entry<String, Tuple2<StructKind, Class>>> targetFields = selectConfig.getFields();

                /**
                 * find final key to get the value and to convert the type from old type to new type
                 * * the key
                 * * the origin key
                 * * the real type
                 * * the taregt type
                 */
                PreparedStatement preparedStatement = resource.prepareStatement(s);
                params.forEach(param -> {
                    int index = param.getIndex();
                    SqlTypeName sqlTypeName = param.getType().getSqlTypeName();
                    try {
                        preparedStatement.setObject(index + 1, dataContext.get(param.getName()), sqlTypeName.getJdbcOrdinal());
                    } catch (SQLException e) {
                        e.printStackTrace();
                    }
                });

                long queryStart = System.currentTimeMillis();
                try (ResultSet rs = preparedStatement.executeQuery()) {
                    log.debug("Main Query Consume: {}", (System.currentTimeMillis() - queryStart));

                    /**
                     * get value from targetField
                     */
                    for (Map.Entry<String, Tuple2<StructKind, Class>> targetField : targetFields) {
                        String key = targetField.getKey();
                        if (key.startsWith(SqlUtil.GENERATED_EXPR_ALIAS_PREFIX)) {
                            String index = key.substring(SqlUtil.GENERATED_EXPR_ALIAS_PREFIX.length());
                            //from 0
                            key = rs.getMetaData().getColumnLabel(Integer.parseInt(index) + 1);
                        }

                        result.appendField(key, targetField.getValue()._2);
                    }

                    while (rs.next()) {
                        result.newRecord();
                        result.getFields().forEach(x -> {
                            Object rs1 = MasterStorageHelper.getRs(rs, x._1, x._2);
                            result.append(x._1, rs1);
                        });
                    }
                }
            }

            if (hasPolymorphic(selectConfig)) {
                log.debug("Query with Polymorphic");
                //query multi
                int i = 0;
                for (Tuple2<String, Class> field : result.getFields()) {
                    if (field._1.equalsIgnoreCase("ID")) {
                        break;
                    }
                    i++;
                }

                if (i < result.getFields().size()) {
                    int finalId = i;
                    List<Long> idsList = result.getValues().stream().map(x -> x.get(finalId)).filter(Objects::nonNull)
                            .map(Objects::toString).map(Long::parseLong).collect(Collectors.toList());
                    Map<String, List<Tuple3<Long, String, Long>>> tableMapping = storage.selectSysKindColumn(idsList, entityClassGroup.getEntityClass());
                    selectConfig.getContext().put("poly", tableMapping);
                    result.getExtra().put("poly", tableMapping);
                }
            }

            //TODO mysql related
            if (hasCount(selectConfig)) {
                long start = System.currentTimeMillis();
                RelNode countNode = buildCount(relBuilder, optimizedNode);
                SqlNode countSqlNode = new RelToSqlConverter(isLow ? MysqlSqlDialectEx.DEFAULT_LOW : MysqlSqlDialectEx.DEFAULT_HIGH).visitRoot(countNode).asStatement();
                String countSql = Util.toLinux(countSqlNode.toSqlString(isLow ? MysqlSqlDialectEx.DEFAULT_LOW : MysqlSqlDialectEx.DEFAULT_HIGH).getSql()).replaceAll("\n", " ");
                log.debug("Count Query Consume: {}", (System.currentTimeMillis() - start));
                PreparedStatement statement = resource.prepareStatement(countSql);
                //do prepare
                params.forEach(param -> {
                    int index = param.getIndex();
                    SqlTypeName sqlTypeName = param.getType().getSqlTypeName();
                    try {
                        statement.setObject(index + 1, dataContext.get(param.getName()), sqlTypeName.getJdbcOrdinal());
                    } catch (SQLException e) {
                        e.printStackTrace();
                    }
                });
                try (ResultSet resultSet = statement.executeQuery()) {
                    if (resultSet.next()) {
                        long count = resultSet.getLong("c");
                        selectConfig.getContext().put("show_count", count);
                        result.getExtra().put("show_count", count);
                    }
                }
            }
            
            if(hasVer(selectConfig)) {
                int i = 0;
                for (Tuple2<String, Class> field : result.getFields()) {
                    if (field._1.equalsIgnoreCase("ID")) {
                        break;
                    }
                    i++;
                }

                if (i < result.getFields().size()) {
                    int finalId = i;
                    String ids = result.getValues().stream().map(x -> x.get(finalId))
                            .filter(Objects::nonNull)
                            .map(Object::toString)
                            .collect(Collectors.joining(","));

                    String systemSQL = String.format(QUERY_RAW_VER, String.join(",", entityClassGroup.getEntityClass().masterQueryTable()), ids);
                    log.debug("Version SQL is  {}", systemSQL);
                    PreparedStatement statement = resource.prepareStatement(systemSQL);
                    Map<Long, Integer> verMapping = new HashMap<>();
                    try (ResultSet resultSet = statement.executeQuery()) {
                        while (resultSet.next()) {
                            verMapping.put(resultSet.getLong(1), resultSet.getInt(2));
                        }
                    }

                    result.getExtra().put("ver", verMapping);
                }
            }
            
            return result;
        } catch (Throwable throwable) {
            throwable.printStackTrace();
            throw throwable;
        } finally {
////            calciteQueryPackage.destroyInstance();
//            if (null != resource && !resource.isClosed()) {
//                resource.close();
//            }
        }
    }

    private String findTargetTable(Tuple2<Long, String> tuple) {
        Optional<IEntityClass> load = engine.load(tuple._1.toString(), tuple._2);
        if(load.isPresent()) {
            return load.get().masterQueryTable();
        } else {
            return null;
        }
    }

    private boolean isPlainArrayJson(Object rs1) {
        if (rs1 instanceof String) {
            if (!StringUtils.isEmpty((String) rs1)) {
                if (((String) rs1).startsWith("[") && ((String) rs1).endsWith("]")) {
                    if (!((String) rs1).contains("{") && !((String) rs1).contains("}")) {
                        String substring = ((String) rs1).substring(1, ((String) rs1).length() - 1);
                        if (!substring.contains("[") && !substring.contains("]")) {
                            return true;
                        }
                    }
                }
            }
        }

        return false;
    }

    private boolean extractPlainArrayJson(SelectConfig selectConfig) {
        return selectConfig.getHints().stream().anyMatch(x -> x.hintName.equalsIgnoreCase("extract_array_json"));
    }

    /**
     * remove page size and limit
     *
     * @param builder
     * @param optimizedNode
     * @return
     */
    private RelNode buildCount(RelBuilder builder, RelNode optimizedNode) {
        RemoveSortWithLimitVisitor removeSortWithLimitVisitor = new RemoveSortWithLimitVisitor();
        builder.clear();
//        log.debug("check count explain {}", optimizedNode.explain());
        builder.push(optimizedNode.accept(removeSortWithLimitVisitor));
        builder.push(builder.aggregate(builder.groupKey(), builder.countStar("c")).build());
        return builder.build();
    }

    private boolean hasCount(SelectConfig selectConfig) {
        return selectConfig.getHints().stream().anyMatch(x -> x.hintName.equalsIgnoreCase("show_count"));
    }

    private boolean hasVer(SelectConfig selectConfig) {
        return selectConfig.getHints().stream().anyMatch(x -> x.hintName.equalsIgnoreCase("show_ver"));
    }

    private boolean hasPolymorphic(SelectConfig selectConfig) {
        return selectConfig.getHints().stream().anyMatch(x -> x.hintName.equalsIgnoreCase("polymorphic"));
    }

    private boolean onlyCount(SelectConfig selectConfig) {
        return selectConfig.getHints().stream().anyMatch(x -> x.hintName.equalsIgnoreCase("only_count"));
    }

    private RexNode[] toSorts(RelBuilder relBuilder, IEntityClass entityClass, List<Map.Entry<String, RelFieldCollation.Direction>> sorts) {
        int sortSize = sorts.size();
        RexNode[] rexNodes = new RexNode[sortSize];
        for (int i = 0; i < sortSize; i++) {
            Map.Entry<String, RelFieldCollation.Direction> sort = sorts.get(i);

            RexInputRef rexInputRef = relBuilder.field(nameConvert(sort.getKey(), entityClass));

            if (sort.getValue().isDescending()) {
                rexNodes[i] = relBuilder.desc(rexInputRef);
            } else {
                rexNodes[i] = rexInputRef;
            }
        }
        return rexNodes;
    }

    private RelBuilder makeRelBuilder(FrameworkConfig config) {
        return RelBuilder.create(config);
    }

    private RelBuilder.AggCall aggregateConvert(AggregateCall aggCall, RelBuilder relBuilder, RelDataType older, RelDataType newer, IEntityClass entityClass, String alise) {

        if (null == newer.getFieldNames() || newer.getFieldNames().isEmpty()) {
            throw new RuntimeException("agg convert failed, table relDataType is invalid.");
        }

        //  遍历获取新的位置.
        List<Integer> newArgList = new ArrayList<>();
        List<String> nameList = new ArrayList<>();
        Consumer<Integer> integerConsumer = arg -> {
            String field = older.getFieldList().get(arg).getName();
            String tableColumnName = field;

            if (!tableColumnName.equals("*")) {
                tableColumnName = nameConvert(field, entityClass);
            }
            nameList.add(tableColumnName);
            for (int i = 0; i < newer.getFieldNames().size(); i++) {
                if (tableColumnName.equals(newer.getFieldNames().get(i))) {
                    newArgList.add(i);
                    break;
                }
            }
        };
        aggCall.getArgList().forEach(integerConsumer);

        String name = aggCall.getName();
        if (!StringUtils.isEmpty(alise)) {
            name = alise;
        }

//        RelBuilder.AggCall aggCall1 = relBuilder.aggregateCall(AggregateCall.create(aggCall.getAggregation()
////                , aggCall.isDistinct(), aggCall.isApproximate(), aggCall.ignoreNulls()
////                , newArgList, aggCall.filterArg
////                , aggCall.distinctKeys, aggCall.getCollation(), newer, name));
        RelBuilder.AggCall aggCall1 = relBuilder.aggregateCall(aggCall.getAggregation(), nameList.stream().map(x -> relBuilder.field(x)).collect(Collectors.toList()));
        return aggCall1;
    }

    /**
     * table visitor.
     */
    public static class TableRexVisitor extends RexVisitorImpl<RexNode> {
        private RelBuilder builder;
        private IEntityClass entityClass;
        private RelDataType dataType;
        private RefNameValue refNameValue;


        public TableRexVisitor(RelBuilder builder, RelDataType type, IEntityClass entityClass) {
            super(true);
            this.builder = builder;
            this.entityClass = entityClass;
            this.dataType = type;
        }

        @Override
        public RexNode visitCall(RexCall call) {
            refNameValue = new RefNameValue(builder);

            switch (call.getKind()) {
                case AND:
                    builder = builder.filter(builder.and(call.getOperands().stream().map(operand -> operand.accept(this)).toArray(RexNode[]::new)));
                    return call;
                case OR:
                    builder = builder.filter(builder.or(call.getOperands().stream().map(operand -> operand.accept(this)).toArray(RexNode[]::new)));
                    return call;
                case EQUALS:
                    travel(call);
                    builder = builder.filter(builder.equals(refNameValue.toRexField(), refNameValue.toFirstRexLiteral()));
                    return call;
                case NOT_EQUALS:
                    travel(call);
                    builder = builder.filter(builder.notEquals(refNameValue.toRexField(), refNameValue.toFirstRexLiteral()));
                    return call;
                case GREATER_THAN:
                    travel(call);
                    builder = builder.filter(builder.greaterThan(refNameValue.toRexField(), refNameValue.toFirstRexLiteral()));
                    return call;
                case GREATER_THAN_OR_EQUAL:
                    travel(call);
                    builder = builder.filter(builder.greaterThanOrEqual(refNameValue.toRexField(), refNameValue.toFirstRexLiteral()));
                    return call;
                case LESS_THAN:
                    travel(call);
                    builder = builder.filter(builder.lessThan(refNameValue.toRexField(), refNameValue.toFirstRexLiteral()));
                    return call;
                case LESS_THAN_OR_EQUAL:
                    travel(call);
                    builder = builder.filter(builder.lessThanOrEqual(refNameValue.toRexField(), refNameValue.toFirstRexLiteral()));
                    return call;
                case IN: {
                    travel(call);
                    builder = builder.filter(builder.in(refNameValue.toRexField(), refNameValue.toRexLiterals()));
                    return call;
                }
                case SEARCH:
                case BETWEEN: {
                    travel(call);
                    RexNode[] rexNodes = refNameValue.toRexLiterals();

                    if (rexNodes.length != 2) {
                        throw new RuntimeException("between must have two range value");
                    }
                    builder = builder.filter(builder.between(refNameValue.toRexField(), rexNodes[0], rexNodes[1]));
                    return call;
                }
                case LIKE:
                    travel(call);
                    builder = builder.filter(builder
                            .getRexBuilder()
                            .makeCall(SqlStdOperatorTable.LIKE, refNameValue.toRexField(), refNameValue.toFirstRexLiteral()));
                    return call;
                case CAST:
                    travel(call);
                    return call;
                case PLUS:
                case MINUS:
                case TIMES:
                case DIVIDE:
                default:
                    throw new RuntimeException("Unsupported RexCall type: " + call.getKind());
            }
        }

        @Override
        public RexNode visitInputRef(RexInputRef inputRef) {
            String originField = dataType.getFieldList().get(inputRef.getIndex()).getName();
            String field = nameConvert(originField, entityClass);
            refNameValue.setField(field);
            return inputRef;
        }

        @Override
        public RexNode visitLiteral(RexLiteral literal) {
            rexLiteralParser(refNameValue, literal);
            return literal;
        }

        private void travel(RexCall call) {
            call.operands.forEach(c -> {
                c.accept(this);
            });
        }


        private Object rexLiteralParser(RefNameValue refNameValue, RexLiteral literal) {
            if (null == refNameValue.getField()) {
                throw new RuntimeException("filter field could not be null");
            }

            if (null == literal.getValue2()) {
                return literal;
            }

            RelDataTypeField realType = builder.peek().getRowType().getField(refNameValue.getField(), false, false);

//            String originValue = literal.getValue2().toString();

            SqlTypeName typeName = literal.getTypeName();
            switch (typeName) {
                case BOOLEAN:
                    // Unlike SqlLiteral, we do not allow boolean null.
                case INTEGER: // not allowed -- use Decimal
                case TINYINT:
                case SMALLINT:
                case DECIMAL:
                case DOUBLE:
                case FLOAT:
                case REAL:
                case BIGINT:
                case DATE:
                case TIME:
                case TIME_WITH_LOCAL_TIME_ZONE:
                case TIMESTAMP:
                case TIMESTAMP_WITH_LOCAL_TIME_ZONE:
                case INTERVAL_YEAR:
                case INTERVAL_YEAR_MONTH:
                case INTERVAL_MONTH:
                case INTERVAL_DAY:
                case INTERVAL_DAY_HOUR:
                case INTERVAL_DAY_MINUTE:
                case INTERVAL_DAY_SECOND:
                case INTERVAL_HOUR:
                case INTERVAL_HOUR_MINUTE:
                case INTERVAL_HOUR_SECOND:
                case INTERVAL_MINUTE:
                case INTERVAL_MINUTE_SECOND:
                case INTERVAL_SECOND:
                case VARCHAR: // not allowe
                case CHAR:
                case VARBINARY: // not allowed -- use Binary
                case BINARY:
                case SYMBOL:
                case ROW:
                case GEOMETRY:
                case ANY:
                case NULL:
                    Object singleValue = literal.getValue2();
                    String convert = ConvertHelper.convert(singleValue);
                    addTargetValue(realType, convert, singleValue);
                    break;
                case SARG:
                    Object rangeValue = literal.getValue2();
                    Sarg sarg = (Sarg) rangeValue;
                    RangeSet rangeSet = sarg.rangeSet;
                    if (sarg.isPoints()) {
                        //in
                        Set<Range> ranges = sarg.rangeSet.asRanges();
                        ranges.stream().forEach(range -> {
                            String converted = ConvertHelper.convert(range.lowerEndpoint());
                            addTargetValue(realType, converted, range.lowerEndpoint());
                        });
                    } else {
                        Optional<Range> first = sarg.rangeSet.asRanges().stream().findFirst();
                        Range range = first.get();
                        if (range.hasLowerBound()) {
                            String converted = ConvertHelper.convert(range.lowerEndpoint());
                            addTargetValue(realType, converted, range.lowerEndpoint());
                        }

                        if (range.hasUpperBound()) {
                            String converted = ConvertHelper.convert(range.upperEndpoint());
                            addTargetValue(realType, converted, range.upperEndpoint());
                        }
                    }
                    break;
                case MULTISET:
                default:
                    throw Util.unexpected(typeName);
            }

            return literal;
        }

        private void addTargetValue(RelDataTypeField realType, String originValue, Object rawObj) {
            if (null != realType) {
                switch (realType.getType().getSqlTypeName()) {
                    case BIGINT:
                        refNameValue.getLiterals().add(builder.literal(Long.parseLong(originValue)));
                        break;
                    case DECIMAL:
                        refNameValue.getLiterals().add(builder.literal(new BigDecimal(originValue)));
                        break;
                    case VARCHAR:
                        refNameValue.getLiterals().add(builder.literal(originValue));
                        break;
                    case BOOLEAN:
                        refNameValue.getLiterals().add(builder.literal(Boolean.parseBoolean(originValue)));
                        break;
                    case TIMESTAMP: {
                        if (rawObj instanceof Long) {
                            //turn long to date
                            LocalDateTime dateTime = LocalDateTime.ofInstant(Instant.ofEpochMilli((long) rawObj), zoneId);
                            //Timestamp timeStamp = Timestamp.from(Instant.ofEpochMilli((long) rawObj));
                            //TO String
                            refNameValue.getLiterals().add(builder.literal(dateTime.format(formatter)));
                        } else if (rawObj instanceof BigDecimal) {
                            //turn long to date
                            LocalDateTime dateTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(((BigDecimal) rawObj).longValue()), zoneId);
                            //Timestamp timeStamp = Timestamp.from(Instant.ofEpochMilli((long) rawObj));
                            //TO String
                            refNameValue.getLiterals().add(builder.literal(dateTime.format(formatter)));
                        } else {
                            refNameValue.getLiterals().add(builder.literal(originValue));
                        }

                        break;
                    }
                    default:
                        refNameValue.getLiterals().add(builder.literal(rawObj));
                }
            }
        }
    }

    private static class RefNameValue {
        private String field;
        private List<RexLiteral> literals;
        private RelBuilder relBuilder;

        public RefNameValue(RelBuilder relBuilder) {
            this.relBuilder = relBuilder;
            this.literals = new ArrayList<>();
        }

        public String getField() {
            return field;
        }

        public void setField(String field) {
            this.field = field;
        }

        public List<RexLiteral> getLiterals() {
            return literals;
        }

        public RexNode toRexField() {
            if (null != field) {
                return relBuilder.field(field);
            }

            throw new RuntimeException("build rexField error, field is not init");
        }

        public RexNode toFirstRexLiteral() {
            if (null == literals || literals.isEmpty()) {
                throw new RuntimeException("build rexField error, literals is not init");
            }

            return literals.get(0);
        }

        public RexNode[] toRexLiterals() {
            if (null == literals || literals.isEmpty()) {
                throw new RuntimeException("build rexField error, literals is not init");
            }

            RexNode[] rexNodes = new RexNode[literals.size()];
            for (int i = 0; i < literals.size(); i++) {
                rexNodes[i] = literals.get(i);
            }
            return rexNodes;
        }
    }


}
