package com.xforceplus.ultraman.oqsengine.plus.master.mysql;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import com.xforceplus.ultraman.metadata.constants.SystemField;
import com.xforceplus.ultraman.metadata.engine.EntityClassEngine;
import com.xforceplus.ultraman.metadata.engine.EntityClassGroup;
import com.xforceplus.ultraman.metadata.entity.EntityClassRef;
import com.xforceplus.ultraman.metadata.entity.IEntity;
import com.xforceplus.ultraman.metadata.entity.IEntityClass;
import com.xforceplus.ultraman.metadata.entity.IEntityField;
import com.xforceplus.ultraman.metadata.values.IValue;
import com.xforceplus.ultraman.oqsengine.plus.common.datasource.PackageInternal;
import com.xforceplus.ultraman.oqsengine.plus.common.iterator.DataIterator;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.MasterQueryEntity;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.MasterQueryResult;
import com.xforceplus.ultraman.oqsengine.plus.master.dto.MasterStorageEntity;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.executor.ConditionQueryExecutor;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.executor.CreateExecutor;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.executor.DeleteExecutor;
import com.xforceplus.ultraman.oqsengine.plus.master.mysql.executor.UpdateExecutor;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.Operation;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.QueryResult;
import com.xforceplus.ultraman.oqsengine.plus.meta.pojo.dto.table.SystemColumn;
import com.xforceplus.ultraman.oqsengine.plus.storage.pojo.dto.EntityId;
import com.xforceplus.ultraman.oqsengine.plus.storage.pojo.dto.EntityPackage;
import com.xforceplus.ultraman.oqsengine.plus.storage.pojo.dto.select.SelectConfig;
import com.xforceplus.ultraman.oqsengine.plus.storage.route.dynamic.DynamicDataSource;
import com.xforceplus.ultraman.sdk.core.event.EntityChanged;
import com.xforceplus.ultraman.sdk.core.event.EntityCreated;
import com.xforceplus.ultraman.sdk.core.utils.MasterStorageHelper;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.apache.calcite.adapter.jdbc.JdbcSchema;
import org.apache.calcite.config.Lex;
import org.apache.calcite.plan.Context;
import org.apache.calcite.schema.Schema;
import org.apache.calcite.schema.SchemaPlus;
import org.apache.calcite.sql.parser.SqlParser;
import org.apache.calcite.tools.FrameworkConfig;
import org.apache.calcite.tools.Frameworks;
import org.apache.calcite.tools.RelBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ConnectionCallback;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.TransactionTemplate;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.stream.Collectors;

import static com.xforceplus.ultraman.metadata.values.DateTimeValue.ZONE_ID;


/**
 * Created by justin.xu on 03/2023.
 *
 * @since 1.8
 */
@Slf4j
public class SQLMasterStorage implements MasterStorage {

    private PlatformTransactionManager manager;

    private TransactionTemplate template;

    private JdbcTemplate jdbcTemplate;

    private DynamicDataSource dynamicDataSource;

    private EntityClassEngine engine;

    private ObjectMapper objectMapper;

    private ExecutorService masterDispatcher;

    /**
     * TODO
     */
    @Autowired(required = false)
    public ApplicationEventPublisher publisher;

    //TODO maybe multi
    private Map<String, OQSFrameworkConfig> oqsFrameworkConfigMap = new HashMap<>();

    private Map<String, Schema> mapping = new ConcurrentHashMap<>();

    private OQSFrameworkConfig defaultConfig;

    private long queryTimeout;

    private long refreshTime;

    private long initTime;

    private ScheduledExecutorService scheduler;

    private static final class OQSFrameworkConfig {
        private SchemaPlus rootSchema;
        private JdbcSchema oqsSchema;
        private FrameworkConfig frameworkConfig;

        public OQSFrameworkConfig(DataSource dataSource) {
            rootSchema = Frameworks.createRootSchema(false);
            oqsSchema = JdbcSchema.create(rootSchema, "oqs", dataSource, null, null);
            rootSchema.add("oqs", oqsSchema);

            Frameworks.ConfigBuilder configBuilder = Frameworks.newConfigBuilder();
            this.frameworkConfig = configBuilder.defaultSchema(rootSchema)
                    .parserConfig(SqlParser.config().withLex(Lex.MYSQL).withCaseSensitive(false))
                    .context(new Context() {
                        @Override
                        public <C> C unwrap(Class<C> aClass) {
                            if (aClass == RelBuilder.Config.class) {
                                return (C) RelBuilder.Config.DEFAULT.withSimplify(false);
                            } else {
                                return null;
                            }
                        }
                    })
                    .build();

        }

        public JdbcSchema getOqsSchema() {
            return oqsSchema;
        }

        public SchemaPlus getRootSchema() {
            return rootSchema;
        }


        public FrameworkConfig getFrameworkConfig() {
            return frameworkConfig;
        }
    }

    public SQLMasterStorage(PlatformTransactionManager manager, DynamicDataSource dynamicDataSource, ObjectMapper objectMapper, ExecutorService masterDispatcher, EntityClassEngine engine, long queryTimeout, long refreshTime, long initTime) {
        this.manager = manager;
        this.dynamicDataSource = dynamicDataSource;
        this.objectMapper = objectMapper;
        this.masterDispatcher = masterDispatcher;
        this.queryTimeout = queryTimeout;
        this.engine = engine;
        this.scheduler = Executors.newScheduledThreadPool(1);
        this.refreshTime = refreshTime;
        this.initTime = initTime;
        this.template = new TransactionTemplate(manager);
        this.jdbcTemplate = new JdbcTemplate(dynamicDataSource);

        prepareSchema();
    }

    /**
     * get schema from database or datasource
     */
    private void prepareSchema() {
        PackageInternal packageInternal = dynamicDataSource.allMaster();
        packageInternal.getResourceMapping().forEach(
                (k, v) -> {
                    oqsFrameworkConfigMap.put((String) k, new OQSFrameworkConfig((DataSource) v));
                }
        );
        defaultConfig = new OQSFrameworkConfig((DataSource) dynamicDataSource.allMaster().getFirst());
    }

    private void checkId(IEntity entity) throws SQLException {
        if (entity.id() == 0) {
            throw new SQLException("Invalid entity`s id.");
        }
    }

    @Override
    public boolean build(IEntity entity, IEntityClass entityClass) throws SQLException {
        EntityClassGroup group = engine.describe(entityClass, entityClass.profile());
        return writeProcessing(entity, group, Operation.BUILD);
    }

    @Override
    public void build(EntityPackage entityPackage) throws SQLException {
        packageWriteProcessing(entityPackage, Operation.BUILD);
    }

    @Override
    public boolean replace(IEntity entity, IEntityClass entityClass) throws SQLException {
        EntityClassGroup group = engine.describe(entityClass, entityClass.profile());
        return writeProcessing(entity, group, Operation.REPLACE);
    }

    @Override
    public void replace(EntityPackage entityPackage) throws SQLException {
        packageWriteProcessing(entityPackage, Operation.REPLACE);
    }

    @Override
    public boolean delete(IEntity entity, IEntityClass entityClass) throws SQLException {
        EntityClassGroup group = engine.describe(entityClass, entityClass.profile());
        return writeProcessing(entity, group, Operation.DELETE);
    }

    @Override
    public void delete(EntityPackage entityPackage) throws SQLException {
        packageWriteProcessing(entityPackage, Operation.DELETE);
    }

    @Override
    public DataIterator<MasterStorageEntity> iterator(IEntityClass entityClass, long startTime, long endTime, long lastId, boolean useSelfEntityClass) throws SQLException {
        return null;
    }

    @Override
    public DataIterator<MasterStorageEntity> iterator(IEntityClass entityClass, long startTime, long endTime, long lastId, int size, boolean useSelfEntityClass) throws SQLException {
        return null;
    }

    @Override
    public Collection<QueryResult> selectEntities(SelectConfig selectConfig, IEntityClass entityClass) throws SQLException {

        OQSFrameworkConfig temp = oqsFrameworkConfigMap.get(dynamicDataSource.currentKey());
        final OQSFrameworkConfig oqsFrameworkConfig = null == temp ? defaultConfig : temp;

        selectConfig.setScheme(oqsFrameworkConfig.getOqsSchema());
        return template.execute(status -> {
            return jdbcTemplate.execute(new ConnectionCallback<Collection<QueryResult>>() {
                @Override
                public Collection<QueryResult> doInConnection(Connection conn) throws SQLException, DataAccessException {
                    MasterQueryResult result = null;
                    result = ConditionQueryExecutor.build(conn
                            , queryTimeout, engine.describe(entityClass, selectConfig.getProfile()), oqsFrameworkConfig.getFrameworkConfig()).execute(selectConfig);
                    Collection<QueryResult> entities = new ArrayList<>();
                    if (!result.getValues().isEmpty()) {
                        List<String> fieldsNames = result.getFieldsNames();
                        for (List<Object> value : result.getValues()) {
                            QueryResult queryResult = new QueryResult(null);
                            List<QueryResult.SelectItem> items = new ArrayList<>();
                            queryResult.setSelectItems(items);
                            for (int i = 0; i < fieldsNames.size(); i++) {
                                String field = fieldsNames.get(i);
                                items.add(new QueryResult.SelectItem(field, field, value.get(i)));
                            }
                            entities.add(queryResult);
                        }
                    }
                    return entities;
                }
            });
        });
    }

    private QueryResult toQueryResult(MasterQueryEntity masterQueryEntity, IEntityClass entityClass, Object o) {
        return null;
    }

    private boolean checkResult(boolean[] rs) {
        if (null == rs) {
            return false;
        }

        for (boolean r : rs) {
            if (!r) {
                return false;
            }
        }
        return true;
    }

//    private QueryResult toQueryResult(MasterQueryEntity masterQueryEntity, IEntityClass entityClass, SelectConfig selectConfig) {
//
//        long entityClassId = masterQueryEntity.getSystemColumn().getEntityClass();
//        String profile = masterQueryEntity.getSystemColumn().getProfile();
//        IEntityClass realEntityClass;
//        QueryResult queryResult = new QueryResult(masterQueryEntity.getSystemColumn());
//        if(entityClassId != 0) {
//            if (entityClassId == entityClass.id()) {
//                realEntityClass = entityClass;
//            } else {
//                realEntityClass = engine.load(Long.toString(entityClassId), profile).orElseThrow(() -> new RuntimeException("Not Related class"));
//            }
//
//
//            addSelectItem(queryResult, masterQueryEntity, realEntityClass);
//        } else {
//            addSelectItem(queryResult, masterQueryEntity);
//        }
//
//        return queryResult;
//    }


//    private void addSelectItem(QueryResult queryResult, MasterQueryEntity masterQueryEntity, IEntityClass entityClass) {
//        masterQueryEntity.getBusinessStaticFields().forEach(
//                s -> {
//                    queryResult.getSelectItems().add(s);
//                }
//        );
//
//
//        Map<String, Object> vs = Optional.ofNullable(masterQueryEntity.getBusinessDynamicFields())
//                .map(MasterStorageHelper::buildFromJson)
//                .orElse(Collections.emptyMap());
//
//        for (Map.Entry<String, Object> v : vs.entrySet()) {
//            Optional<IEntityField> rs = entityClass.field(v.getKey());
//            rs.ifPresent(entityField -> queryResult.getSelectItems()
//                    .add(new QueryResult.SelectItem(entityField.name(), entityField.name(), v.getValue())));
//        }
//    }

//    private void addSelectItem(QueryResult queryResult, MasterQueryEntity masterQueryEntity) {
//        masterQueryEntity.getBusinessStaticFields().forEach(
//                s -> {
//                    queryResult.getSelectItems().add(s);
//                }
//        );
//    }

    private boolean writeProcessing(IEntity entity, EntityClassGroup entityClass, Operation op) throws SQLException {
        checkId(entity);

        boolean result = template.execute(status -> {
            return jdbcTemplate.execute(new ConnectionCallback<Boolean>() {
                @Override
                public Boolean doInConnection(Connection con) throws SQLException, DataAccessException {
                    List<MasterStorageEntity> masterStorageEntities = toMasterStorageEntities(entityClass, entity, op);

                    boolean[] rs;
                    switch (op) {
                        case BUILD:
                            rs = CreateExecutor.build(con, queryTimeout).execute(masterStorageEntities);
                            break;
                        case REPLACE:
                            rs = UpdateExecutor.build(con, queryTimeout).execute(masterStorageEntities);
                            break;
                        case DELETE:
                            rs = DeleteExecutor.build(con, queryTimeout).execute(masterStorageEntities);
                            break;
                        default:
                            rs = null;
                    }


                    if(publisher != null) {
                        Instant instant = LocalDateTime.now().atZone(ZONE_ID).toInstant();
                        long currentTime = instant.toEpochMilli();

                        EntityClassRef entityClassRef = entity.entityClassRef();

                        publisher.publishEvent(new EntityChanged(Collections
                                .singletonList(Tuple.of(entityClassRef.getProfile(), entityClassRef.getId())), currentTime));
                    }

                    return checkResult(rs);

                }
            });
        });

        if (result) {
            entity.neat();
        }

        return result;
    }

    private void packageWriteProcessing(EntityPackage entityPackage, Operation op) throws SQLException {
        template.executeWithoutResult(status -> {
            jdbcTemplate.execute(new ConnectionCallback<Boolean>() {
                @Override
                public Boolean doInConnection(Connection con) throws SQLException, DataAccessException {
                    List<MasterStorageEntity> executionPackages = new ArrayList<>();
                    entityPackage.stream().forEach(er -> {
                        EntityClassGroup group = engine.describe(er.getValue(), er.getValue().profile());
                        executionPackages.addAll(toMasterStorageEntities(group, er.getKey(), op));
                    });

                    List<List<MasterStorageEntity>> executionPart = Lists.partition(executionPackages, EntityPackage.MAX_SIZE);

                    for (List<MasterStorageEntity> parts : executionPart) {
                        switch (op) {
                            case BUILD:
                                CreateExecutor.build(con, queryTimeout).execute(parts);
                                break;
                            case REPLACE:
                                UpdateExecutor.build(con, queryTimeout).execute(parts);
                                break;
                            case DELETE:
                                DeleteExecutor.build(con, queryTimeout).execute(parts);
                                break;
                            default:
                        }
                    }

                    if(publisher != null) {
                        //do publisher
                        List<Tuple2<String, Long>> list = entityPackage.stream().map(entry -> {
                            String profile = entry.getKey().entityClassRef().getProfile();
                            long entityClassId = entry.getKey().entityClassRef().getId();
                            return Tuple.of(profile, entityClassId);
                        }).collect(Collectors.toList());

                        Instant instant = LocalDateTime.now().atZone(ZONE_ID).toInstant();
                        long currentTime = instant.toEpochMilli();

                        publisher.publishEvent(new EntityChanged(list, currentTime));
                    }
                    return true;
                }
            });
        });
    }

    private List<MasterStorageEntity.TypedStorageValue> extractBocpSystemField(EntityClassGroup group, IEntity entity) {
        Collection<IEntityField> allFields = group.getAllFields();
        List<MasterStorageEntity.TypedStorageValue> values = new ArrayList<>();
        for (IEntityField entityField : allFields) {
            Optional<SystemField> systemField = Arrays.stream(SystemField.values()).filter(x -> x.getName().equalsIgnoreCase(entityField.name())).findFirst();
            if (systemField.isPresent()) {
                Optional<IValue> valueOp = entity.entityValue().getValue(entityField);
                if (valueOp.isPresent()) {
                    values.add(new MasterStorageEntity.TypedStorageValue(entityField, valueOp.get().storageValue()));
                }
            }
        }

        return values;
    }

    private MasterStorageEntity merge(MasterStorageEntity entity, List<MasterStorageEntity.TypedStorageValue> values) {
        MasterStorageEntity masterStorageEntity = new MasterStorageEntity();
        masterStorageEntity.setTableName(entity.getTableName());
        masterStorageEntity.setBusinessDynamicFields(entity.getBusinessDynamicFields());
        masterStorageEntity.setSystemColumn(entity.getSystemColumn());
        List<MasterStorageEntity.TypedStorageValue> mergedValues = new ArrayList<>(values);
        Optional.ofNullable(entity.getBusinessStaticFields())
                .orElseGet(Collections::emptyList).stream().filter(x -> {
                    return values.stream().noneMatch(v -> {
                        return v.getEntityField() == x.getEntityField();
                    });
                }).forEach(x -> {
                    mergedValues.add(x);
                });
        masterStorageEntity.setBusinessStaticFields(mergedValues);
        masterStorageEntity.setBusinessDynamicFieldsRemove(entity.getBusinessDynamicFieldsRemove());
        return masterStorageEntity;
    }

    private List<MasterStorageEntity> toMasterStorageEntities(EntityClassGroup group, IEntity entity, Operation op) {
        List<MasterStorageEntity> masterStorageEntities = new ArrayList<>();

        //extract system field and spread on every object
        List<MasterStorageEntity.TypedStorageValue> values = extractBocpSystemField(group, entity);


        //  加入自己.
        MasterStorageEntity masterStorageEntity = toMasterStorageEntity(group.getEntityClass(), entity, op);
        masterStorageEntity = merge(masterStorageEntity, values);
        masterStorageEntities.add(masterStorageEntity);

        //  加入父类.
        if (null != group.getFatherEntityClass() && !group.getFatherEntityClass().isEmpty()) {
            Collection<IEntityClass> entityClasses = group.getFatherEntityClass();

            for (IEntityClass e : entityClasses) {
                masterStorageEntities.add(merge(toMasterStorageEntity(e, entity, op), values));
            }
        }
        return masterStorageEntities;
    }

    private MasterStorageEntity toMasterStorageEntity(IEntityClass e, IEntity entity, Operation op) {
        Collection<IEntityField> selfFields = e.selfWithIndex();

        MasterStorageEntity masterStorageEntity = new MasterStorageEntity();
        //  设置系统字段
        SystemColumn systemColumn = new SystemColumn();
        systemColumn.setProfile(e.profile());
        systemColumn.setOperateTime(entity.time());
        systemColumn.setEntityClass(e.id());
        systemColumn.setId(entity.id());
        masterStorageEntity.setSystemColumn(systemColumn);
        //TODO
        masterStorageEntity.setTableName(e.masterWriteTable());

        List<MasterStorageEntity.TypedStorageValue> staticValues = new ArrayList<>();

        //  delete操作不处理任何的字段值.
        if (!op.equals(Operation.DELETE)) {
            List<IValue> dynamicValues = new ArrayList<>();
            for (IEntityField entityField : selfFields) {
                Optional<IValue> valueOp = entity.entityValue().getValue(entityField);
                if (valueOp.isPresent()) {

                    if (entityField.isDynamic()) {
                        dynamicValues.add(valueOp.get());
                    } else {
                        staticValues.add(new MasterStorageEntity.TypedStorageValue(entityField, valueOp.get().storageValue()));
                    }
                }
            }

            masterStorageEntity.setBusinessStaticFields(staticValues);

            //  设置动态.
            Map<String, Object> painValues = MasterStorageHelper.toPainValues(dynamicValues);

            if (op.equals(Operation.REPLACE)) {
                masterStorageEntity.setBusinessDynamicFields(MasterStorageHelper.buildReplace(painValues));
                masterStorageEntity.setBusinessDynamicFieldsRemove(MasterStorageHelper.buildRemove(painValues));
            } else {
                masterStorageEntity.setBusinessDynamicFields(MasterStorageHelper.toBuildJson(painValues));
            }

            systemColumn.setDeleted(false);
        } else {
            systemColumn.setDeleted(true);
        }

        return masterStorageEntity;
    }
}
