package com.xforceplus.ultraman.oqsengine.sdk.business.meta.service;

import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.sdk.business.meta.*;
import com.xforceplus.ultraman.oqsengine.sdk.exception.FieldValidationException;
import com.xforceplus.ultraman.oqsengine.sdk.facade.EntityFacade;
import com.xforceplus.ultraman.oqsengine.sdk.facade.result.*;
import com.xforceplus.ultraman.oqsengine.sdk.lock.LockConfig;
import com.xforceplus.ultraman.oqsengine.sdk.query.dsl.ExpRange;
import com.xforceplus.ultraman.oqsengine.sdk.query.dsl.ExpRel;
import com.xforceplus.ultraman.oqsengine.sdk.query.dsl.ExpSort;
import com.xforceplus.ultraman.oqsengine.sdk.vo.DataCollection;
import graphql.ExecutionResult;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Stream;

/**
 * business facade
 */
public interface BusinessFacade {

    /**
     * @param code
     * @param id
     * @param data
     * @return
     */
    Optional<EntityInstance> toEntityInstance(String code, long id, Map<String, Object> data);

    /**
     * is assignable from
     *
     * @param entityClassA
     * @param entityClassB
     * @return
     */
    boolean isAssignableFrom(IEntityClass entityClassA, IEntityClass entityClassB);

    //# tag::load[]

    EntityFacade entity();

    /**
     * 根据元数据对象Code加载元数据结构
     *
     * @param code
     * @return
     */
    IEntityClass load(String code);

    /**
     * 根据元数据对象Id加载元数据结构
     *
     * @param id
     * @return
     */
    IEntityClass load(Long id);
    //# end::load[]


    //# tag::create[]

    /**
     * 根据元数据对象结构创建一条数据实例，如果成功返回long值为唯一主键id，该方法无法识别返回状态码
     *
     * @param entityClass
     * @param value
     * @return
     */
    Long create(IEntityClass entityClass, Map<String, Object> value);

    /**
     * 根据元数据对象结构创建一条数据实例，如果成功返回long值为唯一主键id，同时可以传入状态码函数来对状态码进行判断
     *
     * @param entityClass
     * @param body
     * @param handler
     * @return
     */
    Long create(IEntityClass entityClass, Map<String, Object> body, Function<CreateOneResult, Long> handler);

    /**
     * 根据元数据对象结构创建多条数据实例，如果成功返回创建成功的数量，该方法暂时不会抛出事件, 该方法无法识别返回状态码
     *
     * @param entityClass
     * @param value
     * @return
     */
    Integer createMulti(IEntityClass entityClass, List<Map<String, Object>> value);

    /**
     * 根据元数据对象结构创建多条数据实例，如果成功返回创建成功的数量，该方法暂时不会抛出事件
     */
    Integer createMulti(IEntityClass entityClass, List<Map<String, Object>> body, Function<CreateMultiResult, Integer> handler);

    //# end::create[]

    /**
     * 根据传入的 EntityKey来进行保存，如果存在则且启用了ignoreIfExists则进行replace 如果不存在则进行通常的创建
     *
     * @param entityKey     包含了元数据结构以及一组key-value的值
     * @param value         创建数据
     * @param ignoreIfExist 如果启用则会用replace进行覆盖
     * @return
     */
    Long saveByUniqueKey(EntityKey entityKey, Map<String, Object> value, boolean ignoreIfExist);

    //# tag::update[]

    /**
     * 基于业务主键更新记录，（暂时不可用）
     *
     * @param entityKey
     * @param body
     * @return
     */
    Integer updateByKey(EntityKey entityKey, Map<String, Object> body);

    /**
     * 基于ID更新记录
     *
     * @param entityId entityClass + id
     * @param body
     * @return
     */
    Integer updateById(EntityId entityId, Map<String, Object> body);

    /**
     * 基于ID更新记录 可以操控返回码
     *
     * @param entityId
     * @param body
     * @param handler
     * @return
     */
    Integer updateById(EntityId entityId, Map<String, Object> body, Function<UpdateOneResult, Integer> handler);

    /**
     * 基于ID 更新多条记录
     *
     * @param entityClass
     * @param body
     * @return
     */
    Integer updateMulti(IEntityClass entityClass, List<Map<String, Object>> body);

    /**
     * 基于ID 更新多条记录 可以处理返回码
     *
     * @param entityClass
     * @param body
     * @return
     */
    Integer updateMulti(IEntityClass entityClass, List<Map<String, Object>> body, Function<UpdateMultiResult, Integer> handler);


    Integer updateByCondition(IEntityClass entityClass, Map<String, Object> body, ExpRel expRel);


    Integer updateByCondition(IEntityClass entityClass, Map<String, Object> body, ExpRel expRel, Function<UpdateResult, Integer> handler);


    Integer replaceByCondition(IEntityClass entityClass, Map<String, Object> body, ExpRel expRel);


    Integer replaceByCondition(IEntityClass entityClass, Map<String, Object> body, ExpRel expRel, Function<UpdateResult, Integer> handler);


    /**
     * 基于业务主键更新记录，（暂时不可用）
     *
     * @param entityKey
     * @param body
     * @return
     */
    Integer replaceByKey(EntityKey entityKey, Map<String, Object> body);

    /**
     * 基于ID 替换记录
     *
     * @param entityId
     * @param body
     * @return
     */
    Integer replaceById(EntityId entityId, Map<String, Object> body);

    /**
     * 基于ID 替换记录 可以操控返回码
     *
     * @param entityId
     * @param body
     * @param handler
     * @return
     */
    Integer replaceById(EntityId entityId, Map<String, Object> body, Function<UpdateOneResult, Integer> handler);

    /**
     * 基于ID 替换多条记录
     *
     * @param entityClass
     * @param body
     * @return
     */
    Integer replaceMulti(IEntityClass entityClass, List<Map<String, Object>> body);

    /**
     * 基于ID 替换多条记录 可以处理返回码
     *
     * @param entityClass
     * @param body
     * @return
     */
    Integer replaceMulti(IEntityClass entityClass, List<Map<String, Object>> body, Function<UpdateMultiResult, Integer> handler);

    //# end::update[]

    //# tag::delete[]

    /**
     * 基于业务主键删除,暂时不可用
     *
     * @param entityKey
     * @return
     */
    Integer deleteByKey(EntityKey entityKey);

    /**
     * 基于ID删除一条记录
     *
     * @param entityId
     * @return
     */
    Integer deleteOne(EntityId entityId);

    /**
     * 基于ID删除一条记录 可以处理返回码
     *
     * @param entityId
     * @return
     */
    Integer deleteOne(EntityId entityId, Function<DeleteOneResult, Integer> handler);

    /**
     * 基于ID 删除多条记录 返回affected row个数
     *
     * @param entityId
     * @return
     */
    Integer deleteMulti(List<EntityId> entityId);

    /**
     * 基于ID 删除多条记录 返回affected row个数 可以操控返回码
     *
     * @param entityId
     * @return
     */
    Integer deleteMulti(List<EntityId> entityId, Function<DeleteMultiResult, Integer> handler);

    /**
     * 根据查询结果 多次单挑删除
     *
     * @param entityClass
     * @param expRel
     * @return
     */
    Integer deleteByCondition(IEntityClass entityClass, ExpRel expRel);

    /**
     * 根据查询结果 批量删除
     *
     * @param entityClass
     * @param expRel
     * @return
     */
    Integer deleteByConditionByBatch(IEntityClass entityClass, ExpRel expRel);

    //# end::delete[]

    //# tag::query[]

    /**
     * 查询单条
     *
     * @param entityId
     * @return
     */
    Optional<EntityInstance> findOne(EntityId entityId);

    /**
     * 根据业务key 查询
     *
     * @param entityKey
     * @return
     */
    Optional<EntityInstance> findOneByKey(EntityKey entityKey);

    /**
     * 条件查询 返回结果时只返回投影中指定的字段
     *
     * @param entityClass
     * @param expRel
     * @param simplify
     * @return
     */
    DataCollection<EntityInstance> findByCondition(IEntityClass entityClass, ExpRel expRel, boolean simplify);

    /**
     * 条件查询 返回结果返回全部字段
     *
     * @param entityClass
     * @param expRel
     * @return
     */
    DataCollection<EntityInstance> findByCondition(IEntityClass entityClass, ExpRel expRel);

    /**
     * 查询所有数据，search after 实现可以突破数量限制，无排序
     *
     * @param entityClass
     * @param filter
     * @return
     */
    Iterator<EntityInstance> findAll(IEntityClass entityClass, ExpRel filter);

    /**
     * 查询所有数据，search after 实现可以突破数量限制，无排序 使用Stream返回
     *
     * @param entityClass
     * @param filter
     * @return
     */
    Stream<EntityInstance> findAllStream(IEntityClass entityClass, ExpRel filter);

    /**
     * 简单的翻页查询 依然有查询限制，但可以支持排序
     *
     * @param entityInstance
     * @param toManyRelationCode
     * @param page
     * @param sort
     * @return
     */
    DataCollection<EntityInstance> findAllByRelation(EntityInstance entityInstance, String toManyRelationCode, ExpRange page, ExpSort sort);

    /**
     * 查询关系所有数据，使用了 EntityFacade的迭代方法可以突破限制 无法支持排序
     *
     * @param entityInstance
     * @param toManyRelationCode
     * @param filter
     * @return
     */
    Iterator<EntityInstance> findAllByRelation(EntityInstance entityInstance,
                                               String toManyRelationCode,
                                               ExpRel filter);

    /**
     * 查询关系所有数据，使用了 EntityFacade的迭代方法可以突破限制 无法支持排序 返回Stream
     *
     * @param entityInstance
     * @param toManyRelationCode
     * @param filter
     * @return
     */
    Stream<EntityInstance> findAllByRelationStream(EntityInstance entityInstance,
                                                   String toManyRelationCode,
                                                   ExpRel filter);

    /**
     * 根据关系查询一条数据 主要是对一查询
     *
     * @param entityInstance
     * @param toOneRelationCode
     * @return
     */
    EntityInstance findOneByRelation(EntityInstance entityInstance, String toOneRelationCode);

    /**
     * 返回计数 无条件
     *
     * @param entityClass
     * @return
     */
    Long countAll(IEntityClass entityClass);

    /**
     * 查询某一个数据的关系数据计数
     *
     * @param entityInstance
     * @param toManyRelationCode
     * @return
     */
    Long countAllByRelation(EntityInstance entityInstance, String toManyRelationCode);

    /**
     * 返回计数 有条件
     *
     * @param entityClass
     * @param expRel
     * @return
     */
    Long count(IEntityClass entityClass, ExpRel expRel);
    //# end::query[]

    Optional<EntityInstance> replay(EntityIdVersion entityIdVersion, boolean onlySelf);

    List<ChangeVersion> changelogList(EntityId entityId, boolean onlySelf, int pageIndex, int pageSize);

    Map<Long, Long> changelogCount(long entityClassId, List<Long> ids);

    //# end::visible[]

    /**
     * @param id
     * @param consumer
     * @deprecated current is dangerous
     */
    @Deprecated
    void tryLock(List<Long> id, Consumer<List<Long>> consumer);

    /**
     * @param id
     * @param consumer
     * @deprecated current is dangerous
     */
    @Deprecated
    void tryLock(List<Long> id, Consumer<List<Long>> consumer, LockConfig config);

    /**
     * @deprecated current is dangerous
     */
    @Deprecated
    <T> T tryLockGet(List<Long> id, Function<List<Long>, T> mapper, LockConfig config);

    /**
     * @deprecated current is dangerous
     */
    @Deprecated
    <T> T tryLockGet(List<Long> id, Function<List<Long>, T> mapper);


    //# tag::gl-query[]

    /**
     * 支持直接输入 graphQL查询字符串
     *
     * @param queryString
     * @return
     */
    ExecutionResult query(String queryString);

    /**
     * 支持直接输入 graphQL查询字符串
     *
     * @param mutationString
     * @return
     */
    ExecutionResult mutation(String mutationString);
    //# end::gl-query[]

    /**
     * TODO with agg return
     *
     * @param entityClass
     * @param expRel
     * @param facet
     * @return
     */
    //List<EntityInstance> findByConditionWithAgg(EntityClass entityClass, ExpRel expRel, List<Aggregation> aggs);

    //search(String text, ExpRel expRel);

    //search(String text, ExpRel expRel, List<Aggregation> aggs);

//    /**
//     * create one entity
//     * @param instance
//     * @param toManyCode
//     * @param value
//     * @return
//     */
//    Long createByToManyRelation(EntityInstance instance, String toManyCode, Map<String, Object> value);
//
//    /**
//     * create one entity and update the raw entity
//     * @param instance
//     * @param toManyCode
//     * @param value
//     * @return
//     */
//    Long createByToOneRelation(EntityInstance instance, String toManyCode, Map<String, Object> value);

    void validate(IEntityClass entityClass, Map<String, Object> body) throws FieldValidationException;
}
