package com.xforceplus.ultraman.oqsengine.sdk.business.meta.service.impl;

import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.FieldLikeRelationType;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.impl.Relation;
import com.xforceplus.ultraman.oqsengine.pojo.reader.record.GeneralRecord;
import com.xforceplus.ultraman.oqsengine.pojo.reader.record.Record;
import com.xforceplus.ultraman.oqsengine.pojo.utils.PropertyHelper;
import com.xforceplus.ultraman.oqsengine.sdk.ChangelogCountResponse;
import com.xforceplus.ultraman.oqsengine.sdk.ChangelogResponse;
import com.xforceplus.ultraman.oqsengine.sdk.business.meta.*;
import com.xforceplus.ultraman.oqsengine.sdk.business.meta.service.BusinessFacade;
import com.xforceplus.ultraman.oqsengine.sdk.exception.FieldValidationException;
import com.xforceplus.ultraman.oqsengine.sdk.exception.OptimizeConflictException;
import com.xforceplus.ultraman.oqsengine.sdk.facade.EntityFacade;
import com.xforceplus.ultraman.oqsengine.sdk.facade.ProfileFetcher;
import com.xforceplus.ultraman.oqsengine.sdk.facade.QueryProvider;
import com.xforceplus.ultraman.oqsengine.sdk.facade.result.*;
import com.xforceplus.ultraman.oqsengine.sdk.graphql.config.GraphQLSchemaHolder;
import com.xforceplus.ultraman.oqsengine.sdk.graphql.gen.BatchDataLoader;
import com.xforceplus.ultraman.oqsengine.sdk.lock.LockConfig;
import com.xforceplus.ultraman.oqsengine.sdk.lock.RemoteMultiLock;
import com.xforceplus.ultraman.oqsengine.sdk.lock.synchronizer.RemoteMultiLockSynchronizer;
import com.xforceplus.ultraman.oqsengine.sdk.query.dsl.*;
import com.xforceplus.ultraman.oqsengine.sdk.service.core.ExecutionConfig;
import com.xforceplus.ultraman.oqsengine.sdk.store.engine.IEntityClassEngine;
import com.xforceplus.ultraman.oqsengine.sdk.store.engine.IEntityClassGroup;
import com.xforceplus.ultraman.oqsengine.sdk.tracing.TraceHelper;
import com.xforceplus.ultraman.oqsengine.sdk.transactional.OqsTransactionManager;
import com.xforceplus.ultraman.oqsengine.sdk.transactional.annotation.Propagation;
import com.xforceplus.ultraman.oqsengine.sdk.vo.DataCollection;
import graphql.ExecutionInput;
import graphql.ExecutionResult;
import graphql.GraphQL;
import graphql.execution.ExecutionId;
import io.opentracing.Span;
import io.opentracing.Tracer;
import io.sentry.ISpan;
import io.sentry.Sentry;
import io.vavr.control.Either;
import org.dataloader.DataLoader;
import org.dataloader.DataLoaderRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.ultraman.oqsengine.sdk.utils.IteratorUtils.toStream;

/**
 * business facade implementation
 */
public class BusinessFacadeImpl implements BusinessFacade {

    private Logger logger = LoggerFactory.getLogger(BusinessFacade.class);

    private final ContextService contextService;

    private final ProfileFetcher fetcher;

    private final EntityFacade entityFacade;

    private final IEntityClassEngine engine;

    private ExecutionConfig config;

    @Autowired(required = false)
    private OqsTransactionManager transactionManager;

    @Autowired(required = false)
    private RemoteMultiLockSynchronizer synchronizer;

    @Autowired(required = false)
    private Tracer tracer;

    @Autowired
    private GraphQLSchemaHolder schemaHolder;

    @Autowired
    private List<QueryProvider> queryProviders;

    public BusinessFacadeImpl(EntityFacade entityFacade, IEntityClassEngine engine, ProfileFetcher fetcher, ExecutionConfig config, @Autowired(required = false) ContextService contextService) {
        this.entityFacade = entityFacade;
        this.contextService = contextService;
        this.config = config;
        this.engine = engine;
        this.fetcher = fetcher;
    }

    public GraphQLSchemaHolder getSchemaHolder() {
        return schemaHolder;
    }

    public void setSchemaHolder(GraphQLSchemaHolder schemaHolder) {
        this.schemaHolder = schemaHolder;
    }

    public List<QueryProvider> getQueryProviders() {
        return queryProviders;
    }

    public void setQueryProviders(List<QueryProvider> queryProviders) {
        this.queryProviders = queryProviders;
    }

    public RemoteMultiLockSynchronizer getSynchronizer() {
        return synchronizer;
    }

    public void setSynchronizer(RemoteMultiLockSynchronizer synchronizer) {
        this.synchronizer = synchronizer;
    }

    /**
     * for test?
     *
     * @param entityFacade
     * @param engine
     * @param fetcher
     * @param config
     * @param contextService
     */
    public BusinessFacadeImpl(EntityFacade entityFacade, IEntityClassEngine engine, ProfileFetcher fetcher
            , ExecutionConfig config
            , @Autowired(required = false) ContextService contextService
            , RemoteMultiLockSynchronizer synchronizer
            , OqsTransactionManager transactionManager) {
        this.entityFacade = entityFacade;
        this.contextService = contextService;
        this.config = config;
        this.engine = engine;
        this.fetcher = fetcher;
        this.synchronizer = synchronizer;
        this.transactionManager = transactionManager;
    }

    private Map<String, Object> getContext() {
        return Optional.ofNullable(contextService).map(ContextService::getAll)
                .orElseGet(Collections::emptyMap);
    }

    /**
     * TODO
     *
     * @param completionStage
     * @param <T>
     * @return
     */
    private <T> T get(CompletionStage<T> completionStage) {
        try {
            return completionStage
                    .toCompletableFuture()
                    .join();
        } catch (RuntimeException ex) {
            if (ex.getCause() != null) {
                throw (RuntimeException) ex.getCause();
            } else {
                throw ex;
            }
        }
    }

    /**
     * @param record value record
     * @param group  main class structure
     * @return
     */
    private EntityInstance toEntityInstance(Record record, IEntityClassGroup group) {
        Long typeId = record.getTypeId();

        if (typeId != null && typeId > 0) {
            //a typed record
            if (typeId.equals(group.getEntityClass().id())) {
                //is the current entityClass
                return new GeneralEntityInstance(record, group.getEntityClass(), this);
            } else if (!group.getFatherEntityClass().isEmpty() && group.getFatherEntityClass().stream().anyMatch(parent -> typeId.equals(parent.id()))) {

                Optional<IEntityClass> firstParent = group.getFatherEntityClass().stream().filter(parent -> typeId.equals(parent.id())).findFirst();

                //parent entityClass
                return new GeneralEntityInstance(record, firstParent.get(), this);
            } else if (!group.getChildrenEntityClass().isEmpty() && group.getChildrenEntityClass().stream().anyMatch(child -> typeId.equals(child.id()))) {
                Optional<IEntityClass> firstChild = group.getChildrenEntityClass().stream().filter(child -> typeId.equals(child.id())).findFirst();
                return new GeneralEntityInstance(record, firstChild.get(), this);
            }
        }

        return new ValueBasedEntityInstance(record);
    }

    /**
     * utils for create EntityInstance
     *
     * @return
     */
    @Override
    public Optional<EntityInstance> toEntityInstance(String code, long id, Map<String, Object> data) {

        IEntityClass targetEntityClass = load(code);

        if (targetEntityClass != null) {
            GeneralRecord generalRecord = new GeneralRecord(targetEntityClass.fields(), 0);
            generalRecord.fromMap(data);
            generalRecord.setId(id);
            return Optional.of(new GeneralEntityInstance(generalRecord, targetEntityClass, this));
        }

        return Optional.empty();
    }

    @Override
    public boolean isAssignableFrom(IEntityClass entityClassA, IEntityClass entityClassB) {
        IEntityClass ptr = entityClassB;
        if (entityClassB.id() == entityClassA.id()) {
            return true;
        }

        /**
         * has parent
         */
        if (entityClassB.extendEntityClass() != null) {
            if (entityClassB.extendEntityClass().id() == entityClassA.id()) {
                return true;
            } else {
                //
                IEntityClassGroup group = engine.describe(entityClassB, fetcher.getProfile(Collections.emptyMap()));
                return group.getFatherEntityClass().stream().anyMatch(x -> x.id() == entityClassA.id());
            }
        }

        return false;
    }

    @Override
    public EntityFacade entity() {
        return entityFacade;
    }

    @Override
    public IEntityClass load(String code) {
        return entityFacade.loadByCode(code, fetcher.getProfile(Collections.emptyMap()))
                .orElseThrow(() -> new RuntimeException("entityClass with code [" + code + "] not found"));
    }

    @Override
    public IEntityClass load(Long id) {
        return entityFacade.load(id.toString(), fetcher.getProfile(Collections.emptyMap()))
                .orElseThrow(() -> new RuntimeException("entityClass with id [" + id + "] not found"));
    }

    @Override
    public Long create(IEntityClass entityClass, Map<String, Object> body) {
        return this.create(entityClass, body, x -> {
            if (x.getOriginCause() == ResultStatus.OriginStatus.HALF_SUCCESS) {
                return x.getId();
            } else {
                throw new RuntimeException(x.getMessage());
            }
        });
    }

    @Override
    public Long create(IEntityClass entityClass, Map<String, Object> body, Function<CreateOneResult, Long> handler) {
        return get(entityFacade.create(entityClass, body, getContext()))
                .getOrElseGet(x -> handler.apply(x));
    }

    @Override
    public Integer createMulti(IEntityClass entityClass, List<Map<String, Object>> body) {
        return this.createMulti(entityClass, body, x -> {
            if (x.getOriginCause() == ResultStatus.OriginStatus.HALF_SUCCESS) {
                return x.getInsertedRows();
            } else {
                throw new RuntimeException(x.getMessage());
            }
        });
    }

    @Override
    public Integer createMulti(IEntityClass entityClass, List<Map<String, Object>> body, Function<CreateMultiResult, Integer> handler) {
        return get(entityFacade.createMulti(entityClass, body, getContext()))
                .getOrElseGet(x -> handler.apply(x));
    }

    /**
     * will retrieve the record before doing
     *
     * @param entityKey
     * @param value
     * @param ignoreIfExists if ignore will return the former id or will do the replace;
     * @return
     */
    @Override
    public Long saveByUniqueKey(EntityKey entityKey, Map<String, Object> value, boolean ignoreIfExists) {

        //TODO check

        Optional<EntityInstance> oneByKey = this.findOneByKey(entityKey);

        if (oneByKey.isPresent()) {
            if (!ignoreIfExists) {
                this.replaceById(oneByKey.get().id(), value);
            }
            return oneByKey.get().id().getId();
        } else {
            return this.create(entityKey.getSchema(), value);
        }
    }

    @Override
    public Integer updateByKey(EntityKey entityKey, Map<String, Object> body) {
        //turn entityKey to condition
        Optional<EntityInstance> oneByKey = findOneByKey(entityKey);
        return oneByKey.map(x ->
                updateById(x.id(), body)
        ).orElse(0);
    }

    @Override
    public Integer replaceByKey(EntityKey entityKey, Map<String, Object> body) {
        //turn entityKey to condition
        Optional<EntityInstance> oneByKey = findOneByKey(entityKey);
        return oneByKey.map(x -> replaceById(x.id(), body)).orElse(0);
    }

    @Override
    public Integer updateById(EntityId entityId, Map<String, Object> body) {
        logger.info("update {} by id {}", entityId.getiEntityClass().code(), entityId.getId());
        return this.updateById(entityId, body, x -> {
            if (x.getOriginCause() == ResultStatus.OriginStatus.HALF_SUCCESS) {
                return 1;
            } else if (x.getOriginCause() == ResultStatus.OriginStatus.NOT_FOUND) {
                return 0;
            } else if (x.getOriginCause() == ResultStatus.OriginStatus.CONFLICT) {
                throw new OptimizeConflictException();
            } else {
                throw new RuntimeException(x.getMessage());
            }
        });
    }

    @Override
    public Integer updateMulti(IEntityClass entityClass, List<Map<String, Object>> body) {
        return this.updateMulti(entityClass, body, x -> {
            if (x.getOriginCause() == ResultStatus.OriginStatus.HALF_SUCCESS) {
                return x.getUpdatedRows();
            } else {
                throw new RuntimeException(x.getMessage());
            }
        });
    }

    @Override
    public Integer updateMulti(IEntityClass entityClass, List<Map<String, Object>> body, Function<UpdateMultiResult, Integer> handler) {
        return get(entityFacade.updateMulti(entityClass, body, getContext()))
                .getOrElseGet(x -> handler.apply(x));
    }

    @Override
    public Integer updateByCondition(IEntityClass entityClass, Map<String, Object> body, ExpRel expRel) {
        return this.updateByCondition(entityClass, body, expRel, x -> {
            if (x.getOriginCause() == ResultStatus.OriginStatus.HALF_SUCCESS) {
                return 0;
            } else {
                throw new RuntimeException(x.getMessage());
            }
        });
    }

    @Override
    public Integer updateByCondition(IEntityClass entityClass, Map<String, Object> body, ExpRel expRel, Function<UpdateResult, Integer> handler) {
        return get(entityFacade.updateByCondition(entityClass, expRel, body, getContext()))
                .getOrElseGet(x -> handler.apply(x));
    }

    @Override
    public Integer replaceByCondition(IEntityClass entityClass, Map<String, Object> body, ExpRel expRel) {
        return this.replaceByCondition(entityClass, body, expRel, x -> {
            if (x.getOriginCause() == ResultStatus.OriginStatus.HALF_SUCCESS) {
                return 0;
            } else {
                throw new RuntimeException(x.getMessage());
            }
        });
    }

    @Override
    public Integer replaceByCondition(IEntityClass entityClass, Map<String, Object> body, ExpRel expRel, Function<UpdateResult, Integer> handler) {
        return get(entityFacade.updateByCondition(entityClass, expRel, body, getContext()))
                .getOrElseGet(x -> handler.apply(x));
    }

    @Override
    public Integer replaceMulti(IEntityClass entityClass, List<Map<String, Object>> body) {
        return this.replaceMulti(entityClass, body, x -> {
            if (x.getOriginCause() == ResultStatus.OriginStatus.HALF_SUCCESS) {
                return x.getUpdatedRows();
            } else {
                throw new RuntimeException(x.getMessage());
            }
        });
    }

    @Override
    public Integer replaceMulti(IEntityClass entityClass, List<Map<String, Object>> body, Function<UpdateMultiResult, Integer> handler) {
        return get(entityFacade.replaceMulti(entityClass, body, getContext()))
                .getOrElseGet(x -> handler.apply(x));
    }

    @Override
    public Integer updateById(EntityId entityId, Map<String, Object> body, Function<UpdateOneResult, Integer> handler) {
        logger.info("update {} by id {}", entityId.getiEntityClass().code(), entityId.getId());
        return get(entityFacade.updateById(entityId.getiEntityClass(), entityId.getId(), body, getContext()))
                .getOrElseGet(x -> handler.apply(x));
    }

    @Override
    public Integer replaceById(EntityId entityId, Map<String, Object> body) {
        return this.replaceById(entityId, body, x -> {
            if (x.getOriginCause() == ResultStatus.OriginStatus.HALF_SUCCESS) {
                return 1;
            } else if (x.getOriginCause() == ResultStatus.OriginStatus.NOT_FOUND) {
                return 0;
            } else if (x.getOriginCause() == ResultStatus.OriginStatus.CONFLICT) {
                throw new OptimizeConflictException();
            } else {
                throw new RuntimeException(x.getMessage());
            }
        });
    }

    @Override
    public Integer replaceById(EntityId entityId, Map<String, Object> body, Function<UpdateOneResult, Integer> handler) {
        return get(entityFacade.replaceById(entityId.getiEntityClass(), entityId.getId(), body, getContext()))
                .getOrElseGet(x -> handler.apply(x));
    }

    @Override
    public Integer deleteOne(EntityId entityId) {
        return this.deleteOne(entityId, x -> {
            if (x.getOriginCause() == ResultStatus.OriginStatus.CONFLICT) {
                throw new OptimizeConflictException();
            } else if (x.getOriginCause() == ResultStatus.OriginStatus.NOT_FOUND) {
                return 0;
            } else {
                throw new RuntimeException(x.getMessage());
            }
        });
    }

    @Override
    public Integer deleteOne(EntityId entityId, Function<DeleteOneResult, Integer> handler) {
        return get(entityFacade.deleteOne(entityId.getiEntityClass(), entityId.getId(), getContext()))
                .getOrElseGet(x -> handler.apply(x));
    }

    /**
     * will not throw any exception
     *
     * @param entityId
     * @return
     */
    @Override
    public Integer deleteMulti(List<EntityId> entityId) {
        return this.deleteMulti(entityId, x -> {
            return x.getAffectedRows();
        });
    }

    @Override
    public Integer deleteMulti(List<EntityId> entityId, Function<DeleteMultiResult, Integer> handler) {
        if (entityId.isEmpty()) {
            return 0;
        } else {
            EntityId sample = entityId.get(0);
            List<Long> ids = entityId.stream().map(x -> x.getId()).collect(Collectors.toList());
            return get(entityFacade.deleteMulti(sample.getiEntityClass(), ids, getContext()))
                    .getOrElseGet(x -> handler.apply(x));
        }
    }

    @Override
    public Integer deleteByCondition(IEntityClass entityClass, ExpRel expRel) {

        AtomicInteger counter = new AtomicInteger(0);
        AtomicInteger failed = new AtomicInteger(0);
        this.findByCondition(entityClass, expRel).getRows().forEach(x -> {
            try {
                Integer retValue = deleteOne(x.id());
                counter.addAndGet(retValue);
            } catch (Exception ex) {
                logger.error("{}", ex);
                failed.getAndIncrement();
            }
        });

        return counter.get();
    }

    @Override
    public Integer deleteByConditionByBatch(IEntityClass entityClass, ExpRel expRel) {
        List<EntityId> entityIds = this.findByCondition(entityClass, expRel)
                .getRows()
                .stream()
                .map(x -> x.id()).collect(Collectors.toList());

        return this.deleteMulti(entityIds);
    }

    @Override
    public Integer deleteByKey(EntityKey entityKey) {
        return this.deleteOne(findOneByKey(entityKey).get().id());
    }

    @Override
    public Optional<EntityInstance> findOne(EntityId entityId) {

        IEntityClassGroup group = engine.describe(entityId.getiEntityClass(), fetcher.getProfile(getContext()));
        return get(entityFacade.findOneById(entityId.getiEntityClass(), entityId.getId(), getContext()))
                .map(x -> toEntityInstance(x, group))
                .peekLeft(x -> logger.error("find one got error {}", x))
                .toJavaOptional();
    }

    @Override
    public Optional<EntityInstance> findOneByKey(EntityKey entityKey) {

        Map<String, Object> businessKey = entityKey.getBusinessKey();

        ExpQuery expQuery = ExpFactory.createFrom(businessKey);

        ExpContext expContext = new ExpContext().setSchema(entityFacade.getReader(entityKey.getSchema(), contextService.getAll())).withContext(getContext());
        /**
         * append range to avoid range constraints
         */
        Either<QueryResult, DataCollection<Record>> dataCollections = get(entityFacade.query(expContext, ExpFactory.withRange(expQuery, new ExpRange(1, 10))));

        IEntityClassGroup group = engine.describe(entityKey.getSchema(), fetcher.getProfile(getContext()));

        Optional<EntityInstance> entityInstance = dataCollections.map(x -> {
            Integer rowNum = x.getRowNum();
            if (rowNum > 1) {
                logger.warn("expected one but return multi on {} with Map[{}]", entityKey.getSchema().code(), businessKey);
            }

            if (rowNum > 0) {
                Record record = x.getRows().get(0);

                return toEntityInstance(record, group);
            }

            return null;
        }).toJavaOptional();

        return entityInstance;
    }

    private DataCollection<EntityInstance> toEntityInstances(Either<QueryResult, DataCollection<Record>> dataCollections, IEntityClass entityClass) {

        IEntityClassGroup group = engine.describe(entityClass, fetcher.getProfile(getContext()));
        List<EntityInstance> result = dataCollections.map(x ->
                        x.getRows().stream()
                                .map(record -> toEntityInstance(record, group))
                                .collect(Collectors.toList()))
                .getOrElseThrow(x -> new RuntimeException(x.getMessage()));

        return new DataCollection<>(dataCollections.get().getRowNum(), result);
    }

    private Long toCount(Either<QueryResult, DataCollection<Record>> dataCollections) {
        return dataCollections.map(x -> new Long(x.getRowNum())).getOrElseThrow(x -> new RuntimeException(x.getMessage()));
    }

    @Override
    public DataCollection<EntityInstance> findByCondition(IEntityClass entityClass, ExpRel expRel, boolean simplify) {
        Map<String, Object> contextMap = new HashMap<>();
        contextMap.putAll(getContext());
        contextMap.put(EntityFacade.SIMPLIFY, simplify);

        Either<QueryResult, DataCollection<Record>> dataCollections = get(entityFacade.query(entityClass, expRel, contextMap));
        return toEntityInstances(dataCollections, entityClass);
    }

    @Override
    public DataCollection<EntityInstance> findByCondition(IEntityClass entityClass, ExpRel expRel) {
        ISpan span = Sentry.getSpan();
        if (span == null) {
            span = Sentry.startTransaction("query", "selectByConditions");
        }

        Either<QueryResult, DataCollection<Record>> dataCollections;
        try {
            dataCollections = get(entityFacade.query(entityClass, expRel, getContext()));
        } finally {
            span.finish();
        }

        return toEntityInstances(dataCollections, entityClass);
    }

    @Override
    public Long countAll(IEntityClass entityClass) {

        Either<QueryResult, DataCollection<Record>> dataCollections = get(entityFacade.query(entityClass, ExpFactory.EMPTY
                , getContext()));
        return toCount(dataCollections);
    }

    /**
     * @param entityInstance
     * @param toManyRelationCode
     * @param filter             sort will be removed
     * @return
     */
    @Override
    public Iterator<EntityInstance> findAllByRelation(EntityInstance entityInstance,
                                                      String toManyRelationCode,
                                                      ExpRel filter) {
        //check if entityInstance has such relation
        IEntityClass type = entityInstance.type();

        if (type != null) {
            EntityId id = entityInstance.id();
            Optional<Relation> related = type.relations().stream()
                    .filter(x -> x.getRelationType().equalsIgnoreCase(FieldLikeRelationType.ONE2MANY.getName())
                            && x.getName().equalsIgnoreCase(toManyRelationCode)).findAny();

            if (related.isPresent()) {
                //generate condition
                Optional<IEntityClass> relatedEntityClassOp = entityFacade
                        .getReader(type, contextService.getAll()).relatedEntityClass(related.get().getName());

                IEntityClass relatedEntityClass;


                if (relatedEntityClassOp.isPresent()) {
                    relatedEntityClass = relatedEntityClassOp.get();
                } else {
                    return Collections.emptyIterator();
                }

                IEntityClassGroup group = engine.describe(relatedEntityClass, fetcher.getProfile(getContext()));

                ExpQuery newQuery = new ExpQuery();

                ExpField expId = ExpField.field("id");
                newQuery.project(filter.getProjects())
                        .filters(filter.getFilters())
                        .filters(ExpCondition.call(ExpOperator.EQUALS
                                , ExpField.field(toManyRelationCode + ".id")
                                , ExpValue.from(id.getId())))
                        .filters(ExpCondition.call("$id", ExpOperator.GREATER_THAN, expId, ExpValue.from(0)))
                        .sort(ExpSort.init().withSort("id", "asc"))
                        .range(1, 1000);

                /**
                 *  IEntityClass entityClass
                 *  , ExpRel initRel
                 *  , BiFunction<ExpRel, Record, ExpRel> conditionTransformer
                 *  , Function<Record, T> transformer
                 *  , Map<String, Object> context
                 */
                return entityFacade.queryIterate(relatedEntityClass, newQuery, (query, record) -> {
                    if (record != null) {
                        ((ExpQuery) query).mutateCondition(cond -> {
                            if ("$id".equals(cond.getMark())) {
                                /**
                                 * TODO
                                 */
                                List<ExpNode> expNodes = cond.getExpNodes();
                                expNodes.clear();
                                expNodes.add(expId);
                                expNodes.add(ExpValue.fromSingle(record.getId()));
                            }
                        });
                    }
                    return query;
                }, x -> toEntityInstance(x, group), getContext()).iterator();
            }
        }

        return Collections.emptyIterator();
    }

    @Override
    public Stream<EntityInstance> findAllByRelationStream(EntityInstance entityInstance, String toManyRelationCode, ExpRel filter) {
        return toStream(findAllByRelation(entityInstance, toManyRelationCode, filter));
    }

    @Override
    public Iterator<EntityInstance> findAll(IEntityClass entityClass, ExpRel filter) {

        ExpQuery newQuery = new ExpQuery();

        ExpField expId = ExpField.field("id");
        newQuery.project(filter.getProjects())
                .filters(filter.getFilters())
                .filters(ExpCondition.call("$id", ExpOperator.GREATER_THAN, expId, ExpValue.from(0)))
                .sort(ExpSort.init().withSort("id", "asc"))
                .range(1, 5000);

        IEntityClassGroup group = engine.describe(entityClass, fetcher.getProfile(getContext()));

        return entityFacade.queryIterate(entityClass, newQuery, (query, record) -> {
            if (record != null) {
                ((ExpQuery) query).mutateCondition(cond -> {
                    if ("$id".equals(cond.getMark())) {
                        /**
                         * TODO
                         */
                        List<ExpNode> expNodes = cond.getExpNodes();
                        expNodes.clear();
                        expNodes.add(expId);
                        expNodes.add(ExpValue.fromSingle(record.getId()));
                    }
                });
            }
            return query;
        }, x -> toEntityInstance(x, group), getContext()).iterator();
    }

    @Override
    public Stream<EntityInstance> findAllStream(IEntityClass entityClass, ExpRel filter) {
        return toStream(findAll(entityClass, filter));
    }

    /**
     * may not used
     *
     * @param entityInstance
     * @param toManyRelationCode
     * @param page
     * @param sort
     * @return
     */
    @Deprecated
    @Override
    public DataCollection<EntityInstance> findAllByRelation(EntityInstance entityInstance, String toManyRelationCode, ExpRange page, ExpSort sort) {

        //check if entityInstance has such relation
        IEntityClass type = entityInstance.type();
        if (type != null) {
            EntityId id = entityInstance.id();
            Optional<Relation> related = type.relations().stream()
                    .filter(x -> x.getRelationType().equalsIgnoreCase(FieldLikeRelationType.ONE2MANY.getName())
                            && x.getName().equalsIgnoreCase(toManyRelationCode)).findAny();

            if (related.isPresent()) {
                //generate condition
                Optional<IEntityClass> relatedEntityClassOp = entityFacade.getReader(type, contextService.getAll()).relatedEntityClass(related.get().getName());

                IEntityClass relatedEntityClass;

                if (relatedEntityClassOp.isPresent()) {
                    relatedEntityClass = relatedEntityClassOp.get();
                } else {
                    throw new RuntimeException("Cannot find related entityClass");
                }

                ExpQuery expQuery = new ExpQuery();
                if (page != null) {
                    expQuery.setRange(page);
                }

                if (sort != null) {
                    expQuery.sort(sort);
                }

                expQuery.filters(ExpCondition.call(ExpOperator.EQUALS
                        , ExpField.field(toManyRelationCode + ".id")
                        , ExpValue.from(id.getId())));

                Either<QueryResult, DataCollection<Record>> dataCollections = get(entityFacade.query(relatedEntityClass
                        , ExpFactory.withRange(expQuery, new ExpRange(1, 1000))
                        , getContext()));

                return toEntityInstances(dataCollections, relatedEntityClass);
            }
        }

        return DataCollection.empty();
    }

    @Override
    public Long countAllByRelation(EntityInstance entityInstance, String toManyRelationCode) {
        IEntityClass type = entityInstance.type();
        if (type != null) {
            EntityId id = entityInstance.id();
            Optional<Relation> related = type.relations().stream()
                    .filter(x -> x.getRelationType().equalsIgnoreCase(FieldLikeRelationType.ONE2MANY.getName())
                            && x.getName().equalsIgnoreCase(toManyRelationCode)).findAny();

            if (related.isPresent()) {
                //generate condition
                Optional<IEntityClass> relatedEntityClassOp = entityFacade.getReader(type, contextService.getAll()).relatedEntityClass(related
                        .get().getName());

                IEntityClass relatedEntityClass;

                if (relatedEntityClassOp.isPresent()) {
                    relatedEntityClass = relatedEntityClassOp.get();
                } else {
                    throw new RuntimeException("Cannot find related entityClass");
                }

                ExpQuery expQuery = new ExpQuery();
                expQuery.filters(ExpCondition.call(ExpOperator.EQUALS, ExpField.field(PropertyHelper
                                .generateRelatedFieldName(toManyRelationCode, "id"))
                        , ExpValue.from(id.getId())));

                Either<QueryResult, DataCollection<Record>> dataCollections = get(entityFacade.query(relatedEntityClass
                        , expQuery, getContext()));

                return toCount(dataCollections);
            }
        }

        logger.warn("Entity Type is missing or no such relation {}", toManyRelationCode);

        return 0L;
    }

    @Override
    public EntityInstance findOneByRelation(EntityInstance entityInstance, String toOneRelationCode) {

        //check if entityInstance has such relation
        IEntityClass type = entityInstance.type();
        if (type != null) {
            Optional<Relation> related = type.relations().stream()
                    .filter(x -> (
                            x.getRelationType().equalsIgnoreCase(FieldLikeRelationType.MANY2ONE.getName()) ||
                                    x.getRelationType().equalsIgnoreCase(FieldLikeRelationType.ONE2ONE.getName()))
                            && x.getName().equalsIgnoreCase(toOneRelationCode)).findAny();

            if (related.isPresent()) {
                //generate condition
                Optional<IEntityClass> relatedEntityClassOp = entityFacade.getReader(type, contextService.getAll()).relatedEntityClass(related.get().getName());

                IEntityClass relatedEntityClass;

                if (relatedEntityClassOp.isPresent()) {
                    relatedEntityClass = relatedEntityClassOp.get();
                } else {
                    throw new RuntimeException("Cannot find related entityClass");
                }

                Optional<Object> value = entityInstance.getValue(toOneRelationCode + ".id");
                IEntityClassGroup relatedGroup = engine.describe(relatedEntityClass, fetcher.getProfile(getContext()));
                if (value.isPresent()) {
                    Object idObj = value.get();
                    Either<QueryOneResult, Record> recordE = get(entityFacade.findOneById(relatedEntityClass, Long.parseLong(idObj.toString()), getContext()));
                    return recordE.map(e -> toEntityInstance(e, relatedGroup)).getOrElseGet(x -> {
                        logger.error("Got error msg {}", x);
                        return null;
                    });
                }
            }
        }

        return null;
    }

    @Override
    public Long count(IEntityClass entityClass, ExpRel expRel) {
        return toCount(get(entityFacade.query(entityClass, expRel, getContext())));
    }

    @Override
    public Optional<EntityInstance> replay(EntityIdVersion entityIdVersion, boolean onlySelf) {
        long objId = entityIdVersion.getEntityId().getId();
        long version = entityIdVersion.getVersion();
        IEntityClass entityClass = entityIdVersion.getEntityId().getiEntityClass();
        IEntityClassGroup group = engine.describe(entityClass, fetcher.getProfile(getContext()));

        CompletionStage<Either<ResultStatus, Record>> replay = entityFacade.replay(objId, group, version, onlySelf);

        return get(replay)
                .map(x -> toEntityInstance(x, group))
                .peekLeft(x -> logger.error("replay got error {}", x))
                .toJavaOptional();
    }

    @Override
    public List<ChangeVersion> changelogList(EntityId entityId, boolean onlySelf, int pageNo, int pageSize) {

        long objId = entityId.getId();
        long entityclassId = entityId.getiEntityClass().id();

        CompletionStage<Either<ResultStatus, List<ChangelogResponse>>> changelogList = entityFacade.getChangelogList(objId, entityclassId, onlySelf, pageNo, pageSize);

        return get(changelogList).map(x -> x.stream().map(cp -> {
            ChangeVersion changeVersion = new ChangeVersion();
            changeVersion.setComment(cp.getComment());
            changeVersion.setId(cp.getId());
            changeVersion.setSource(cp.getSource());
            changeVersion.setTimestamp(cp.getTimestamp());
            changeVersion.setUsername(cp.getUsername());
            changeVersion.setVersion(cp.getVersion());
            return changeVersion;
        }).collect(Collectors.toList())).getOrElseGet(x -> Collections.emptyList());
    }

    @Override
    public Map<Long, Long> changelogCount(long entityClassId, List<Long> ids) {
        CompletionStage<Either<ResultStatus, ChangelogCountResponse>> changelogCount =
                entityFacade.getChangelogCount(entityClassId, ids);

        Map<Long, Long> map = new HashMap<>();
        get(changelogCount).forEach(x -> {
            x.getCountList().forEach(single -> {
                map.put(single.getObjId(), single.getCount());
            });
        });

        return map;
    }

    @Override
    public void tryLock(List<Long> ids, Consumer<List<Long>> consumer) {
        tryLock(ids, consumer, LockConfig.getDefault());
    }

    @Override
    public void tryLock(List<Long> ids, Consumer<List<Long>> consumer, LockConfig config) {
        if (config.getOpenTrans()) {
            try {
                transactionManager.transactionExecution(Propagation.REQUIRES_NEW
                        , -1, "lock multi"
                        , new Class[]{}, new Class[]{Exception.class}
                        , new Callable<Void>() {
                            @Override
                            public Void call() throws Exception {
                                RemoteMultiLock lock = new RemoteMultiLock(ids, synchronizer, config);
                                try {
                                    lock.lock();
                                    consumer.accept(ids);
                                } finally {
                                    lock.unlock();
                                }

                                return null;
                            }
                        });
            } catch (Throwable throwable) {
                throw new RuntimeException(throwable);
            }
        } else {
            RemoteMultiLock lock = new RemoteMultiLock(ids, synchronizer, config);
            try {
                lock.lock();
                consumer.accept(ids);
            } finally {
                lock.unlock();
            }
        }
    }

    @Override
    public <T> T tryLockGet(List<Long> ids, Function<List<Long>, T> mapper, LockConfig config) {
        if (config.getOpenTrans()) {
            try {
                transactionManager.transactionExecution(Propagation.REQUIRES_NEW
                        , -1, "lock multi"
                        , new Class[]{}, new Class[]{Exception.class}
                        , new Callable<T>() {
                            @Override
                            public T call() throws Exception {
                                RemoteMultiLock lock = new RemoteMultiLock(ids, synchronizer, config);
                                try {
                                    lock.lock();
                                    return mapper.apply(ids);
                                } finally {
                                    lock.unlock();
                                }
                            }
                        });
            } catch (Throwable throwable) {
                throw new RuntimeException(throwable);
            }
        } else {
            RemoteMultiLock lock = new RemoteMultiLock(ids, synchronizer, config);
            try {
                lock.lock();
                return mapper.apply(ids);
            } finally {
                lock.unlock();
            }
        }

        return null;
    }

    @Override
    public <T> T tryLockGet(List<Long> ids, Function<List<Long>, T> mapper) {
        return tryLockGet(ids, mapper, LockConfig.getDefault());
    }

    @Override
    public ExecutionResult query(String queryString) {
        GraphQL graphQL = schemaHolder.getGraphQL(fetcher.getProfile(contextService.getAll()));

        Optional<Span> currentSpan = TraceHelper.currentSpan(tracer);
        DataLoaderRegistry registry = new DataLoaderRegistry();
        //prepare
        engine.getRepository().codes().forEach(code -> {
            registry.computeIfAbsent(code, lazy -> DataLoader.newDataLoader(new BatchDataLoader(queryProviders)));
        });

        ExecutionInput executionInput = ExecutionInput.newExecutionInput()
                .query(queryString)
                .dataLoaderRegistry(registry)
                .variables(contextService.getAll())
                .executionId(ExecutionId.from(currentSpan.map(x -> x.toString()).orElse("")))
                .build();
        return graphQL.execute(executionInput);
    }

    @Override
    public ExecutionResult mutation(String mutation) {
        GraphQL graphQL = schemaHolder.getGraphQL(fetcher.getProfile(contextService.getAll()));

        Optional<Span> currentSpan = TraceHelper.currentSpan(tracer);

        ExecutionInput executionInput = ExecutionInput.newExecutionInput()
                .query(mutation)
                .variables(contextService.getAll())
                .executionId(ExecutionId.from(currentSpan.map(x -> x.toString()).orElse("")))
                .build();
        return graphQL.execute(executionInput);
    }

    @Override
    public void validate(IEntityClass entityClass, Map<String, Object> body) throws FieldValidationException {
        entityFacade.validate(entityClass, body);
    }

}