package com.xforceplus.ultraman.oqsengine.sdk.service.export.impl;

import akka.NotUsed;
import akka.stream.ActorMaterializer;
import akka.stream.javadsl.Flow;
import akka.stream.javadsl.Source;
import akka.util.ByteString;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.impl.ColumnField;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.*;
import io.vavr.Tuple2;
import org.apache.commons.text.StringEscapeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

/**
 * CSV
 * default
 * entity export Service
 */
public class CSVEntityExportServiceImpl extends AbstractEntityExportService {

    private Logger logger = LoggerFactory.getLogger(CSVEntityExportServiceImpl.class);

    public CSVEntityExportServiceImpl(
            List<ExportSource> exportSourceList
            , ExportSink exportSink
            , ExportCallBack callBack
            , ExportStringTransformer transformer
            , ActorMaterializer mat
    ) {
        super(exportSourceList, exportSink, transformer, callBack, mat);
    }

    @Override
    public boolean isAccept(String extension) {
        return !"xls".equals(extension);
    }

    @Override
    public boolean isSupportMultiSchema() {
        return false;
    }

    @Override
    public String generateFileType() {
        return "csv";
    }

    /**
     * addition bom source
     *
     * @return
     */
    private Source<ByteString, NotUsed> getCSVHeaderSource() {
        byte[] bom = new byte[]{(byte) 0xef, (byte) 0xbb, (byte) 0xbf};
        return Source.single(ByteString.fromArray(bom));
    }

    /**
     * custom logic for csv
     *
     * @param rawSource
     * @return
     */
    @Override
    protected Source<ByteString, ?> toByteStringSource(
              List<Tuple2<String,IEntityClass>> entityClasses
            , Source<ClassifiedRecord, NotUsed> rawSource
            , Map<String, ExportSchemaConfig> schemaMapping
            , boolean skipTransformer
            , Map<String, Object> context) {

        Flow<ClassifiedRecord, String, NotUsed> flow = getFlow(schemaMapping, context, skipTransformer);

        Source<ByteString, NotUsed> bomSource = getCSVHeaderSource();
        return bomSource.concat(rawSource
                .via(flow)
                .map(x -> ByteString.fromString(x, StandardCharsets.UTF_8)));
    }

    // \t is a tricky for csv see
    //     https://qastack.cn/superuser/318420/formatting-a-comma-delimited-csv-to-force-excel-to-interpret-value-as-a-string
    // using escape instead
    private String csvEscape(String strValue){
        return StringEscapeUtils.escapeCsv("\t" + strValue);
    }

    /**
     * get flow
     * @param schemaMapping
     * @param context
     * @return
     */
    private Flow<ClassifiedRecord, String, NotUsed> getFlow(
            Map<String, ExportSchemaConfig> schemaMapping
            , Map<String, Object> context
            , boolean skipTransformer) {

        AtomicBoolean isFirstLine = new AtomicBoolean(true);

        return Flow.<ClassifiedRecord>create().map(record -> {

            String classifyStr = record.getClassifyStr();

            ExportSchemaConfig schemaConfig = schemaMapping.get(classifyStr);

            Map<String, FormattedString> nameMapping = Optional.ofNullable(schemaConfig).map(ExportSchemaConfig::getNameMapping).orElseGet(Collections::emptyMap);
            List<String> orderedColumn = Optional.ofNullable(schemaConfig).map(ExportSchemaConfig::getOrderedColumn).orElseGet(Collections::emptyList);

            StringBuilder sb = new StringBuilder();
            if (isFirstLine.get()) {

                String header = record.getRecord()
                        .stream(orderedColumn)
                        .map(Tuple2::_1)
                        .map(x -> Optional.ofNullable(x)
                                .map(y -> {
                                    String fieldName = y.name();
                                    FormattedString name = nameMapping.get(fieldName);
                                    if (name == null) {
                                        return y.cnName();
                                    } else {
                                        return name.getText();
                                    }
                                })
                                .orElse(""))
                        .collect(Collectors.joining(","));

                //setup headers
                sb.append(header);
                sb.append("\n");
                isFirstLine.set(false);
            }

            //may have null convert to ""
            String line = record.getRecord()
                    .stream(orderedColumn)
                    .map(x -> {
                        ColumnField field = (ColumnField)x._1();
                        Object value = x._2();
                        String safeSourceValue = Optional.ofNullable(value).map(Object::toString).orElse("");
                        if(!skipTransformer) {
                            safeSourceValue = getStringValue(field.originEntityClass(), field, value, context, nameMapping);
                        }
                        return csvEscape(safeSourceValue);
                    })
                    .collect(Collectors.joining(","));
            sb.append(line);
            sb.append("\n");
            return sb.toString();
        });
    }
}