package com.xforceplus.ultraman.oqsengine.sdk.service.export.impl;

import akka.NotUsed;
import akka.stream.ActorMaterializer;
import akka.stream.IOResult;
import akka.stream.javadsl.Sink;
import akka.stream.javadsl.Source;
import akka.stream.javadsl.StreamConverters;
import akka.util.ByteString;
import builders.dsl.spreadsheet.builder.poi.PoiSpreadsheetBuilder;
import builders.dsl.spreadsheet.parser.data.json.JsonSpreadsheetParser;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.FieldConfig;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.FieldType;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityField;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.impl.ColumnField;
import com.xforceplus.ultraman.oqsengine.pojo.reader.record.Record;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.*;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.config.ExportExcelConfig;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.impl.excel.PoiCopySheet;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import lombok.SneakyThrows;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 *
 */
public class ExcelEntityExportServiceImpl extends AbstractEntityExportService {

    private ExportExcelConfig excelConfig;

    private final ActorMaterializer mat;

    private ExecutorService executorService;

    private Logger logger = LoggerFactory.getLogger(ExcelEntityExportServiceImpl.class);

    public ExcelEntityExportServiceImpl(
            List<ExportSource> exportSourceList
            , ExportSink exportSink
            , ExportCallBack callBack
            , ExportStringTransformer transformer
            , ExportExcelConfig excelConfig
            , ActorMaterializer mat
            , ExecutorService executorService
    ) {
        super(exportSourceList, exportSink, transformer, callBack, mat);
        this.mat = mat;
        this.excelConfig = excelConfig;
        this.executorService = executorService;
    }

    @Override
    public boolean isAccept(String extension) {
        return "xlsx".equals(extension) || "xls".equals(extension);
    }

    @Override
    public boolean isSupportMultiSchema() {
        return true;
    }

    @Override
    public String generateFileType() {
        return "xlsx";
    }

    @Override
    public Source<ByteString, ?> toByteStringSource
            (List<Tuple2<String, IEntityClass>> entityClasses,
             Source<ClassifiedRecord, NotUsed> rawSource
                    , Map<String, ExportSchemaConfig> schemaMapping,
             boolean isSkipTransformer,
             Map<String, Object> context) {

        Object template = context.get("template");

        InputStream inputStreamCompletionStage = rawSource
                .runWith(excelSink(entityClasses, schemaMapping
                                , isSkipTransformer, context
                                , Optional.ofNullable(template).map(Object::toString).orElse(null))
                        , mat);
        Source<ByteString, CompletionStage<IOResult>> source = StreamConverters.fromInputStream(() -> inputStreamCompletionStage);

        return source;
    }

    /**
     * TODO if need init column for each query
     *
     * @param wb
     * @param entityClasses
     * @param schemaMapping
     * @param sheetMapping
     */
    private void prepareSheet(SXSSFWorkbook wb, List<Tuple2<String, IEntityClass>> entityClasses, Map<String, ExportSchemaConfig> schemaMapping, Map<String, Tuple2<Sheet, AtomicInteger>> sheetMapping) {
        entityClasses.forEach(entityClassTuple -> {
            String sheetCode = entityClassTuple._1();
            Tuple2<Sheet, AtomicInteger> sheetTuple = sheetMapping.get(sheetCode);

            ExportSchemaConfig schemaConfig = schemaMapping.get(sheetCode);
            Map<String, FormattedString> mapping = Optional.ofNullable(schemaConfig)
                    .map(ExportSchemaConfig::getNameMapping).orElseGet(Collections::emptyMap);

            String sheetName = Optional.ofNullable(mapping.get(sheetCode)).map(FormattedString::getText).orElse(sheetCode);

            if (sheetTuple == null) {
                Sheet sheet = wb.createSheet(sheetName);
                AtomicInteger linesInSheet = new AtomicInteger(1);
                sheetTuple = Tuple.of(sheet, linesInSheet);
                sheetMapping.put(sheetCode, sheetTuple);
                insertHeader(sheet, schemaConfig, entityClassTuple._2());
            }
        });
    }

    private void insertHeader(Sheet sheet, ExportSchemaConfig schemaConfig, IEntityClass entityClass) {

        Row row = sheet.createRow(0);

        List<String> orderedColumn = Optional.ofNullable(schemaConfig).map(x -> x.getOrderedColumn())
                .orElseGet(Collections::emptyList);
        if (orderedColumn.isEmpty()) {
            orderedColumn = entityClass.fields().stream().map(field -> field.name()).collect(Collectors.toList());
        }
        Map<String, FormattedString> nameMappings = Optional.ofNullable(schemaConfig).map(x -> x.getNameMapping())
                .orElseGet(Collections::emptyMap);
        orderedColumn = orderedColumn.stream().map((x) -> {
            return x.startsWith("_") ? x.substring(1) : x;
        }).collect(Collectors.toList());

        AtomicInteger cellIndex = new AtomicInteger(0);
        orderedColumn.forEach(x -> {
            Cell cell = row.createCell(cellIndex.getAndIncrement());
            String key = x;
            String columnName = Optional.ofNullable(nameMappings).
                    flatMap(np -> Optional.ofNullable(np.get(key)).map(FormattedString::getText)).orElse(x);
            cell.setCellValue(Optional.ofNullable(columnName).orElse(key));
        });
    }

    /**
     * from 2.1.3
     *
     * @param record
     * @param row
     * @param orderedColumn
     * @param context
     * @param isSkipTransformer
     */
    private void insertCell(Record record, Row row, List<String> orderedColumn, Map<String, Object> context, boolean isSkipTransformer, Map<Integer, CellStyle> columnNumberFormatStyles, Map<String, FormattedString> mapping) {
        AtomicInteger ai = new AtomicInteger(0);
        /**
         * Deal with the related
         */
        List<String> normalizeOrderedColumn = orderedColumn.stream().map(x -> {
            if (x.startsWith("_")) {
                return x.substring(1);
            } else {
                return x;
            }
        }).collect(Collectors.toList());

        //make every field with '.' as related field
        Stream<String> reNormalizeOrderedColumn = normalizeOrderedColumn.stream().map(x -> {
            if (x.contains(".")) {
                return "_".concat(x);
            } else {
                return x;
            }
        });

        if (normalizeOrderedColumn.isEmpty()) {
            //using origin function which will missing column when the related instance is not present
            record.stream(normalizeOrderedColumn).forEach(x -> {
                Cell cell = row.createCell(ai.getAndIncrement());
                ColumnField field = (ColumnField) x._1();
                IEntityClass entityClass = field.originEntityClass();
                if (x._2() != null) {
                    try {
                        String value = x._2().toString();
                        if (!isSkipTransformer) {
                            value = getStringValue(entityClass, field, x._2().toString(), context, mapping);
                        }

                        // 只针对decimal类型数据做数字转换
                        if (field.originField().type().equals(FieldType.DECIMAL)) {
                            setNumberFormatStyleAndCellValue(cell, columnNumberFormatStyles, field.originField().config(), new BigDecimal(value));
                        } else {
                            cell.setCellValue(value);
                        }
                    } catch (Exception ex) {
                        logger.error("{}", ex);
                        cell.setBlank();
                        cell.setCellErrorValue(FormulaError.VALUE.getCode());
                    }
                }

            });
        } else {
            List<Tuple2<IEntityField, Object>> columnList = record.stream(orderedColumn).collect(Collectors.toList());
            //always using orderColumn
            orderedColumn.forEach(x -> {
                Cell cell = row.createCell(ai.getAndIncrement());
                Optional<Tuple2<IEntityField, Object>> columnOp = columnList.stream().filter(col -> col._1.acceptName(x)).findFirst();
                if (columnOp.isPresent()) {
                    ColumnField field = (ColumnField) columnOp.get()._1();
                    IEntityClass entityClass = field.originEntityClass();
                    if (columnOp.get()._2() != null) {
                        try {
                            String value = columnOp.get()._2().toString();
                            if (!isSkipTransformer) {
                                value = getStringValue(entityClass, field, columnOp.get()._2().toString(), context, mapping);
                            }
                            // 只针对decimal类型数据做数字转换
                            if (field.originField().type().equals(FieldType.DECIMAL)) {
                                setNumberFormatStyleAndCellValue(cell, columnNumberFormatStyles, field.originField().config(), new BigDecimal(value));
                            } else {
                                cell.setCellValue(value);
                            }
                        } catch (Exception ex) {
                            logger.error("{}", ex);
                            cell.setBlank();
                            cell.setCellErrorValue(FormulaError.VALUE.getCode());
                        }
                    }
                }
            });
        }
    }

    /**
     * default workbook insert logic
     *
     * @param tuple
     * @param sheetMapping
     * @param schemaMapping
     * @param wb
     * @param context
     * @param isSkipTransformer
     * @param columnNumberFormatStyles
     */
    private void insertInDefaultWorkBook(ClassifiedRecord tuple
            , Map<String, Tuple2<Sheet, AtomicInteger>> sheetMapping
            , Map<String, ExportSchemaConfig> schemaMapping
            , Workbook wb, Map<String, Object> context
            , boolean isSkipTransformer
            , Map<Integer, CellStyle> columnNumberFormatStyles) {
        try {
            String sheetCode = tuple.getClassifyStr();
            Tuple2<Sheet, AtomicInteger> sheetTuple = sheetMapping.get(sheetCode);

            Record record = tuple.getRecord();

            ExportSchemaConfig schemaConfig = schemaMapping.get(sheetCode);
            Map<String, FormattedString> mapping = Optional.ofNullable(schemaConfig)
                    .map(ExportSchemaConfig::getNameMapping).orElseGet(Collections::emptyMap);

            List<String> orderedColumn = Optional.ofNullable(schemaConfig).map(x -> x.getOrderedColumn()).orElseGet(Collections::emptyList);

            String sheetName = Optional.ofNullable(mapping.get(sheetCode)).map(FormattedString::getText).orElse(sheetCode);

            if (sheetTuple == null) {
                Sheet sheet = wb.createSheet(sheetName);
                AtomicInteger linesInSheet = new AtomicInteger(0);
                sheetTuple = Tuple.of(sheet, linesInSheet);
                sheetMapping.put(sheetCode, sheetTuple);
            }

            Row row = sheetTuple._1.createRow(sheetTuple._2.getAndIncrement());
            insertCell(record, row, orderedColumn, context, isSkipTransformer, columnNumberFormatStyles, mapping);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

    /**
     * ClassifiedRecord will get a classified
     *
     * @param dynamicRow
     * @param record
     * @return
     */
    private boolean isTrigger(Row dynamicRow, ClassifiedRecord record) {
        Iterator<Cell> cellIterator = dynamicRow.cellIterator();
        while (cellIterator.hasNext()) {
            Cell next = cellIterator.next();
            if (next.getCellType() == CellType.STRING && (next.getStringCellValue().startsWith("$") || next.getStringCellValue().startsWith("@"))) {
                String dynamicVariable = next.getStringCellValue();
                //dynamic
                Tuple2<String, String> parsed = parseVariable(dynamicVariable);
                String classifyStr = record.getClassifyStr();
                if (parsed._1.equalsIgnoreCase(classifyStr)) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * ClassifiedRecord tuple
     * , Map<String, Tuple2<Sheet, AtomicInteger>> sheetMapping
     * , Map<String, ExportSchemaConfig> schemaMapping
     * , Workbook wb, Map<String, Object> context
     * , boolean isSkipTransformer
     * , Map<Integer, CellStyle> columnNumberFormatStyles
     * <p>
     * iterate all the sheet to render the
     */
    public void renderTemplate(Workbook wb, ClassifiedRecord record, Map<String, Object> context, boolean isSkipTransformer, Map<Integer, CellStyle> columnNumberFormatStyles, Map<String, ExportSchemaConfig> schemaMapping) {

        int numberOfSheets = wb.getNumberOfSheets();
        int i = 0;
        for (; i < numberOfSheets; i++) {

            Sheet sheetAt = wb.getSheetAt(i);
            ExportSchemaConfig schemaConfig = schemaMapping.get(sheetAt.getSheetName());
            Map<String, FormattedString> mapping = Optional.ofNullable(schemaConfig)
                    .map(ExportSchemaConfig::getNameMapping).orElseGet(Collections::emptyMap);
            String sheetIndexName = getSheetIndexName(sheetAt, i);
            Object rowTemplate = context.get(sheetIndexName);
            List<Tuple2<AtomicInteger, Row>> dynamicRows = (List<Tuple2<AtomicInteger, Row>>) rowTemplate;
            //TODO current only deal the first row
            if (!dynamicRows.isEmpty()) {
                Tuple2<AtomicInteger, Row> cursorMapping = dynamicRows.get(0);
                AtomicInteger nextIndex = cursorMapping._1();
                Row dynamicRow = cursorMapping._2;
                //check if should add the row
                //record -> row
                if (!isTrigger(dynamicRow, record)) {
                    continue;
                }

                Row row = sheetAt.createRow(nextIndex.getAndIncrement());

                //insert cell with celss
                Iterator<Cell> cellIterator = dynamicRow.cellIterator();
                int cellIndex = 0;
                while (cellIterator.hasNext()) {
                    Cell next = cellIterator.next();
                    Cell cell = row.createCell(cellIndex++);
                    if (next.getCellType() == CellType.STRING && (next.getStringCellValue().startsWith("$") || next.getStringCellValue().startsWith("@"))) {
                        String dynamicVariable = next.getStringCellValue();
                        //dynamic
                        insertSingleCell(dynamicVariable, record, cell, isSkipTransformer, columnNumberFormatStyles, mapping, context);
                    } else {
                        copyFromCell(next, cell);
                    }
                }
            }
        }
    }

    /**
     * @return
     */
    private Tuple2<String, String> parseVariable(String dynamicVariable) {
        if (dynamicVariable.startsWith("@")) {
            int varPos = dynamicVariable.indexOf(".");
            if (varPos != -1) {
                String classifier = dynamicVariable.substring(1, varPos);
                return Tuple.of(classifier, dynamicVariable.substring(varPos + 1));
            } else {
                return Tuple.of("", dynamicVariable);
            }
        }
        return Tuple.of("", dynamicVariable);
    }

    /**
     * entity is main
     * dynamicVariable
     * - $id
     * - @headId.$sss
     * - @headId.$_src.sss
     * - $_headId.sss
     *
     * @param dynamicVariable
     * @param record
     * @param cell
     * @param isSkipTransformer
     * @param columnNumberFormatStyles
     * @param mapping
     * @param context
     */
    private void insertSingleCell(String dynamicVariable, ClassifiedRecord record, Cell cell, boolean isSkipTransformer, Map<Integer, CellStyle> columnNumberFormatStyles, Map<String, FormattedString> mapping, Map<String, Object> context) {

        try {

            String classifyStr = record.getClassifyStr();
            Tuple2<String, String> parsedVariable = parseVariable(dynamicVariable);
            String checked = parsedVariable._1;

            if (!classifyStr.equals(checked)) {
                return;
            }

            String realDynamicVariable = parsedVariable._2;

            String[] parts = realDynamicVariable.split("\\.");
            Optional<ColumnField> fieldOp = Optional.empty();
            Optional<Object> valueOp = Optional.empty();

            if (parts.length == 1) {
                //self
                String fieldCode = realDynamicVariable.substring(1);
                List<Tuple2<IEntityField, Object>> target = record.getRecord().stream(Arrays.asList(fieldCode)).collect(Collectors.toList());
                if (!target.isEmpty()) {
                    Tuple2<IEntityField, Object> t = target.get(0);
                    fieldOp = Optional.of((ColumnField) t._1);
                    valueOp = Optional.of(t._2);
                }
            } else if (parts.length == 2) {
                //src
                if (realDynamicVariable.startsWith("$src.") && record.getSourceRecord() != null) {
                    String fieldCode = realDynamicVariable.substring("$src.".length());
                    List<Tuple2<IEntityField, Object>> target = record.getSourceRecord().stream(Arrays.asList(fieldCode)).collect(Collectors.toList());
                    if (!target.isEmpty()) {
                        Tuple2<IEntityField, Object> t = target.get(0);
                        fieldOp = Optional.of((ColumnField) t._1);
                        valueOp = Optional.of(t._2);
                    }
                } else {
                    String fieldCode = realDynamicVariable.substring(1);
                    List<Tuple2<IEntityField, Object>> target = record.getRecord().stream(Arrays.asList(fieldCode)).collect(Collectors.toList());
                    if (!target.isEmpty()) {
                        Tuple2<IEntityField, Object> t = target.get(0);
                        fieldOp = Optional.of((ColumnField) t._1);
                        valueOp = Optional.of(t._2);
                    }
                }
            }

            if (fieldOp.isPresent() && valueOp.isPresent()) {
                ColumnField columnField = fieldOp.get();
                Object rawValue = valueOp.get();
                String value = rawValue.toString();
                if (!isSkipTransformer) {
                    value = getStringValue(columnField.originEntityClass(), columnField.getOriginObject(), valueOp.get(), context, mapping);
                }

                // 只针对decimal类型数据做数字转换
                if (columnField.getOriginObject().type().equals(FieldType.DECIMAL)) {
                    setNumberFormatStyleAndCellValue(cell, columnNumberFormatStyles, columnField.getOriginObject().config(), new BigDecimal(rawValue.toString()));
                } else {
                    cell.setCellValue(value);
                }
            } else {
                cell.setCellValue(dynamicVariable);
            }
        } catch (Exception ex) {
            logger.error("{}", ex);
            cell.setBlank();
            cell.setCellErrorValue(FormulaError.VALUE.getCode());
        }
    }

    private void copyFromCell(Cell srcCell, Cell dstCell) {
        switch (srcCell.getCellType()) {
            case STRING:
                dstCell.setCellValue(srcCell.getStringCellValue());
                break;
            case NUMERIC:
                dstCell.setCellValue(srcCell.getNumericCellValue());
                break;
            case BLANK:
                dstCell.setBlank();
                break;
            case BOOLEAN:
                dstCell.setCellValue(srcCell.getBooleanCellValue());
                break;
            case ERROR:
                dstCell.setCellErrorValue(srcCell.getErrorCellValue());
                break;
            case FORMULA:
                dstCell.setCellFormula(srcCell.getCellFormula());
                break;
            default:
                break;
        }
    }

    private void prepareTemplateContext(Workbook wb, Map<String, Object> context) {

        Iterator<Sheet> sheetIterator = wb.sheetIterator();
        int i = 0;
        while (sheetIterator.hasNext()) {
            Sheet sheet = sheetIterator.next();
            List<Tuple2<AtomicInteger, Row>> dynamicRow = new ArrayList<>();
            Iterator<Row> rowIterator = sheet.rowIterator();
            while (rowIterator.hasNext()) {
                Row next = rowIterator.next();
                //the rowNum
                Iterator<Cell> cellIterator = next.cellIterator();
                List<Cell> tempList = new ArrayList<>();
                boolean isDynamicRow = false;
                while (cellIterator.hasNext()) {
                    Cell targetCell = cellIterator.next();
                    tempList.add(targetCell);
                    if (targetCell.getCellType() == CellType.STRING) {
                        String stringCellValue = targetCell.getStringCellValue();
                        if (stringCellValue.startsWith("$") || stringCellValue.startsWith("@")) {
                            //this is a variable row
                            isDynamicRow = true;
                        }
                    }
                }

                if (isDynamicRow) {
                    dynamicRow.add(Tuple.of(new AtomicInteger(next.getRowNum()), next));
                }
            }

            context.put(getSheetIndexName(sheet, i), dynamicRow);
            i++;
        }
    }


    private String getSheetIndexName(Sheet sheet, int i) {
        return sheet.getSheetName() + "_$" + i;
    }

    @SneakyThrows
    public Sink<ClassifiedRecord, InputStream> excelSink(
            List<Tuple2<String, IEntityClass>> entityClasses
            , Map<String, ExportSchemaConfig> schemaMapping
            , boolean isSkipTransformer
            , Map<String, Object> context
            , String template
    ) {

        SXSSFWorkbook wb = new SXSSFWorkbook(excelConfig.getInMemRow());

        PipedOutputStream out = new PipedOutputStream();
        PipedInputStream in = new PipedInputStream();

        AtomicBoolean useTemplate = new AtomicBoolean(false);
        Map<String, Tuple2<Sheet, AtomicInteger>> sheetMapping = null;
        File tempTemplateFile = null;
        XSSFWorkbook templateWB = null;
        if (StringUtils.isEmpty(template)) {
            sheetMapping = new ConcurrentHashMap<>(36);
            prepareSheet(wb, entityClasses, schemaMapping, sheetMapping);
        } else {
            /**
             * delete further
             */
            try {
                tempTemplateFile = File.createTempFile("excel-template", "xls");
                JsonSpreadsheetParser parser = new JsonSpreadsheetParser(PoiSpreadsheetBuilder.create(tempTemplateFile));
                parser.parse(IOUtils.toInputStream(template, StandardCharsets.UTF_8));
                templateWB = new XSSFWorkbook(tempTemplateFile);
                Iterator<Sheet> sheetIterator = templateWB.sheetIterator();
                while (sheetIterator.hasNext()) {
                    Sheet next = sheetIterator.next();
                    PoiCopySheet.copySheet(next, wb.createSheet(next.getSheetName()));
                }
                try {
                    //remove the template file
                    templateWB.close();
                    tempTemplateFile.delete();
                } catch (Throwable throwable) {
                    logger.error("{}", throwable);
                }
                //prepare dynamic
                prepareTemplateContext(wb, context);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
            useTemplate.set(true);
        }
        try {
            in.connect(out);
        } catch (IOException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }

        Map<Integer, CellStyle> columnNumberFormatStyles = generateColumnNumberFormatStyles(wb);

        /**
         * a stateful sink
         */
        Map<String, Tuple2<Sheet, AtomicInteger>> finalSheetMapping = sheetMapping;
        Sink<ClassifiedRecord, InputStream> excelSink = Sink
                .<ClassifiedRecord>foreach(tuple -> {
                    if (useTemplate.get()) {
                        //SpreadsheetCriteria spreadsheetCriteria = PoiSpreadsheetCriteria.FACTORY.forFile(finalTempTemplateFile);
                        renderTemplate(wb, tuple, context, isSkipTransformer, columnNumberFormatStyles, schemaMapping);

                    } else {
                        /**
                         *             , Map<String, Tuple2<Sheet, AtomicInteger>> sheetMapping
                         *             , Map<String, ExportSchemaConfig> schemaMapping
                         *             , Workbook wb, Map<String, Object> context
                         *             ,  boolean isSkipTransformer
                         *             ,  Map<Integer, CellStyle> columnNumberFormatStyles
                         */
                        insertInDefaultWorkBook(tuple, finalSheetMapping, schemaMapping, wb, context, isSkipTransformer, columnNumberFormatStyles);
                    }
                }).mapMaterializedValue(x -> {
                    x.thenRunAsync(() -> {
                        try {
                            wb.write(out);
                            wb.close();
                            wb.dispose();
                        } catch (Throwable ex) {
                            ex.printStackTrace();
                        } finally {
                            try {
                                out.close();
                            } catch (IOException e) {
                                e.printStackTrace();
                            }
                        }
                    }, executorService);
                    return in;
                });

        return excelSink;
    }


    /**
     * 提前创建好数值列需要的格式化样式
     * <p>
     * 限制:oqs能保存整数位19位+小数位19位,小数位极限是19位
     * Excel精度是15位,超过部分会被舍去,不是四舍五入
     * Excel小数位最多设置到30位
     *
     * @return 返回的是根据字段precision精度位匹配的样式Map
     * 比如,获取精度是2的格式化样式,可以使用result.get(2),拿到样式
     */
    private Map<Integer, CellStyle> generateColumnNumberFormatStyles(Workbook wb) {
        Map<Integer, CellStyle> result = new HashMap<>();
        // Excel小数位格式化最多可以设置到30位
        String formatTemplate = "000000000000000000000000000000";
        for (int precision = 0; precision < 20; precision++) {
            CellStyle cellStyle = wb.createCellStyle();
            DataFormat dataFormat = wb.createDataFormat();
            // 基于oqs存储限制,最多显示到19位小数
            if (precision > 19) {
                precision = 19;
            }
            if (result.containsKey(precision)) {
                continue;
            }
            // 动态生成对应小数位精度的格式化信息
            StringBuilder format = new StringBuilder();
            format.append("0");
            if (precision > 0) {
                format.append(".");
                format.append(formatTemplate.substring(0, precision));
            }
            cellStyle.setDataFormat(dataFormat.getFormat(format.toString()));
            result.put(precision, cellStyle);
        }
        return result;
    }


    /**
     * 设置单元格的数值格式化样式并塞值
     * 特殊情况会保留字符串格式
     * 限制:oqs能保存整数位19位+小数位19位,小数位极限是19位
     * Excel精度是15位,超过部分会被舍去,不是四舍五入
     */
    @SneakyThrows
    private void setNumberFormatStyleAndCellValue(Cell cell, Map<Integer, CellStyle> columnNumberFormatStyles, FieldConfig fieldConfig, BigDecimal value) {
        // 小数位
        int precition = fieldConfig.getPrecision();
        if (precition > 19) {
            precition = 19;
        }
        // excel限制精度15,有效数字超过15位则会被excel截取
        String plainStringValue = value.toPlainString();
        if (plainStringValue.length() > 15) {
            if (plainStringValue.indexOf('.') > -1) {
                /* 去掉 字符串小数  末尾的 0 */
                plainStringValue = plainStringValue.replaceAll("(0)+$", "");
            }
            // 去除前面的0,为了获取有效数字
            plainStringValue = new BigDecimal(plainStringValue.replaceAll("\\.", "")).toPlainString();
        }
        if (plainStringValue.length() > 15) {
            cell.setCellValue(value.toString());
            return;
        }

        CellStyle cellStyle = columnNumberFormatStyles.get(fieldConfig.getPrecision());
        // 小数位超19位只保留19位
        cellStyle = columnNumberFormatStyles.get(precition);
        if (cellStyle == null) {
            throw new Exception("数值格式化样式丢失,index: " + fieldConfig.getPrecision());
        }
        cell.setCellStyle(cellStyle);
        cell.setCellValue(value.doubleValue());
    }
}
