package com.xforceplus.ultraman.oqsengine.sdk.service.export.impl;

import akka.stream.ActorMaterializer;
import akka.stream.IOResult;
import akka.stream.javadsl.Source;
import akka.stream.javadsl.StreamConverters;
import akka.util.ByteString;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.FieldType;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityField;
import com.xforceplus.ultraman.oqsengine.sdk.facade.EntityFacade;
import com.xforceplus.ultraman.oqsengine.sdk.service.DictService;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.ExportSink;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.ImportService;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.ImportTemplateService;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.PostImportAware;
import com.xforceplus.ultraman.oqsengine.sdk.store.engine.IEntityClassGroup;
import com.xforceplus.ultraman.oqsengine.sdk.transactional.OqsTransactionManager;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.DictItem;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.NameMapping;
import io.vavr.Tuple2;
import io.vavr.Tuple3;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.ss.SpreadsheetVersion;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddressList;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

/**
 *
 */
public class ImportDefaultExcelTemplateServiceImpl implements ImportTemplateService {

    private Logger logger = LoggerFactory.getLogger(ImportService.class);

    protected EntityFacade entityFacade;

    protected ContextService contextService;

    protected ExecutorService importThreadPool;

    protected OqsTransactionManager manager;

    private ActorMaterializer mat;

    @Autowired
    public ApplicationEventPublisher publisher;

    @Autowired(required = false)
    List<PostImportAware> postImportAwareList = new ArrayList<>();

    private int defaultStep = 1000;

    @Autowired
    private DictService dictService;

    private final String MULTI_ENUM = "请选择一项或者多项, 以,号隔开:\n\n%s\n\n 存在多个, 请以,号隔开";


    /**
     * reuse the file-sink
     */
    private ExportSink sink;

    public ImportDefaultExcelTemplateServiceImpl(EntityFacade entityFacade, ContextService contextService
            , ExportSink sink
            , ActorMaterializer mat
            , ExecutorService importThreadPool
            , OqsTransactionManager manager
    ) {
        this.entityFacade = entityFacade;
        this.contextService = contextService;
        this.importThreadPool = importThreadPool;
        this.sink = sink;
        this.mat = mat;
        this.manager = manager;
    }

    @Override
    public String fileType() {
        return "xlsx";
    }

    @Override
    public String fileName(IEntityClass entityClass) {
        return entityClass.name() + "-template";
    }

    @Override
    public InputStream getTemplateInputStream(IEntityClassGroup reader, boolean skipSystem) {

        IEntityClass entityClass = reader.getEntityClass();
        XSSFWorkbook workbook = new XSSFWorkbook();
        XSSFSheet mainSheet = workbook.createSheet(entityClass.name());
        XSSFRow headerRow = mainSheet.createRow(0);
        CellStyle cellStyle = workbook.createCellStyle();
        AtomicInteger index = new AtomicInteger(0);

        cellStyle.setFillForegroundColor(IndexedColors.LIGHT_ORANGE.getIndex());
        cellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);

        short dataFormatIndex = workbook.createDataFormat().getFormat("yyyy-m-dd h:mm:ss");
        CellStyle dateStyle = workbook.createCellStyle();
        dateStyle.setDataFormat(dataFormatIndex);

        List<IEntityField> fields = reader.getAllFields().stream().filter(x -> {
            return !skipSystem || (x.config() == null || !x.config().isSystem());
        }).collect(Collectors.toList());
        fields.forEach(field -> {
            int currentIndex = index.getAndIncrement();
            XSSFCell cell = headerRow.createCell(currentIndex);
            List<IEntityField> duplicateCnNameField = fields.stream().filter(fieldInner -> fieldInner.cnName() != null && fieldInner.cnName().equals(field.cnName())).collect(Collectors.toList());
            if (duplicateCnNameField.size() > 1) {
                cell.setCellValue(Optional.ofNullable(field.cnName()).orElse(field.name()) + "[" + field.name() + "]");
            } else {
                cell.setCellValue(Optional.ofNullable(field.cnName()).orElse(field.name()));
            }

            setCellStyle(workbook, mainSheet, field, currentIndex, cell);
            mainSheet.autoSizeColumn(currentIndex);

        });

        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();

        try {
            workbook.write(byteArrayOutputStream);
        } catch (IOException e) {
        }

        return new ByteArrayInputStream(byteArrayOutputStream.toByteArray());
    }

    private List<IEntityField> getOrderedFieldsFromNameMapping(List<NameMapping> mappingList, List<IEntityField> fields) {
        if (mappingList == null || mappingList.isEmpty()) {
            return fields;
        } else {
            return mappingList.stream().map(x -> {
                Optional<IEntityField> first = fields.stream().filter(field -> field.name().equals(x.getCode())).findFirst();
                return first;
            }).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
        }
    }

    @Override
    public String getCustomTemplate(IEntityClassGroup reader, List<NameMapping> mappingList
            , Map<String, Tuple3<List<NameMapping>, List<NameMapping>, String>> subMapping) {
        IEntityClass entityClass = reader.getEntityClass();
        XSSFWorkbook workbook = new XSSFWorkbook();

        /**
         * nameMapping key or value cannot be null
         */
        Map<String, String> nameMapping = mappingList.stream()
                .collect(Collectors.toMap(x -> x.getCode(), x -> x.getText(), (a, b) -> a));
        renderSheet(workbook, entityClass.name(), getOrderedFieldsFromNameMapping(mappingList, reader.getAllFields()), nameMapping, null, null);

        subMapping.forEach((code, subRelMapping) -> {
            List<NameMapping> relationMapping = subRelMapping._2();
            List<NameMapping> mapping = subRelMapping._1();
            Map<String, String> subNameMapping = mapping.stream()
                    .collect(Collectors.toMap(x -> x.getCode(), x -> x.getText(), (a, b) -> a));
            Optional<IEntityClass> relatedEntityClassOp = reader.relatedEntityClass(code);
            if (relatedEntityClassOp.isPresent()) {
                IEntityClass relatedEntityClass = relatedEntityClassOp.get();
                IEntityClassGroup relatedGroup = entityFacade.getReader(relatedEntityClass, contextService.getAll());
                renderSheet(workbook, subRelMapping._3, getOrderedFieldsFromNameMapping(mapping, relatedGroup.getAllFields()), subNameMapping, relationMapping, entityClass);
            }
        });

        String fileName = fileName(reader.getEntityClass());
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        String uuid = UUID.randomUUID().toString();
        try {
            workbook.write(byteArrayOutputStream);
        } catch (IOException e) {
            logger.error("{}", e);
        } finally {

        }

        Source<ByteString, CompletionStage<IOResult>> source = StreamConverters.fromInputStream(() -> new ByteArrayInputStream(byteArrayOutputStream.toByteArray()));
        Tuple2<IOResult, String[]> xlsx = source.runWith(sink.getSink("xlsx", uuid, fileName), mat).toCompletableFuture().join();
        return sink.getDownloadUrl("xlsx", xlsx._2);
    }

    private void renderSheet(XSSFWorkbook workbook, String name, List<IEntityField> fields, Map<String, String> nameMapping, List<NameMapping> additionalCell, IEntityClass mainEntityClass) {
        XSSFSheet mainSheet = workbook.createSheet(name);
        XSSFRow headerRow = mainSheet.createRow(0);

        AtomicInteger index = new AtomicInteger(0);

        //render headerRow
        if (additionalCell != null && !additionalCell.isEmpty()) {
            additionalCell.forEach(x -> {
                IEntityField iEntityField = mainEntityClass.fields().stream().filter(field -> field.name().equals(x.getCode())).findFirst().get();
                int currentIndex = index.getAndIncrement();
                XSSFCell cell = headerRow.createCell(currentIndex);
                String cellName = x.getText();
                String cellCode = x.getCode();
                addComment(workbook, mainSheet, cell, cellCode);
                cell.setCellValue(cellName);
//                cell.setCellValue(cellName + "[" + cellCode + "]");

                setCellStyle(workbook, mainSheet, iEntityField, currentIndex, cell);
                mainSheet.autoSizeColumn(currentIndex);
            });
        }

        fields.stream().filter(x -> {
            return nameMapping.containsKey(x.name());
        }).forEach(field -> {
            int currentIndex = index.getAndIncrement();
            XSSFCell cell = headerRow.createCell(currentIndex);
            String title = Optional.ofNullable(nameMapping.get(field.name())).orElse(field.cnName());
            List<IEntityField> duplicateCnNameField = fields.stream().filter(fieldInner -> fieldInner.cnName() != null && fieldInner.cnName().equals(field.cnName())).collect(Collectors.toList());
            if (duplicateCnNameField.size() > 1) {
                cell.setCellValue(title + "[" + field.name() + "]");
            } else {
                cell.setCellValue(title);
            }

            setCellStyle(workbook, mainSheet, field, currentIndex, cell);
            mainSheet.setColumnWidth(currentIndex, title.getBytes().length * 256);
        });
    }

    private void setCellStyle(XSSFWorkbook workbook, XSSFSheet mainSheet, IEntityField field, int currentIndex, XSSFCell cell) {
        CellStyle headCellStyle = workbook.createCellStyle();
        CellStyle columnStyle = workbook.createCellStyle();

        try {
            headCellStyle.setBorderBottom(BorderStyle.THIN); //下边框
            headCellStyle.setBorderLeft(BorderStyle.THIN);//左边框
            headCellStyle.setBorderTop(BorderStyle.THIN);//上边框
            headCellStyle.setBorderRight(BorderStyle.THIN);//右边框
            headCellStyle.setFillForegroundColor(IndexedColors.LIGHT_GREEN.getIndex());
            headCellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);

            if (field.config().isRequired()) {
                headCellStyle.setFillForegroundColor(IndexedColors.LIGHT_ORANGE.getIndex());
                addComment(workbook, mainSheet, cell, "必填项");
//                Font font = workbook.createFont();
//                font.setColor(IndexedColors.RED.getIndex());
//                font.setBold(true);
//                headCellStyle.setFont(font);
            }
            boolean isLongThan15 = false;

            if (field.type() == FieldType.DATETIME) {
                //todo-rayder 日期格式取元数据格式化
                short dataFormatIndex = workbook.createDataFormat().getFormat("yyyy-m-dd h:mm:ss");
                columnStyle.setDataFormat(dataFormatIndex);
            } else if (field.type() == FieldType.ENUM) {
                String dictId = field.dictId();
                List<DictItem> dictItemsByCode = dictService.findDictItems(dictId, "");
                List values = dictItemsByCode.stream().map(DictItem::getText).collect(Collectors.toList());
                addSelectionList(mainSheet, cell, values);
            } else if (field.type() == FieldType.STRINGS && StringUtils.isNotEmpty(field.dictId())) {
                String dictId = field.dictId();
                List<DictItem> dictItemsByCode = dictService.findDictItems(dictId, "");
                List values = dictItemsByCode.stream().map(DictItem::getText).collect(Collectors.toList());
                String comment = String.format(MULTI_ENUM, values.stream().collect(Collectors.joining("\n")));
                addComment(workbook, mainSheet, cell, comment);
            } else if (field.type() == FieldType.BOOLEAN) {
                addSelectionList(mainSheet, cell, Arrays.asList("TRUE", "FALSE"));
            } else if (field.type() == FieldType.DECIMAL) {
                StringBuilder formatBuilder = new StringBuilder();
                formatBuilder.append("0");
                if (field.config().getPrecision() > 0) {
                    formatBuilder.append(".");
                }
                for (int i = 0; i < field.config().getPrecision(); i++) {
                    formatBuilder.append("0");
                }
                short dataFormat = workbook.createDataFormat().getFormat(formatBuilder.toString());
                columnStyle.setDataFormat(dataFormat);
                addDecimalMaxValidation(mainSheet, cell);
            } else if (field.type() == FieldType.LONG) {
                if (field.config().getMax() <= 15) {
                    short dataFormat = workbook.createDataFormat().getFormat("#");
                    columnStyle.setDataFormat(dataFormat);
                    addIntegerMaxValidation(mainSheet, cell);
                } else {
                    // 整型如果超过15位,excel最大精度15位,按字符串样式处理
                    isLongThan15 = true;
                }
            }
            if (field.type() == FieldType.STRING
                    || field.type() == FieldType.STRINGS
                    || field.type() == FieldType.ENUM
                    || isLongThan15
            ) {
                // 文本格式
                // 字符串单元格默认是Excel的常规格式，如果是纯数字导入时会变成数字格式，比如导入值为123，实际会变为123.0，这里需要设置文本格式解决
                short stringFormatIndex = workbook.createDataFormat().getFormat("@");
                columnStyle.setDataFormat(stringFormatIndex);
                if (field.config().getMax() > 0 && field.type() != FieldType.ENUM) {
                    addTextLengthValidation(mainSheet, cell, field.config().getMax());
                }
            }
        } catch (Exception e) {
            logger.error("genCellStyle error", e);
        }

        cell.setCellStyle(headCellStyle);
        mainSheet.setDefaultColumnStyle(currentIndex, columnStyle);

    }

    public void addComment(Workbook workbook, Sheet sheet, Cell cell, String commentText) {
        CreationHelper factory = workbook.getCreationHelper();

        ClientAnchor anchor = factory.createClientAnchor();
        anchor.setCol1(cell.getColumnIndex() + 1); //the box of the comment starts at this given column...
        anchor.setCol2(cell.getColumnIndex() + 3); //...and ends at that given column
        anchor.setRow1(cell.getRowIndex() + 1); //one row below the cell...
        anchor.setRow2(cell.getRowIndex() + 3 + commentText.split("\n").length); //...and 4 rows high

        Drawing drawing = sheet.createDrawingPatriarch();
        Comment comment = drawing.createCellComment(anchor);
        comment.setString(factory.createRichTextString(commentText));

        cell.setCellComment(comment);
    }

    public void addSelectionList(Sheet sheet, Cell cell, List<String> enumValues) {

        DataValidationHelper dvHelper = sheet.getDataValidationHelper();
        DataValidationConstraint dvConstraint = dvHelper.createExplicitListConstraint(enumValues.toArray(new String[enumValues.size()]));
        CellRangeAddressList addressList = new CellRangeAddressList(cell.getRowIndex() + 1, SpreadsheetVersion.EXCEL2007.getLastRowIndex(), cell.getColumnIndex(), cell.getColumnIndex());// 第一个参数是开始行，第二个是结束行，第三个是开始列，第四个是结束列
        DataValidation validation = dvHelper.createValidation(dvConstraint, addressList);

        validation.setSuppressDropDownArrow(true);
        validation.setEmptyCellAllowed(true);
        validation.createErrorBox("校验出错", "只能输入下拉框里的值");
        validation.setShowErrorBox(true);
        sheet.addValidationData(validation);
    }

    public void addIntegerMaxValidation(Sheet sheet, Cell cell) {

        DataValidationHelper dvHelper = sheet.getDataValidationHelper();
        DataValidationConstraint integerConstraint = dvHelper.createIntegerConstraint(DataValidationConstraint.OperatorType.LESS_THAN, String.valueOf(Long.MAX_VALUE), "");
        CellRangeAddressList addressList = new CellRangeAddressList(cell.getRowIndex() + 1, SpreadsheetVersion.EXCEL2007.getLastRowIndex(), cell.getColumnIndex(), cell.getColumnIndex());// 第一个参数是开始行，第二个是结束行，第三个是开始列，第四个是结束列
        DataValidation validation = dvHelper.createValidation(integerConstraint, addressList);

        validation.setEmptyCellAllowed(true);
        validation.createErrorBox("校验出错", "只能输入整数");
        validation.setShowErrorBox(true);
        sheet.addValidationData(validation);
    }

    public void addDecimalMaxValidation(Sheet sheet, Cell cell) {

        DataValidationHelper dvHelper = sheet.getDataValidationHelper();
        DataValidationConstraint integerConstraint = dvHelper.createDecimalConstraint(DataValidationConstraint.OperatorType.LESS_THAN, String.valueOf(Long.MAX_VALUE), "");
        CellRangeAddressList addressList = new CellRangeAddressList(cell.getRowIndex() + 1, SpreadsheetVersion.EXCEL2007.getLastRowIndex(), cell.getColumnIndex(), cell.getColumnIndex());// 第一个参数是开始行，第二个是结束行，第三个是开始列，第四个是结束列
        DataValidation validation = dvHelper.createValidation(integerConstraint, addressList);

        validation.setEmptyCellAllowed(true);
        validation.createErrorBox("校验出错", "只能输入数字");
        validation.setShowErrorBox(true);
        sheet.addValidationData(validation);
    }

    public void addTextLengthValidation(Sheet sheet, Cell cell, Long maxLength) {

        DataValidationHelper dvHelper = sheet.getDataValidationHelper();
        DataValidationConstraint textLengthConstraint = dvHelper.createTextLengthConstraint(DataValidationConstraint.OperatorType.LESS_OR_EQUAL, String.valueOf(Long.MAX_VALUE), "");
        textLengthConstraint.setFormula1(String.valueOf(maxLength));
        CellRangeAddressList addressList = new CellRangeAddressList(cell.getRowIndex() + 1, SpreadsheetVersion.EXCEL2007.getLastRowIndex(), cell.getColumnIndex(), cell.getColumnIndex());// 第一个参数是开始行，第二个是结束行，第三个是开始列，第四个是结束列
        DataValidation validation = dvHelper.createValidation(textLengthConstraint, addressList);

        validation.setEmptyCellAllowed(true);
        validation.createErrorBox("校验出错", "超过最大长度" + maxLength);
        validation.setShowErrorBox(true);
        sheet.addValidationData(validation);
    }

    @Override
    public InputStream getCustomTemplateInputStream(String uuid) {
        return sink.getInputStream(uuid, fileType());
    }
}