package com.xforceplus.ultraman.oqsengine.sdk.service.export.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.json.JSONUtil;
import com.xforceplus.tech.base.core.context.ContextKeys;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.impl.ColumnField;
import com.xforceplus.ultraman.oqsengine.sdk.command.ImportCmd;
import com.xforceplus.ultraman.oqsengine.sdk.event.EntityImported;
import com.xforceplus.ultraman.oqsengine.sdk.facade.EntityFacade;
import com.xforceplus.ultraman.oqsengine.sdk.facade.result.CreateMultiResult;
import com.xforceplus.ultraman.oqsengine.sdk.facade.result.CreateOneResult;
import com.xforceplus.ultraman.oqsengine.sdk.facade.result.ResultStatus;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.ImportService;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.PostImportAware;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.entity.ImportResult;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.enums.ImportModeEnum;
import com.xforceplus.ultraman.oqsengine.sdk.store.engine.IEntityClassGroup;
import com.xforceplus.ultraman.oqsengine.sdk.transactional.DefaultTransactionManager;
import com.xforceplus.ultraman.oqsengine.sdk.transactional.OqsTransaction;
import com.xforceplus.ultraman.oqsengine.sdk.transactional.OqsTransactionManager;
import com.xforceplus.ultraman.oqsengine.sdk.util.SnowflakeLongIdGenerator;
import io.vavr.control.Either;
import io.vavr.control.Validation;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;

import java.io.IOException;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.stream.Collectors;

/**
 *
 */
public class ImportWizardExcelServiceImpl implements ImportService {
    private final Logger logger = LoggerFactory.getLogger(ImportService.class);

    protected EntityFacade entityFacade;

    protected ContextService contextService;

    protected ExecutorService importThreadPool;

    protected OqsTransactionManager manager;

    protected ApplicationEventPublisher publisher;

    @Autowired(required = false)
    List<PostImportAware> postImportAwareList = new ArrayList<>();

    SnowflakeLongIdGenerator snowflakeLongIdGenerator;

    private int defaultStep = 1000;

    public ImportWizardExcelServiceImpl(EntityFacade entityFacade
            , ContextService contextService
            , ExecutorService importThreadPool
            , OqsTransactionManager manager
            , ApplicationEventPublisher publisher
    ) {
        this.entityFacade = entityFacade;
        this.contextService = contextService;
        this.importThreadPool = importThreadPool;
        this.manager = manager;
        this.publisher = publisher;
        this.snowflakeLongIdGenerator = new SnowflakeLongIdGenerator(new Random(System.currentTimeMillis()).nextInt(1000));
    }

    @Override
    public ImportModeEnum getImportMode() {
        return ImportModeEnum.WIZARD;
    }

    @Override
    public ImportResult doImport(IEntityClassGroup entityClassGroup, ImportCmd cmd) {
        if (cmd.getStep() <= 0) {
            cmd.setStep(defaultStep);
        }
        ImportResult importResult = new ImportResult();
        importResult.setSheetImportResultList(new ArrayList<>());

        XSSFWorkbook xssfwb = null;
        try {
            xssfwb = new XSSFWorkbook(cmd.getInputStream());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        IEntityClass mainEntityClass = entityClassGroup.getEntityClass();

        String transKey = contextService.get(ContextKeys.StringKeys.TRANSACTION_KEY);
        if (transKey == null) {
            //then start new
            OqsTransaction newTransaction = ((DefaultTransactionManager) manager).createNewTransaction(cmd.getTimeout(), "");
            contextService.set(ContextKeys.StringKeys.TRANSACTION_KEY, newTransaction.getId());
            transKey = newTransaction.getId();
        }
        // 字段平铺模式仅支持单sheet
        // 多sheet模式，第一张是主表，其他是从表
        try {
            // 外键依赖解析
            Map<String, Object> contextMap = getContextRelatedHeaderMap(cmd);
            contextMap.put(ContextKeys.StringKeys.TRANSACTION_KEY.name(), transKey);

            List<ImportCmd.Sheet> sheetsConfigList = cmd.getSheets();
            for (int i = 0; i < sheetsConfigList.size(); i++) {
                insertMain(entityClassGroup, xssfwb.getSheetAt(i), contextMap, sheetsConfigList.get(i), cmd, importResult);
                insertRelated(entityClassGroup, xssfwb.getSheetAt(i), contextMap, sheetsConfigList.get(i), cmd, importResult);
            }
            importResult.summary();
            Map<String, Object> notifyContext = new HashMap<>();
            notifyContext.put("appId", cmd.getAppId());
            notifyContext.put("importResultInfo", importResult.toHumanFormatString());
            logger.info(JSONUtil.toJsonStr(importResult));
            logger.info(importResult.toHumanFormatString());
            publisher.publishEvent(new EntityImported(cmd.isAsync() ? "async" : "sync", mainEntityClass.code(), mainEntityClass.code(), notifyContext, contextMap));
            ((DefaultTransactionManager) manager).commit(transKey);
        } catch (Exception e) {
            logger.error("", e);
            if (cmd.getUseBatch()) {
                ((DefaultTransactionManager) manager).rollBack(transKey);
            }
            throw e;
        } finally {
            try {
                xssfwb.close();
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        return importResult;
    }

    private void insertMain(IEntityClassGroup entityClassGroup
            , XSSFSheet sheet
            , Map<String, Object> contextMap
            , ImportCmd.Sheet sheetConfig
            , ImportCmd cmd
            , ImportResult importResult) {
        if (sheet == null) {
            throw new RuntimeException("sheet is null");
        }
        IEntityClass mainEntityClass = entityClassGroup.getEntityClass();
        if (!isMain(sheetConfig, cmd, mainEntityClass)) {
            return;
        }
        ImportResult.SheetImportResult sheetImportResult = initSheetImportResult(sheet, importResult, mainEntityClass.code());
        List<ImportCmd.Sheet.FieldMapping> headerFiledMapping = sheetConfig.getFieldMapping();

        int currentBatchCount = 0;
        Iterator<Row> rowIterator = sheet.rowIterator();
        Row headerRow = rowIterator.next();
        short lastCellNum = headerRow.getLastCellNum();
        List<String> headerFiledCodeList = getHeaderFiledCodeList(mainEntityClass, headerFiledMapping, headerRow, lastCellNum);

        List batchList = new ArrayList();
        while (rowIterator.hasNext()) {
            Map<String, Object> rowContentMap = new HashMap<>();
            currentBatchCount++;
            Row row = rowIterator.next();

            for (int j = 0; j < lastCellNum; j++) {
                Cell cell = row.getCell(j);
                if (mainEntityClass.code().equals(headerFiledMapping.get(j).getBoCode())) {
                    rowContentMap.put(headerFiledCodeList.get(j), cell == null ? null : readCellRawValue(cell));
                }
            }

            Long nextId = snowflakeLongIdGenerator.next();
            // 如果是平铺模式，且主键重复，跳过
            if (CollUtil.isNotEmpty(cmd.getToManyRelations())
                    && !buildRelatedIndex(cmd.getToManyRelations().get(0).getRelationCode(), contextMap, rowContentMap, nextId)
                    && cmd.getSheets().size() == 1
                    ) {
                sheetImportResult.getFailedCount().incrementAndGet();
                sheetImportResult.addErrorResult(row.getRowNum() + 1, "主键重复，跳过");
                continue;
            }
            rowContentMap.put("id", nextId);
            if (cmd.getUseBatch()) {
                batchList.add(rowContentMap);
                if (currentBatchCount % cmd.getStep() == 0 || !rowIterator.hasNext()) {
                    insertBatchRecord(contextMap, mainEntityClass.code(), batchList, row.getRowNum() + 1 - batchList.size(), sheetImportResult);
                    batchList.clear();
                }
            } else {
                insertRecord(contextMap, row.getRowNum() + 1, mainEntityClass.code(), rowContentMap, sheetImportResult);
            }

            if (currentBatchCount % cmd.getStep() == 0 && !cmd.getUseBatch()) {
                String transKey = contextService.get(ContextKeys.StringKeys.TRANSACTION_KEY);
                if (transKey != null) {
                    ((DefaultTransactionManager) manager).commit(transKey);
                }
                //then start new
                OqsTransaction newTransaction = ((DefaultTransactionManager) manager).createNewTransaction(cmd.getTimeout(), "");
                contextService.set(ContextKeys.StringKeys.TRANSACTION_KEY, newTransaction.getId());
                currentBatchCount = 0;
            }
        }

        String transKey = contextService.get(ContextKeys.StringKeys.TRANSACTION_KEY);
        if (transKey != null && currentBatchCount > 0 && !cmd.getUseBatch()) {
            ((DefaultTransactionManager) manager).commit(transKey);
        }

    }

    @Nullable
    private static Boolean isMain(ImportCmd.Sheet sheetConfig, ImportCmd cmd, IEntityClass mainEntityClass) {
        // 关系字段对象Code集合
        Set<String> relationBoCodeSet = sheetConfig.getFieldMapping().stream()
                .filter(fieldMapping -> fieldMapping.getBoCode() != null && !fieldMapping.getBoCode().equals(mainEntityClass.code()))
                .map(fieldMapping -> fieldMapping.getBoCode())
                .collect(Collectors.toSet());

        Boolean isMultiBoHeader = relationBoCodeSet.size() > 0;
        Boolean isMultiSheet = cmd.getSheets().size() > 1;
        // 是主表且是多bo表头，
        if (isMultiBoHeader && !isMultiSheet) {
            return true;
        }
        // 单一表头，是主表
        if (!isMultiBoHeader) {
            return true;
        }
        return false;
    }

    /**
     * insert related
     * toManyRelations里面的关联字段都一样，默认取第一条的字段处理
     */
    private void insertRelated(IEntityClassGroup iEntityClassGroup
            , XSSFSheet sheet
            , Map<String, Object> contextMap
            , ImportCmd.Sheet sheetsConfig
            , ImportCmd cmd
            , ImportResult importResult) {

        IEntityClass mainEntityClass = iEntityClassGroup.getEntityClass();
        List<ImportCmd.Sheet.FieldMapping> headerFiledMapping = sheetsConfig.getFieldMapping();
        Set<String> relationBoCodeSet = headerFiledMapping.stream()
                .filter(fieldMapping -> fieldMapping.getBoCode() != null && !fieldMapping.getBoCode().equals(mainEntityClass.code()))
                .map(fieldMapping -> fieldMapping.getBoCode())
                .collect(Collectors.toSet());
        if (relationBoCodeSet.size() == 0) {
            return;
        }

        ImportResult.SheetImportResult sheetImportResult = initSheetImportResult(sheet, importResult, "");

        Iterator<Row> rowIterator = sheet.rowIterator();
        Row headerRow = rowIterator.next();
        short size = headerRow.getLastCellNum();

        List<Integer> relationCodeColIndexList = getRelationCodeColIndexList(cmd.getToManyRelations(), mainEntityClass, headerFiledMapping, size);

        int currentBatchCount = 0;
        Map<String, List<Map<String, Object>>> boMap = new HashMap<>();
        Map<String, List<Map<String, Object>>> batchListMap = new HashMap<>();
        relationBoCodeSet.forEach(boCode -> boMap.put(boCode, new ArrayList<>()));
        relationBoCodeSet.forEach(boCode -> batchListMap.put(boCode, new ArrayList<>()));
        while (rowIterator.hasNext()) {
            Row row = rowIterator.next();
            currentBatchCount++;

            Long relatedId = getRelatedIdAndSetContent(contextMap, cmd.getToManyRelations(), headerFiledMapping, size, relationCodeColIndexList, row, boMap);

            int index = row.getRowNum() - 1;
            if (relatedId == null) {
                logger.error("{} has no related value,body: {}", cmd.getToManyRelations().get(0).getRelationCode(), boMap);
            } else {
                boMap.forEach((k, v) -> {
                    String relationCode = cmd.getToManyRelations().stream().filter(toManyRelation -> toManyRelation.getBoCode().equals(k)).findFirst().get().getRelationCode();
                    v.get(index).put(relationCode + ".id", relatedId);
                });
            }

            int finalCurrentBatchCount = currentBatchCount;
            boMap.forEach((k, v) -> {
                if (cmd.getUseBatch()) {
                    batchListMap.get(k).add(v.get(index));
                    if (finalCurrentBatchCount % cmd.getStep() == 0 || !rowIterator.hasNext()) {
                        insertBatchRecord(contextMap, k, batchListMap.get(k), row.getRowNum() + 1 - batchListMap.get(k).size(), sheetImportResult);
                        batchListMap.get(k).clear();
                    }
                } else {
                    insertRecord(contextMap, row.getRowNum() + 1, k, v.get(index), sheetImportResult);
                }
            });

            if (currentBatchCount % cmd.getStep() == 0 && !cmd.getUseBatch()) {
                String transKey = contextService.get(ContextKeys.StringKeys.TRANSACTION_KEY);
                if (transKey != null) {
                    ((DefaultTransactionManager) manager).commit(transKey);
                }
                //then start new
                OqsTransaction newTransaction = ((DefaultTransactionManager) manager).createNewTransaction(cmd.getTimeout(), "");
                contextService.set(ContextKeys.StringKeys.TRANSACTION_KEY, newTransaction.getId());
                currentBatchCount = 0;
            }
        }
        String transKey = contextService.get(ContextKeys.StringKeys.TRANSACTION_KEY);
        if (transKey != null && currentBatchCount > 0 && !cmd.getUseBatch()) {
            ((DefaultTransactionManager) manager).commit(transKey);
        }

    }

    @NotNull
    private static ImportResult.SheetImportResult initSheetImportResult(XSSFSheet sheet, ImportResult importResult, String boCode) {
        ImportResult.SheetImportResult sheetImportResult = new ImportResult.SheetImportResult();
        sheetImportResult.setSheetName(sheet.getSheetName());
        sheetImportResult.setTotalCount(sheet.getLastRowNum());
        sheetImportResult.setBoCode(boCode);
        importResult.getSheetImportResultList().add(sheetImportResult);
        return sheetImportResult;
    }

    private void insertRecord(Map<String, Object> contextMap, int rowIndex, String boCode, Map<String, Object> contentMap, ImportResult.SheetImportResult sheetImportResult) {
        IEntityClass iEntityClass = entityFacade.loadByCode(boCode, "").get();
        Optional<Validation<String, Map<String, Object>>> first = postImportAwareList.stream().map(x -> x.doPostFilter(iEntityClass, contentMap, contextMap))
                .filter(Validation::isInvalid).findFirst();
        CompletableFuture<Either<CreateOneResult, Long>> insertResult;
        if (!first.isPresent()) {
            //insert one by one
            Either<CreateOneResult, Long> join = entityFacade.create(iEntityClass, contentMap, contextMap)
                    .toCompletableFuture()
                    .thenApplyAsync(x -> x, importThreadPool).join();
            sheetImportResult.setBoCode(boCode);
            if (join.isLeft()) {
                sheetImportResult.getFailedCount().incrementAndGet();
                sheetImportResult.addErrorResult(rowIndex, join.getLeft().getMessage());
            } else {
                sheetImportResult.getSuccessCount().incrementAndGet();
            }
        } else {
            logger.error("{}, {}, {} is not execute", boCode, contentMap, contextMap);
            sheetImportResult.getFailedCount().incrementAndGet();
            sheetImportResult.addErrorResult(rowIndex, first.get().getError());
        }
    }

    private void insertBatchRecord(Map<String, Object> contextMap, String boCode, List<Map<String, Object>> contentMapList, Integer currentIndex, ImportResult.SheetImportResult sheetImportResult) {
        IEntityClass iEntityClass = entityFacade.loadByCode(boCode, "").get();
        for (int i = 0; i < contentMapList.size(); i++) {
            Map<String, Object> contentMap = contentMapList.get(i);

            Optional<Validation<String, Map<String, Object>>> first = postImportAwareList.stream().map(x -> x.doPostFilter(iEntityClass, contentMap, contextMap))
                    .filter(Validation::isInvalid).findFirst();
            if (first.isPresent()) {
                logger.error("{}, {}, {} is not execute", boCode, contentMap, contextMap);
                sheetImportResult.getFailedCount().incrementAndGet();
                sheetImportResult.addErrorResult(currentIndex + i + 1, first.get().getError());
                contentMapList.remove(i);
            }
        }
        List<Map<String, Object>> validateContentMapList = new ArrayList<>();
        for (int i = 0; i < contentMapList.size(); i++) {
            try {
                Map<String, Object> contentMap = contentMapList.get(i);
                entityFacade.validate(iEntityClass, contentMap);
                validateContentMapList.add(contentMap);
            } catch (Exception e) {
                sheetImportResult.getFailedCount().incrementAndGet();
                sheetImportResult.addErrorResult(currentIndex + i + 1, e.getMessage());
            }
        }

        Either<CreateMultiResult, Integer> join = entityFacade.createMulti(iEntityClass, validateContentMapList, contextMap)
                .toCompletableFuture().join();
        sheetImportResult.setBoCode(boCode);
        if (join.isLeft()) {
            CreateMultiResult left = join.getLeft();
            if (left.getOriginCause() != ResultStatus.OriginStatus.HALF_SUCCESS) {
                throw new RuntimeException(join.getLeft().getEx().getClass().getName() + ":" + join.getLeft().getMessage());
            }
            Integer insertedRowCount = left.getInsertedRows() == null ? 0 : left.getInsertedRows();
            sheetImportResult.getFailedCount().addAndGet(contentMapList.size() - insertedRowCount);
            sheetImportResult.addErrorResult(currentIndex + insertedRowCount, left.getMessage());
        } else {
            sheetImportResult.getSuccessCount().addAndGet(validateContentMapList.size());
        }

    }

    private Long getRelatedIdAndSetContent(Map<String, Object> contextMap
            , List<ImportCmd.ToManyRelation> toManyRelations
            , List<ImportCmd.Sheet.FieldMapping> headerFiledMapping
            , short size
            , List<Integer> relationCodeColIndexList
            , Row row
            , Map<String, List<Map<String, Object>>> boMap) {
        StringBuilder sb = new StringBuilder();
        boMap.values().forEach(x -> x.add(new HashMap<>()));
        for (int j = 0; j < size; j++) {
            Cell cell = row.getCell(j);
            if (cell != null) {
                if (relationCodeColIndexList.contains(j)) {
                    if (sb.length() > 0) {
                        sb.append("%^%");
                    }
                    Object rawValue = readCellRawValue(cell);
                    sb.append(rawValue);
                }
                String boCode = headerFiledMapping.get(j).getBoCode();
                if (boCode == null) {
                    continue;
                }
                if (boMap.containsKey(boCode)) {
                    boMap.get(boCode).get(row.getRowNum() - 1).put(headerFiledMapping.get(j).getCode(), readCellRawValue(cell));
                }
            }
        }

        String s = sb.toString();
        Map<String, Long> indexMapping = getRelatedIndexMapping(contextMap, toManyRelations.get(0).getRelationCode());

        Long related = indexMapping.get(s);
        return related;
    }

    @NotNull
    private List<String> getHeaderFiledCodeList(IEntityClass entityClass, List<ImportCmd.Sheet.FieldMapping> headerFiledMapping, Row headerRow, short lastCellNum) {
        List<String> headerFiledCodeList = new ArrayList();
        for (int j = 0; j < lastCellNum; j++) {
            headerFiledCodeList.add(getBoFieldCodeFromExcelHeader(headerRow.getCell(j).getStringCellValue(), j, headerFiledMapping, entityClass.code(), true));
        }
        if (headerFiledCodeList.size() != headerFiledMapping.size()) {
            throw new RuntimeException(String.format("excel header is not match with import config. config: %s, excel: %s", headerFiledMapping.size(), headerFiledCodeList.size()));
        }
        return headerFiledCodeList;
    }

    @NotNull
    private Map<String, Object> getContextRelatedHeaderMap(ImportCmd cmd) {
        Map<String, Object> contextRelatedHeaderMap = new ConcurrentHashMap<>();
//        Optional.ofNullable(contextService)
//                .map(ContextService::getAll)
//                .orElseGet(Collections::emptyMap);
        contextRelatedHeaderMap.put("index", new HashMap());
        cmd.getToManyRelations().forEach(toManyRelation -> {
            contextRelatedHeaderMap.put(toManyRelation.getRelationCode(), toManyRelation.getRelationFields());
        });
        return contextRelatedHeaderMap;
    }

    @NotNull
    private static List<Integer> getRelationCodeColIndexList(List<ImportCmd.ToManyRelation> toManyRelations, IEntityClass mainEntityClass, List<ImportCmd.Sheet.FieldMapping> headerFiledMapping, short size) {
        List<Integer> relationCodeColIndexList = new ArrayList<>();
        for (int j = 0; j < size; j++) {
            ImportCmd.Sheet.FieldMapping fieldMapping = headerFiledMapping.get(j);
            if (mainEntityClass.code().equals(fieldMapping.getBoCode())
                    && toManyRelations.get(0).getRelationFields().contains(fieldMapping.getCode())) {
                relationCodeColIndexList.add(j);
            }
        }
        return relationCodeColIndexList;
    }

    protected Map<String, Long> getRelatedIndexMapping(Map<String, Object> context, String code) {
        Object index = context.get("index");

        if (index != null) {
            Map<String, Map<String, Long>> indexMapping = (Map<String, Map<String, Long>>) index;
            return Optional.ofNullable(indexMapping.get(code)).orElseGet(Collections::emptyMap);
        } else {
            return Collections.emptyMap();
        }
    }

    /**
     * build index
     */
    protected boolean buildRelatedIndex(String code, Map<String, Object> contextMap, Map<String, Object> body, long id) {
        Object index = contextMap.get("index");
        Object codeMapping = contextMap.get(code);
        if (index == null || codeMapping == null) {
            return false;
        }
        Map<String, Map<String, Long>> indexMapping = (Map<String, Map<String, Long>>) index;
        List<String> relatedCodeList = (List<String>) codeMapping;
        Map<String, Long> indexCodeRelatedMapping = indexMapping.get(code);

        /**
         * initlization
         */
        if (indexCodeRelatedMapping == null) {
            indexCodeRelatedMapping = new HashMap<>();
        }

        indexMapping.put(code, indexCodeRelatedMapping);

        String collect = relatedCodeList.stream().map(x -> body.get(x))
                .filter(Objects::nonNull)
                .map(Object::toString).collect(Collectors.joining("%^%"));
        Long prev = indexCodeRelatedMapping.putIfAbsent(collect, id);
        return prev == null;
    }

    /**
     * this ugly
     *
     * @param code
     */
    private boolean isValid(String code, IEntityClassGroup group) {
        Optional<ColumnField> column = group.column(code);
        return column.isPresent();
    }

}