package com.xforceplus.ultraman.oqsengine.sdk.service.export.impl;

import akka.NotUsed;
import akka.stream.javadsl.Source;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.FieldLikeRelationType;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.impl.Relation;
import com.xforceplus.ultraman.oqsengine.pojo.reader.record.Record;
import com.xforceplus.ultraman.oqsengine.pojo.utils.PropertyHelper;
import com.xforceplus.ultraman.oqsengine.sdk.facade.EntityFacade;
import com.xforceplus.ultraman.oqsengine.sdk.facade.ProfileFetcher;
import com.xforceplus.ultraman.oqsengine.sdk.query.dsl.*;
import com.xforceplus.ultraman.oqsengine.sdk.service.core.ExecutionConfig;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.ClassifiedRecord;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.ExportSource;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.config.ExportConfig;
import com.xforceplus.ultraman.oqsengine.sdk.store.engine.IEntityClassGroup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;
import java.util.stream.Collectors;

/**
 * a nested Export Source
 */
public class NestedExportSource implements ExportSource {

    private Logger logger = LoggerFactory.getLogger(NestedExportSource.class);

    private final int concurrent;

    private final int step;

    private final int maxRetryTimes;

    private final EntityFacade entityService;

    private final ContextService contextService;

    private final ProfileFetcher profileFetcher;

    private final ExecutionConfig config;

    private final int mainBatchSize;

    private final List<SequenceExportSource> exportSourceList;

    public NestedExportSource(
            List<SequenceExportSource> exportSourceList
            , EntityFacade entityService, ProfileFetcher profileFetcher
            , ExportConfig exportConfig
            , ContextService contextService, ExecutionConfig config) {
        this.concurrent = exportConfig.getSubStreamSize();
        this.step = exportConfig.getStepSize();
        this.maxRetryTimes = exportConfig.getMaxRetry();
        this.entityService = entityService;
        this.contextService = contextService;
        this.config = config;
        this.exportSourceList = exportSourceList;
        this.profileFetcher = profileFetcher;
        this.mainBatchSize = exportConfig.getMainBatchSize();
    }

    /**
     * one 2 many
     *
     * @param record
     * @param rel
     * @return
     */
    private ExpRel getRelatedRelQuery(String relationCode, Record record, ExpRel rel) {
        //note mergeAnd will intersect the projects
        ExpQuery idQuery = new ExpQuery().project(rel.getProjects()).filters(
                ExpCondition.call(ExpOperator.EQUALS
                        , ExpField.field(PropertyHelper.generateRelatedFieldName(relationCode, "id"))
                        , ExpValue.from(record.getId())));
        return rel.mergeAnd(idQuery);
    }

    @Override
    public boolean isAccept(IEntityClass entityClass, boolean isMultiSchema, Map<String, Object> context) {
        return isMultiSchema;
    }

//    private Source<ClassifiedRecord, NotUsed> getDependentSource(
//            String classifier
//            , IEntityClass entityClass
//            , ExpRel query
//            , Map<String, Object> context
//            , ClassifiedRecord classifiedRecord
//    ) {
//
//        Optional<ExportSource> sourceOp = exportSourceList.stream().sorted().filter(x -> x.isAccept(entityClass, false, context)).findFirst();
//        return sourceOp.map(x -> x.source(classifier, entityClass, query, null, context, classifiedRecord)).orElseGet(() -> {
//            logger.warn("no suitable source found for {} when {}", entityClass.code(), context);
//            return Source.empty();
//        });
//    }

    protected Source<ClassifiedRecord, NotUsed> getDependentSource(
            String classifier
            , IEntityClass entityClass
            , ExpRel query
            , List<ClassifiedRecord> sourceRecords
            , Map<String, Object> context) {

        Optional<SequenceExportSource> sourceOp = exportSourceList.stream().sorted().filter(x -> x.isAccept(entityClass, false, context)).findFirst();

        List<ClassifiedRecord> classifiedRecords = Optional.ofNullable(sourceRecords).orElseGet(Collections::emptyList);
        Map<Long, ClassifiedRecord> mapping = classifiedRecords.stream().collect(Collectors.toMap(x -> x.getRecord().getId(), x -> x, (a, b) -> a));

        return sourceOp.map(x -> x.source(classifier, entityClass, query, null, context)).orElseGet(() -> {
            logger.warn("no suitable source found for {} when {}", entityClass.code(), context);
            return Source.empty();
        }).map(record -> {
            try {
                if (!mapping.isEmpty()) {
                    String relatedId = generateRelatedFieldName(classifier, "id");
                    Optional<Object> idOp = record.getRecord().get(relatedId);
                    if (idOp.isPresent()) {
                        String id = idOp.get().toString();
                        ClassifiedRecord sourceRecord = mapping.get(Long.parseLong(id));
                        return new ClassifiedRecord(record.getClassifyStr(), record.getRecord(), sourceRecord.getRecord());
                    }
                }
            } catch (Throwable throwable) {
                logger.error("{}", throwable);
            }
            return record;
        });
    }


    @Override
    public int getOrder() {
        return Integer.MAX_VALUE;
    }

    protected String generateRelatedFieldName(String code, String fieldName) {
        StringBuilder sb = new StringBuilder();
        sb.append(code);
        sb.append(".");
        sb.append(fieldName);
        return sb.toString();
    }

    protected ExpRel getRelatedBatchRelQuery(String relationCode, List<ClassifiedRecord> records, ExpRel rel) {

        List<Long> ids = records.stream().map(x -> Optional.ofNullable(x.getRecord()).map(Record::getId).orElse(-1L)).filter(x -> x > 0).collect(Collectors.toList());
        //note mergeAnd will intersect the projects
        ExpQuery idQuery = new ExpQuery().project(rel.getProjects()).filters(
                ExpCondition.call(ExpOperator.IN
                        , ExpField.field(generateRelatedFieldName(relationCode, "id"))
                        , ExpValue.from(ids)));
        return rel.mergeAnd(idQuery);
    }


    @Override
    public Source<ClassifiedRecord, NotUsed> source(
            String classifier
            , IEntityClass entityClass
            , ExpRel query
            , Map<String, ExpRel> subQuery
            , Map<String, Object> context) {

        //the main source
        Source<ClassifiedRecord, NotUsed> mainSource = getDependentSource(classifier, entityClass, query, null, context);

        Set<String> one2ManyCodes = entityClass.relations().stream().filter(x -> FieldLikeRelationType.ONE2MANY.getName()
                .equalsIgnoreCase(x.getRelationType())).map(Relation::getName).collect(Collectors.toSet());

        IEntityClassGroup reader = entityService.getReader(entityClass, context);

        //build sub source
        return mainSource.grouped(mainBatchSize).flatMapMerge(concurrent, i -> {

            Optional<Source<ClassifiedRecord, NotUsed>> reduce = subQuery.entrySet().stream().map(codeQuery -> {

                String relationCode = codeQuery.getKey();
                ExpRel rel = codeQuery.getValue();

                if (one2ManyCodes.contains(relationCode)) {

                    IEntityClass relatedEntityClass = reader.relatedEntityClass(relationCode).orElse(null);

                    if (relatedEntityClass != null) {
                        //do a sub query
                        return getDependentSource(relationCode, relatedEntityClass, getRelatedBatchRelQuery(relationCode, i, rel), i, context);
                    }
                }
                return null;
            }).filter(Objects::nonNull).reduce(Source::concat);

            if (reduce.isPresent()) {
                return Source.from(i).concat(reduce.get());
            } else {
                return Source.from(i);
            }
        });
    }
}
