package com.xforceplus.ultraman.oqsengine.sdk.service.export.impl;

import akka.NotUsed;
import akka.stream.ActorMaterializer;
import akka.stream.IOResult;
import akka.stream.javadsl.Sink;
import akka.stream.javadsl.Source;
import akka.util.ByteString;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityField;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.impl.ColumnField;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.impl.Relation;
import com.xforceplus.ultraman.oqsengine.sdk.event.EntityErrorExported;
import com.xforceplus.ultraman.oqsengine.sdk.event.EntityExported;
import com.xforceplus.ultraman.oqsengine.sdk.facade.EntityFacade;
import com.xforceplus.ultraman.oqsengine.sdk.facade.ProfileFetcher;
import com.xforceplus.ultraman.oqsengine.sdk.query.dsl.ExpRel;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.*;
import com.xforceplus.ultraman.oqsengine.sdk.store.engine.IEntityClassEngine;
import com.xforceplus.ultraman.oqsengine.sdk.store.engine.IEntityClassGroup;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.NameMapping;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import io.vavr.control.Either;
import org.checkerframework.checker.units.qual.A;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * an abstract EntityExportService
 */
public abstract class AbstractEntityExportService implements EntityExportService {

    private Logger logger = LoggerFactory.getLogger(EntityExportService.class);

    /**
     * multi source
     */
    private final List<ExportSource> exportSourceList;

    private final ExportSink exportSink;

    //TODO multi?
    private final ExportStringTransformer transformer;

    private final ActorMaterializer mat;

    private final ExportCallBack callback;

    @Autowired
    private IEntityClassEngine engine;

    @Autowired
    private ProfileFetcher profileFetcher;

    @Value("${xplat.oqsengine.sdk.export.auto-size:10000}")
    private int autoSize;

    private final static String SYNC = "sync";

    private final static String AUTO = "auto";

    public AbstractEntityExportService(
            List<ExportSource> exportSourceList, ExportSink exportSink
            , ExportStringTransformer transformer
            , ExportCallBack exportCallBack
            , ActorMaterializer mat) {
        this.exportSourceList = exportSourceList;
        this.exportSink = exportSink;
        this.callback = exportCallBack;
        this.transformer = transformer;
        this.mat = mat;
    }

    /**
     * only one export Query is single
     *
     * @param exportQuery
     * @return
     */
    private boolean isMultiSchema(ExportQuery exportQuery) {

        Map<String, ExpRel> subQuery = exportQuery.getSubQuery();
        if (subQuery != null && !subQuery.isEmpty()) {
            return true;
        }

        return false;
    }

    /**
     * generate multi schema source
     * one -> one source
     *
     * @return
     */
    private Source<ClassifiedRecord, NotUsed> genMultiSchema(List<ExportQuery> exportQueries, Map<String, Object> context) {

        //multi source
        Optional<Source<ClassifiedRecord, NotUsed>> independentSourceOp = exportQueries.stream().map(exportQuery -> {
            /**
             * create an independentQuery from one exportQuery
             */
            Source<ClassifiedRecord, NotUsed> independentQuery = getIndependentSource(exportQuery, context);
            return independentQuery;
        }).reduce(Source::concat);
        return independentSourceOp.orElseGet(Source::empty);
    }

    /**
     * get Independent source
     *
     * @return
     */
    private Source<ClassifiedRecord, NotUsed> getIndependentSource(
            ExportQuery exportQuery
            , Map<String, Object> context) {

        IEntityClass entityClass = exportQuery.getEntityClass();
        boolean multiSchema = isMultiSchema(exportQuery);
        Optional<ExportSource> sourceOp = exportSourceList.stream().sorted().filter(x -> x.isAccept(entityClass, multiSchema, context)).findFirst();

        ExpRel mainQuery = exportQuery.getMainQuery();
        Map<String, ExpRel> subQuery = exportQuery.getSubQuery();

        return sourceOp.map(x -> x.source(null, entityClass, mainQuery, subQuery, context)).orElseGet(() -> {
            logger.warn("no suitable source found for {} when {}", entityClass.code(), context);
            return Source.empty();
        });
    }

    /**
     * exportQueries is non-empty;
     *
     * @param exportQueries
     * @return
     */
    protected Source<ClassifiedRecord, NotUsed> prepareSource(List<ExportQuery> exportQueries
            , Map<String, Object> context) {

        Source<ClassifiedRecord, NotUsed> source;
        if (exportQueries.size() > 1) {
            //multi source
            if (this.isSupportMultiSchema()) {
                source = genMultiSchema(exportQueries, context);
            } else {
                ExportQuery exportQuery = exportQueries.get(0);
                source = getIndependentSource(exportQuery, context);
            }
        } else {
            ExportQuery exportQuery = exportQueries.get(0);
            source = getIndependentSource(exportQuery, context);
        }

        return source;
    }

    protected Sink<ByteString, CompletionStage<Tuple2<IOResult, String[]>>> prepareSink(String downloadName, String fileName) {
        return exportSink.getSink(this.generateFileType(), downloadName, fileName);
    }

    private List<IEntityClass> getMainEntityClassList(List<ExportQuery> exportQueries) {
        return exportQueries.stream().map(x -> x.getEntityClass()).filter(Objects::nonNull).collect(Collectors.toList());
    }

    private List<Tuple2<String, IEntityClass>> getRelatedMainEntityClassList(List<ExportQuery> exportQueries, Map<String, Object> context) {

        return exportQueries.stream().flatMap(x -> {
            IEntityClass entityClass = x.getEntityClass();
            if (x.getSubQuery() != null) {
                Set<String> keys = x.getSubQuery().keySet();
                if (keys.isEmpty()) {
                    return Stream.of(Tuple.of(entityClass.code(), entityClass));
                } else {
                    IEntityClassGroup describe = engine.describe(entityClass, profileFetcher.getProfile(context));
                    Stream<Tuple2<String, IEntityClass>> subStream = keys.stream().map(key -> {
                        Optional<IEntityClass> relatedEntityClass = describe.relatedEntityClass(key);
                        if (relatedEntityClass.isPresent()) {
                            return Tuple.of(key, relatedEntityClass.get());
                        }
                        return null;
                    }).filter(Objects::nonNull);
                    return Stream.concat(Stream.of(Tuple.of(entityClass.code(), entityClass)), subStream);
                }
            }
            return Stream.of(Tuple.of(entityClass.code(), entityClass));
        }).collect(Collectors.toList());
    }

    /**
     * add default schema config
     * query.name
     *
     * @param exportQueries
     * @return
     */
    private Map<String, ExportSchemaConfig> getExportSchemaConfigMapping(List<ExportQuery> exportQueries) {

        Map<String, ExportSchemaConfig> mapping = new HashMap<>();

        exportQueries.forEach(x -> {
            String entityCode = x.getEntityClass().code();

            Map<String, List<NameMapping>> nameMapping = x.getNameMapping();
            List<NameMapping> nameMappings = Optional.ofNullable(nameMapping.get(entityCode)).orElseGet(Collections::emptyList);
            Map<String, FormattedString> nMapping = nameMappings.stream().filter(nameItem -> {
                return nameItem.getText() != null && nameItem.getCode() != null;
            }).collect(Collectors.toMap(NameMapping::getCode, y -> new FormattedString(y.getText(), y.getFormat()), (a, b) -> a));

            Map<String, FormattedString> modifyMapping = new HashMap<>(nMapping);
            //add default name to entityCode
            modifyMapping.putIfAbsent(entityCode, new FormattedString(x.getEntityClass().name()));

            ExpRel mainQuery = x.getMainQuery();

            //List<String> orderColumn = mainQuery.getOrderedProjectNames();
            List<String> orderColumn = Optional.ofNullable(x.getNameMapping().get(entityCode))
                    .orElseGet(Collections::emptyList).stream()
                    .map(NameMapping::getCode)
                    .collect(Collectors.toList());

            ExportSchemaConfig exportSchemaConfig = new ExportSchemaConfig();
            exportSchemaConfig.setNameMapping(modifyMapping);
            exportSchemaConfig.setOrderedColumn(orderColumn);
            mapping.put(entityCode, exportSchemaConfig);

            if (x.getSubQuery() != null) {
                x.getSubQuery().forEach((key, value) -> {

                    List<NameMapping> subNameMappings = Optional.ofNullable(nameMapping.get(key)).orElseGet(Collections::emptyList);
                    Map<String, FormattedString> subMapping = subNameMappings.stream()
                            .filter(subItem -> subItem.getText() != null && subItem.getCode() != null)
                            .collect(Collectors.toMap(NameMapping::getCode, y -> new FormattedString(y.getText(), y.getFormat()), (a, b) -> a));

                    Map<String, FormattedString> subModifyMapping = new HashMap<>(subMapping);

                    Optional<Relation> first = x.getEntityClass().relations().stream().filter(rel -> rel.getName().equalsIgnoreCase(key)).findFirst();
                    first.ifPresent(relation -> subModifyMapping.putIfAbsent(key, new FormattedString(relation.getName())));

                    ExportSchemaConfig sub = new ExportSchemaConfig();
                    sub.setNameMapping(subModifyMapping);
                    sub.setOrderedColumn(value.getOrderedProjectNames());
                    mapping.put(key, sub);
                });
            }
        });

        return mapping;
    }

    protected String getStringValue(IEntityClass entityClass, IEntityField field, Object value, Map<String, Object> context, Map<String, FormattedString> nameMapping) {
        FormattedString formattedString = nameMapping.get(field.name());
        return transformer.toString(entityClass, field, value, context, formattedString);
    }

    protected abstract Source<ByteString, ?> toByteStringSource(List<Tuple2<String, IEntityClass>> entityClasses, Source<ClassifiedRecord, NotUsed> rawSource
            , Map<String, ExportSchemaConfig> schemaMapping, boolean isSkipTransformer, Map<String, Object> context);

    @Override
    public CompletableFuture<Either<String, String>> export(
            List<ExportQuery> exportQueries
            , String token, String fileName
            , String exportType
            , boolean skipTransformer
            , Map<String, Object> context
            , Map<String, Object> notifyContext) {

        if (exportQueries == null || exportQueries.isEmpty()) {
            return CompletableFuture.completedFuture(Either.left("Empty Query for Export"));
        }

        List<Tuple2<String, IEntityClass>> entityClassWithSubList = getRelatedMainEntityClassList(exportQueries, context);

        List<IEntityClass> entityClassList = getMainEntityClassList(exportQueries);

        CompletableFuture<Either<String, String>> countFuture = new CompletableFuture<>();

        try {
            /**
             * prepare source and sink
             */
            Source<ClassifiedRecord, NotUsed> source = prepareSource(exportQueries, context);
            Sink<ByteString, CompletionStage<Tuple2<IOResult, String[]>>> sink = prepareSink(token, fileName);


            AtomicInteger counter = new AtomicInteger(0);

            source = source.map(x -> {
                if(counter.get() <= autoSize) {
                    counter.incrementAndGet();
                } else {
                    countFuture.complete(Either.right("请求转为异步"));
                }

                return x;
            });
            Source<ByteString, ?> byteStringSource = toByteStringSource(entityClassWithSubList, source, getExportSchemaConfigMapping(exportQueries), skipTransformer, context);

            CompletableFuture<Either<String, String>> syncCompleteResult = byteStringSource.runWith(sink, mat)
                    .toCompletableFuture().thenApply(x -> {
                        String downloadUrl = exportSink.getDownloadUrl(this.generateFileType(), x._2());
                        if (callback != null) {
                            boolean done = countFuture.isDone();
                            callback.onSuccess(() -> new EntityExported(entityClassList, downloadUrl, fileName, exportType, context, notifyContext, done));
                        }
                        return Either.<String, String>right(downloadUrl);
                    }).exceptionally(th -> {
                        if (callback != null) {
                            callback.onFailure(() -> new EntityErrorExported(token, th.getMessage(), context, notifyContext));
                        }
                        return Either.left(th.getMessage());
                    });

            syncCompleteResult.thenApply(countFuture::complete);

            if (SYNC.equalsIgnoreCase(exportType)) {
                return syncCompleteResult;
            } else if(AUTO.equalsIgnoreCase(exportType)) {
                return countFuture;
            } else {
                return CompletableFuture.completedFuture(Either.right("请求完成"));
            }
        } catch (Exception ex) {
            logger.error("{}", ex);
            return CompletableFuture.completedFuture(Either.left(ex.getMessage()));
        }
    }
}