package com.xforceplus.ultraman.oqsengine.sdk.service.export.impl;

import cn.hutool.json.JSONUtil;
import com.xforceplus.tech.base.core.context.ContextService;
import com.xforceplus.tech.base.core.dispatcher.anno.QueryHandler;
import com.xforceplus.tech.base.core.dispatcher.messaging.MetaData;
import com.xforceplus.tech.base.core.dispatcher.messaging.QueryMessage;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.sdk.command.ConditionExportCmd;
import com.xforceplus.ultraman.oqsengine.sdk.command.GetImportTemplateCmd;
import com.xforceplus.ultraman.oqsengine.sdk.command.ImportCmd;
import com.xforceplus.ultraman.oqsengine.sdk.facade.EntityFacade;
import com.xforceplus.ultraman.oqsengine.sdk.facade.ProfileFetcher;
import com.xforceplus.ultraman.oqsengine.sdk.query.export.ExportCmdQuery;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.*;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.config.ExportConfig;
import com.xforceplus.ultraman.oqsengine.sdk.service.export.entity.ImportResult;
import com.xforceplus.ultraman.oqsengine.sdk.store.engine.IEntityClassEngine;
import com.xforceplus.ultraman.oqsengine.sdk.store.engine.IEntityClassGroup;
import com.xforceplus.ultraman.oqsengine.sdk.transactional.OqsTransactionManager;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.NameMapping;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import io.vavr.control.Either;
import org.apache.commons.lang3.StringUtils;

import java.io.InputStream;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * export
 */
public class BulkServiceImpl implements BulkService {

    private final ImportServiceFactory importServiceFactory;
    private EntityFacade entityFacade;

    private List<EntityExportService> exportServices;

    private ImportTemplateService importTemplateService;
    private ContextService contextService;

    private ProfileFetcher fetcher;

    private ExportConfig exportConfig;

    private DownloadFilenameGenerator filenameGenerator;

    private IEntityClassEngine engine;

    private OqsTransactionManager manager;

    public BulkServiceImpl(EntityFacade entityFacade
            , ImportTemplateService importTemplateService
            , ImportService importService
            , List<EntityExportService> exportServices
            , ContextService contextService
            , ExportConfig exportConfig
            , DownloadFilenameGenerator filenameGenerator
            , ProfileFetcher fetcher
            , IEntityClassEngine engine
            , OqsTransactionManager manager
            , ImportServiceFactory importServiceFactory
    ) {
        this.entityFacade = entityFacade;
        this.exportServices = exportServices;
        this.contextService = contextService;
        this.exportConfig = exportConfig;
        this.filenameGenerator = filenameGenerator;
        this.importTemplateService = importTemplateService;
        this.fetcher = fetcher;
        this.engine = engine;
        this.manager = manager;
        this.importServiceFactory = importServiceFactory;
    }

    /**
     * TODO
     */
    private static final String MISSING_ENTITIES = "查询对象不存在";

    @QueryHandler(isDefault = true)
    @Override
    public CompletableFuture<Either<String, String>> conditionExport(QueryMessage<ConditionExportCmd, ?> message) {

        ConditionExportCmd cmd = message.getPayload();

        MetaData metaData = message.getMetaData();

        //#26 user-center not support Chinese
        //remove
        Long currentTime = System.nanoTime();

        /**
         * download token
         */
        String token = Optional.ofNullable(metaData.get("code")).map(Object::toString)
                .orElse(cmd.getBoId()).trim() + "-" + currentTime;

        /**
         * download filename
         */
//        String fileName = Optional.ofNullable(metaData.get("name")).map(Object::toString).map(String::trim)
//                .orElse(Optional.ofNullable(metaData.get("code")).map(Object::toString)
//                        .orElse(cmd.getBoId())).trim() + "-" + currentTime;

        Map<String, Object> context = new HashMap<>();
        context.putAll(metaData);
        context.putAll(Optional.ofNullable(contextService).map(ContextService::getAll).orElseGet(Collections::emptyMap));
        String fileName;
        if (StringUtils.isEmpty(cmd.getSpecifyFileName())) {
            fileName = filenameGenerator.getFileName(cmd, context);
        } else {
            fileName = cmd.getSpecifyFileName();
        }

        String exportType = cmd.getExportType();
        String exportFileType = cmd.getExportFileType();

        List<ExportQuery> exportQueries = toExportQueryList(cmd.getExports(), cmd.version());

        if (exportQueries.isEmpty()) {
            return CompletableFuture.completedFuture(Either.left(MISSING_ENTITIES));
        } else {
            Optional<EntityExportService> exportServiceOp = exportServices.stream().filter(x -> x.isAccept(exportFileType)).findAny();
            if (exportServiceOp.isPresent()) {
                EntityExportService entityExportService = exportServiceOp.get();

                Map<String, Object> notifyContext = new HashMap<>();
                notifyContext.put("appId", Optional.ofNullable(cmd.getAppId()).orElse(exportConfig.getNotifyId()));

                Map<String, Object> contextMap = Optional.ofNullable(contextService)
                        .map(x -> x.getAll())
                        .orElseGet(Collections::emptyMap);

                HashMap<String, Object> enhancedContextMap = new HashMap<>(contextMap);
                enhancedContextMap.put("template", cmd.getTemplate());

                return entityExportService.export(exportQueries
                        , token, fileName
                        , exportType
                        , cmd.isSkipTransformer()
                        , enhancedContextMap
                        , notifyContext);
            }

            return CompletableFuture.completedFuture(Either.left("No suitable ExportService for " + exportFileType));
        }
    }

    private List<ExportQuery> toExportQueryList(Map<String, ExportCmdQuery> exports, String version) {
        return exports.entrySet().stream().map(entry -> {

            Optional<? extends IEntityClass> iEntityClassOp;
            String boId = entry.getKey();

            String profile = fetcher.getProfile(Collections.emptyMap());

            if (version == null) {
                iEntityClassOp = entityFacade.load(boId, profile);
            } else {
                iEntityClassOp = entityFacade.load(boId, profile, version);
            }

            if (iEntityClassOp.isPresent()) {
                ExportQuery exportQuery = new ExportQuery();

                Map<String, List<NameMapping>> map = new HashMap<>();

                ExportCmdQuery cmdQuery = entry.getValue();
                map.put(iEntityClassOp.get().code(), cmdQuery.getMainMapping());
                map.putAll(cmdQuery.getSubMapping());

                exportQuery.setEntityClass(iEntityClassOp.get());
                exportQuery.setNameMapping(map);
                exportQuery.setMainQuery(cmdQuery.getMainQuery());
                exportQuery.setSubQuery(cmdQuery.getSubQuery());

                return exportQuery;
            } else {
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toList());
    }

    @QueryHandler(isDefault = true)
    @Override
    public CompletableFuture<Either<String, String>> importTemplateCustom(GetImportTemplateCmd cmd) {
        String boId = cmd.getBoId();
        String profile = fetcher.getProfile(Collections.emptyMap());

        Optional<IEntityClass> entityClass = entityFacade.load(boId, profile);

        if (entityClass.isPresent()) {
            IEntityClassGroup group = engine.describe(entityClass.get(), profile);
            List<NameMapping> mappingList = cmd.getMappingList();
            String name = importTemplateService.fileName(entityClass.get());
            String fileType = importTemplateService.fileType();

            //generate file url name + "." + fileType, importService.getCustomTemplate(group, mappingList);

            String uuid = importTemplateService.getCustomTemplate(group, mappingList, cmd.getSubMapping());

            return CompletableFuture.completedFuture(Either.right(uuid));
        } else {
            return CompletableFuture.completedFuture(Either.left(MISSING_ENTITIES));
        }
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, Tuple2<String, InputStream>> importTemplate(GetImportTemplateCmd cmd) {

        String boId = cmd.getBoId();
        String profile = fetcher.getProfile(Collections.emptyMap());

        Optional<IEntityClass> entityClass = entityFacade.load(boId, profile);

        if (entityClass.isPresent()) {
            IEntityClassGroup group = engine.describe(entityClass.get(), profile);

            String name = importTemplateService.fileName(entityClass.get());
            String fileType = importTemplateService.fileType();
            return Either.right(Tuple.of(name + "." + fileType, importTemplateService.getTemplateInputStream(group, cmd.isSkipSystem())));
        } else {
            return Either.left(MISSING_ENTITIES);
        }
    }

    /**
     * extract name from headerPart
     *
     * @param headerPart
     */
    private String getKeyFromHeader(String headerPart) {
        int start = headerPart.indexOf("[");
        int end = headerPart.indexOf("]");

        if (start < 0 || end < 0) {
            return headerPart;
        } else {
            return headerPart.substring(start + 1, end);
        }
    }

    @QueryHandler(isDefault = true)
    @Override
    public Either<String, String> batchImport(ImportCmd cmd) {
        String boId = cmd.getBoId();
        String profile = fetcher.getProfile(Collections.emptyMap());
        Optional<IEntityClass> entityClassOp = entityFacade.load(boId, profile);

        if (!entityClassOp.isPresent()) {
            return Either.left(MISSING_ENTITIES);
        }
        ImportService importService = this.importServiceFactory.getImportService(cmd.getImportMode());
        Either<String, String> ret;
        try {
            Map<String, Object> all = contextService.getAll();
            IEntityClassGroup entityClassGroup = engine.describe(entityClassOp.get(), profile);
            if (cmd.isAsync()) {
                CompletableFuture.runAsync(() -> {
                            contextService.fromMap(all);
                            importService.doImport(entityClassGroup, cmd);
                        }
                );
            } else {
                ImportResult importResult = importService.doImport(entityClassGroup, cmd);
                if (importResult != null) {
                    return Either.right(JSONUtil.toJsonStr(importResult));
                }
            }
            return Either.right(null);
        } catch (Throwable e) {
            e.printStackTrace();
            ret = Either.left(e.getMessage());
        }
        return ret;

    }
}
